﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>

#include <nn/util/util_IntrusiveList.h>

#include "i2cTegra_I2cBusAccessor.h"

namespace nnd { namespace i2c { namespace tegra { namespace detail {

// TODO: 資源の取得競合の排除
class I2cBusAccessorAllocator
{
public:
    I2cBusAccessorAllocator(I2cBusAccessor* pBuffer, int bufferCount) NN_NOEXCEPT
    {
        m_FreeList.clear();
        for ( int i = 0; i < bufferCount; ++i )
        {
            m_FreeList.push_back(pBuffer[i]);
        }
        m_AllocateList.clear();
    }

    I2cBusAccessor* Alloc() NN_NOEXCEPT
    {
        if ( m_FreeList.empty() )
        {
            return nullptr;
        }
        auto& busAccessor = m_FreeList.front();
        m_FreeList.pop_front();
        m_AllocateList.push_front(busAccessor);
        return &busAccessor;
    }

    void Free(I2cBusAccessor* p) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(p);
        NN_SDK_ASSERT(p->IsLinkedToList());

        p->RemoveFrom(&m_AllocateList);

        NN_SDK_ASSERT(!p->IsLinkedToList());
        m_FreeList.push_front(*p);
    }

    template<typename FuncT>
    I2cBusAccessor* FindOnAllocateList(FuncT func) NN_NOEXCEPT
    {
        for ( auto&& busAccessor : m_AllocateList )
        {
            if ( func(&busAccessor) )
            {
                return &busAccessor;
            }
        }

        return nullptr;
    }

private:
    I2cBusAccessor::List m_FreeList;
    I2cBusAccessor::List m_AllocateList;
};

template<int MaxI2cBusAccessorCount>
class I2cBusAccessorManager
{
public:
    I2cBusAccessorManager() NN_NOEXCEPT
    {
    }

    I2cBusAccessorAllocator& GetAllocator() NN_NOEXCEPT
    {
        return m_Allocator;
    }

private:
    I2cBusAccessor m_EntryBuffer[MaxI2cBusAccessorCount] { };
    I2cBusAccessorAllocator m_Allocator { m_EntryBuffer, MaxI2cBusAccessorCount };
};

}}}} // nnd::i2c::tegra::detail
