﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>

#include <nn/os/os_SdkMutex.h>
#include <nn/pcv/pcv.h>
#include <nn/result/result_HandlingUtility.h>

#include "i2cTegra_I2cBusAccessor.h"
#include "i2cTegra_TargetSpec.h"

namespace nnd { namespace i2c { namespace tegra { namespace detail {

class I2cBusAccessor;

class BusPowerManager final
{
    NN_DISALLOW_COPY(BusPowerManager);
    NN_DISALLOW_MOVE(BusPowerManager);

public:
    BusPowerManager() NN_NOEXCEPT
    {
        for ( auto&& busAccessorEntry : m_I2cBusAccessorEntries )
        {
            busAccessorEntry.Clear();
        }
    }

    void RegisterI2cBusAccessor(const I2cBusAccessor* pI2cBusAccessor) NN_NOEXCEPT
    {
        for ( const auto& busAccessorEntry : m_I2cBusAccessorEntries )
        {
            if ( busAccessorEntry.pI2cBusAccessor == pI2cBusAccessor )
            {
                // 登録済みなので追加しない
                return;
            }
        }

        for ( auto&& busAccessorEntry : m_I2cBusAccessorEntries )
        {
            if ( !busAccessorEntry.pI2cBusAccessor )
            {
                busAccessorEntry.pI2cBusAccessor = pI2cBusAccessor;
                busAccessorEntry.isPoweredOn = false;
                return;
            }
        }

        NN_ABORT("There is no blank entry.");
    }

    void PowerOn(const I2cBusAccessor* pI2cBusAccessor) NN_NOEXCEPT
    {
        for ( auto&& busAccessorEntry : m_I2cBusAccessorEntries )
        {
            // 登録されている場合のみ PowerOn する
            if ( busAccessorEntry.pI2cBusAccessor == pI2cBusAccessor )
            {
                std::lock_guard<decltype(m_BusPowerMutex)> lock(m_BusPowerMutex);
                if ( !IsPoweredOn() )
                {
                    PowerOnBuses();
                }
                busAccessorEntry.isPoweredOn = true;
                break;
            }
        }
    }

    void PowerOff(const I2cBusAccessor* pI2cBusAccessor) NN_NOEXCEPT
    {
        for ( auto&& busAccessorEntry : m_I2cBusAccessorEntries )
        {
            // 登録されている場合のみ PowerOff する
            if ( busAccessorEntry.pI2cBusAccessor == pI2cBusAccessor )
            {
                std::lock_guard<decltype(m_BusPowerMutex)> lock(m_BusPowerMutex);
                busAccessorEntry.isPoweredOn = false;
                if ( !IsPoweredOn() )
                {
                    PowerOffBuses();
                }
                break;
            }
        }
    }

private:
    bool IsPoweredOn() NN_NOEXCEPT
    {
        for ( const auto& busAccessorEntry : m_I2cBusAccessorEntries )
        {
            if ( busAccessorEntry.pI2cBusAccessor && busAccessorEntry.isPoweredOn )
            {
                return true;
            }
        }
        return false;
    }

    void PowerOnBuses() NN_NOEXCEPT
    {
        nn::pcv::Initialize();
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::pcv::SetVoltageValue(nn::pcv::PowerDomain_Max77620_Ldo6, 2900000));
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::pcv::SetVoltageEnabled(nn::pcv::PowerDomain_Max77620_Ldo6, true));
        nn::pcv::Finalize();

        // LDO6 が立ち上がるまでには 0.56 ms の待ちが必要
        nn::os::SleepThread(::nn::TimeSpan::FromMicroSeconds(560));
        NN_DETAIL_I2C_INFO("Set LDO6 voltage to %d\n",2900000);
    }

    void PowerOffBuses() NN_NOEXCEPT
    {
        NN_DETAIL_I2C_INFO("Disable LDO6 Voltage\n");
        nn::pcv::Initialize();
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::pcv::SetVoltageEnabled(nn::pcv::PowerDomain_Max77620_Ldo6, false));
        nn::pcv::Finalize();
    }

private:
    struct I2cBusAccessorEntry
    {
        const I2cBusAccessor* pI2cBusAccessor;
        bool isPoweredOn;

        void Clear() NN_NOEXCEPT
        {
            pI2cBusAccessor = nullptr;
            isPoweredOn = false;
        }
    };

private:
    //! エントリの Power On 状態と pcv の操作を守るミューテックス
    nn::os::SdkMutex m_BusPowerMutex { };

    //! 登録された @ref I2cBusAccessor と Power On 状態の対応を保存するエントリ
    I2cBusAccessorEntry m_I2cBusAccessorEntries[MaxBuses];
};

}}}} // nnd::i2c::tegra::detail
