﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/ddsf/ddsf_IDriver.h> // To call IDriver::ForEachDevice
#include <nn/gpio/gpio_Result.h>
#include <nn/gpio/detail/gpio_Log.h>

#include "gpioTegra_DdUtil.h"
#include "gpioTegra_RegAccessor.h"
#include "gpioTegra_SuspendHandler.h"

#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
#include <nne/wec/wec.h>
#include "gpioTegra_DisableWakePinSettingWithoutPmic-hardware.nx.h"
#endif

namespace nnd { namespace gpio { namespace tegra { namespace detail {

void SuspendHandler::Initialize(uintptr_t gpioBaseAddress) NN_NOEXCEPT
{
    m_GpioBaseAddress = gpioBaseAddress;

#if defined NN_BUILD_CONFIG_HARDWARE_NX
    nne::wec::Initialize();
#endif
}

void SuspendHandler::SetValueForSleepState(PadTegra* pPad, nn::gpio::GpioValue value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pPad);
    // pOutSession 内の GPIO 番号を アクセスするアドレス・ビットへ変換
    InternalGpioPadNumber internalPadNum = pPad->GetPadNumber();
    int          bitPosition   = ConvertInternalGpioPadNumberToBitPosition(internalPadNum);
    int          portNumber    = ConvertInternalGpioPadNumberToPortNumber(internalPadNum);

    // 値を保存しておく
    m_GpioValuesForSleep[portNumber].IsForceSet |= 1 << bitPosition;
    m_GpioValuesForSleep[portNumber].out        &= ~(1 << bitPosition);
    m_GpioValuesForSleep[portNumber].out        |= (value == nn::gpio::GpioValue_High) ? (1 << bitPosition) : 0;
}

nn::Result SuspendHandler::IsWakeEventActive(bool* pOutIsActive, PadTegra* pPad) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutIsActive);
    NN_SDK_REQUIRES_NOT_NULL(pPad);
    *pOutIsActive = pPad->GetStatus().isWakeActive;
    NN_RESULT_SUCCESS;
}

nn::Result SuspendHandler::SetWakeEventActiveFlagSetForDebug(PadTegra* pPad, bool enable) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pPad);
    pPad->GetStatus().isWakeActiveTest = enable;
    NN_RESULT_SUCCESS;
}

void SuspendHandler::Suspend() NN_NOEXCEPT
{
    NN_DETAIL_GPIO_TRACE("Store All Gpio Pin Register\n");

    // レジスタの値をメモリへ保存
    int port = 0;
    for (auto& i : m_EscapedRegisterValue)
    {
        i.conf   = ReadBitsFromPort<nn::Bit16>(GpioRegisterType_GPIO_CNF,     port, GetGpioBaseAddress());
        i.oe     = ReadBitsFromPort<nn::Bit8>( GpioRegisterType_GPIO_OE,      port, GetGpioBaseAddress());
        i.out    = ReadBitsFromPort<nn::Bit8>( GpioRegisterType_GPIO_OUT,     port, GetGpioBaseAddress());
        i.intEnb = ReadBitsFromPort<nn::Bit8>( GpioRegisterType_GPIO_INT_ENB, port, GetGpioBaseAddress());
        i.intLvl = ReadBitsFromPort<nn::Bit32>(GpioRegisterType_GPIO_INT_LVL, port, GetGpioBaseAddress());
        i.dbCtrl = ReadBitsFromPort<nn::Bit8>( GpioRegisterType_GPIO_DB_CTRL, port, GetGpioBaseAddress());
        i.dbCnt  = ReadBitsFromPort<nn::Bit8>( GpioRegisterType_GPIO_DB_CNT,  port, GetGpioBaseAddress());

        /*
        NN_DETAIL_GPIO_INFO("---------------port = %d--------------------\n", port);
        NN_DETAIL_GPIO_INFO("RegValue conf = %x\n", ReadBitsFromPort<nn::Bit16>(GpioRegisterType_GPIO_CNF, port, GetGpioBaseAddress()));
        NN_DETAIL_GPIO_INFO("RegValue oe   = %x\n", ReadBitsFromPort<nn::Bit8>(GpioRegisterType_GPIO_OE, port, GetGpioBaseAddress()));
        NN_DETAIL_GPIO_INFO("RegValue out  = %x\n", ReadBitsFromPort<nn::Bit8>(GpioRegisterType_GPIO_OUT, port, GetGpioBaseAddress()));
        NN_DETAIL_GPIO_INFO("RegValue intEnb = %x\n", ReadBitsFromPort<nn::Bit8>(GpioRegisterType_GPIO_INT_ENB, port, GetGpioBaseAddress()));
        NN_DETAIL_GPIO_INFO("RegValue intLvl = %x\n", ReadBitsFromPort<nn::Bit32>(GpioRegisterType_GPIO_INT_LVL, port, GetGpioBaseAddress()));
        NN_DETAIL_GPIO_INFO("RegValue dbCtrl = %x\n", ReadBitsFromPort<nn::Bit8>(GpioRegisterType_GPIO_DB_CTRL, port, GetGpioBaseAddress()));
        NN_DETAIL_GPIO_INFO("RegValue dbCnt = %x\n", ReadBitsFromPort<nn::Bit8>(GpioRegisterType_GPIO_DB_CNT, port, GetGpioBaseAddress()));
        */

        // 保存し終えたピンはすべて割り込みを disable に
        nn::Bit8 disable = 0x00;
        WriteBitsFromPort<nn::Bit8>( disable, GpioRegisterType_GPIO_INT_ENB, port, GetGpioBaseAddress());

        // IsForceSet が enable のものは出力する値を設定する
        if(m_GpioValuesForSleep[port].IsForceSet)
        {
            i.out = (i.out & ~m_GpioValuesForSleep[port].IsForceSet) | (m_GpioValuesForSleep[port].out & m_GpioValuesForSleep[port].IsForceSet);
            WriteBitsFromPort<nn::Bit8>(i.out, GpioRegisterType_GPIO_OUT, port, GetGpioBaseAddress());
            NN_DETAIL_GPIO_TRACE("Force Set GPIO value. port = %d, value = %x\n", port, m_GpioValuesForSleep[port].out);
        }

        // 書き込み確定のための DummyRead
        DummyRead(GetGpioRegAccessAddress(GpioRegisterType_GPIO_CNF, static_cast<InternalGpioPadNumber>(port * 8), GetGpioBaseAddress()));

        port++;
    }
}

// Suspend と SuspendLow の役割分担は http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=166038312 を参照
void SuspendHandler::SuspendLow() NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_HARDWARE_NX)

    // 要因を取り始める前に古い要因はクリアしておく
    nne::wec::ClearWakeEvents();

    // 充電 IC の HiZ モードの切り替えをセキュアモニタでおこなう際に割り込みが入ってしまうためセキュアモニタでスリープ直前に wake 要因を取得できるようにする（SIGLO-38305）
    // // これ以降に発生した wake 要因を取得できるようにする。呼ばない場合は WakeDetectMode_HwLpsEntry として要因が取得される
    // NN_DETAIL_GPIO_TRACE("Start wake pin detect\n");
    // nne::wec::WecLpsEntryPrepare(nne::wec::WakeDetectMode_WecLpsEntry);
#endif
}

void SuspendHandler::Resume() NN_NOEXCEPT
{
    NN_DETAIL_GPIO_TRACE("Restore All Gpio Interrupt Pins\n");

    int port = 0;

    for (auto& i : m_EscapedRegisterValue)
    {
        WriteBitsFromPort<nn::Bit8>(i.intEnb, GpioRegisterType_GPIO_INT_ENB, port, GetGpioBaseAddress());

        //NN_DETAIL_GPIO_INFO("RegValue intEnb = %x\n", ReadBitsFromPort<nn::Bit8>(GpioRegisterType_GPIO_INT_ENB, port, GetGpioBaseAddress()));

        port++;
    }
}

// Resume と ResumeLow の役割分担は http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=166038312 を参照
void SuspendHandler::ResumeLow() NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_HARDWARE_NX)

    nne::wec::WakeEvent wakeEventList[nne::wec::WakeEvent_Num];
    int activeWakeEventCount;

    // ここまでに取れた wake 要因を保存する
    nne::wec::WecLpsExitRestore();

    // WakeEvent の Active 状態を読みだす
    nne::wec::GetActiveWakeEvents(wakeEventList, &activeWakeEventCount, nne::wec::WakeEvent_Num);

    // BitFlag に設定する
    m_BaseDriver.ForEachDevice(
        [&wakeEventList](nn::ddsf::IDevice* pDevice) NN_NOEXCEPT -> bool
        {
            NN_SDK_ASSERT_NOT_NULL(pDevice);
            auto& d = pDevice->SafeCastTo<PadTegra>();
            d.GetStatus().isWakeActive = false;
            if ( d.GetStatus().isWakeActiveTest )
            {
                d.GetStatus().isWakeActive = true;
                return true;
            }
            for ( const auto& wakeEvent : wakeEventList )
            {
                if ( wakeEvent == d.GetInfo().wakeEvent )
                {
                    NN_DETAIL_GPIO_TRACE("Active Wake Event (wec : %d, pad number : %d)\n", wakeEvent, d.GetPadNumber());
                    d.GetStatus().isWakeActive = true;
                }
            }
            return true;
        }
    );
#endif

    NN_DETAIL_GPIO_TRACE("Restore All Gpio Pin Register\n");

    // レジスタの値をメモリから復旧
    int port = 0;
    for (auto& i : m_EscapedRegisterValue)
    {
        WriteBitsFromPort<nn::Bit16>(i.conf, GpioRegisterType_GPIO_CNF, port, GetGpioBaseAddress());
        WriteBitsFromPort<nn::Bit8>(i.oe, GpioRegisterType_GPIO_OE, port, GetGpioBaseAddress());
        WriteBitsFromPort<nn::Bit8>(i.out, GpioRegisterType_GPIO_OUT, port, GetGpioBaseAddress());
        WriteBitsFromPort<nn::Bit32>(i.intLvl, GpioRegisterType_GPIO_INT_LVL, port, GetGpioBaseAddress());
        WriteBitsFromPort<nn::Bit8>(i.dbCtrl, GpioRegisterType_GPIO_DB_CTRL, port, GetGpioBaseAddress());
        WriteBitsFromPort<nn::Bit8>(i.dbCnt, GpioRegisterType_GPIO_DB_CNT, port, GetGpioBaseAddress());

        // XXX: Pinmux 書き込み前に InterruptEnable を書き戻すと意図してない割り込みが発生するため、別関数で対応する。
        //WriteBitsFromPort<nn::Bit8>( i.intEnb, GpioRegisterType_GPIO_INT_ENB, port, GetGpioBaseAddress());

        /*
        NN_DETAIL_GPIO_INFO("---------------port = %d--------------------\n", port);
        NN_DETAIL_GPIO_INFO("RegValue conf = %x\n", ReadBitsFromPort<nn::Bit16>(GpioRegisterType_GPIO_CNF, port, GetGpioBaseAddress()));
        NN_DETAIL_GPIO_INFO("RegValue oe   = %x\n", ReadBitsFromPort<nn::Bit8>(GpioRegisterType_GPIO_OE, port, GetGpioBaseAddress()));
        NN_DETAIL_GPIO_INFO("RegValue out  = %x\n", ReadBitsFromPort<nn::Bit8>(GpioRegisterType_GPIO_OUT, port, GetGpioBaseAddress()));
        NN_DETAIL_GPIO_INFO("RegValue intEnb = %x\n", ReadBitsFromPort<nn::Bit8>(GpioRegisterType_GPIO_INT_ENB, port, GetGpioBaseAddress()));
        NN_DETAIL_GPIO_INFO("RegValue intLvl = %x\n", ReadBitsFromPort<nn::Bit32>(GpioRegisterType_GPIO_INT_LVL, port, GetGpioBaseAddress()));
        NN_DETAIL_GPIO_INFO("RegValue dbCtrl = %x\n", ReadBitsFromPort<nn::Bit8>(GpioRegisterType_GPIO_DB_CTRL, port, GetGpioBaseAddress()));
        NN_DETAIL_GPIO_INFO("RegValue dbCnt = %x\n", ReadBitsFromPort<nn::Bit8>(GpioRegisterType_GPIO_DB_CNT, port, GetGpioBaseAddress()));
        */

        port++;
    }

    // 書き込み確定のための DummyRead
    DummyRead(GetGpioRegAccessAddress(GpioRegisterType_GPIO_CNF, static_cast<InternalGpioPadNumber>(0), GetGpioBaseAddress()));
}

void SuspendHandler::SetWakePinDebugMode(nn::gpio::driver::WakePinDebugMode mode) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
    switch (mode)
    {
    case nn::gpio::driver::WakePinDebugMode_AutoImmediateWake :
        // WakeEvent_SDMMC3_DAT1 ピンは L 設定で常に wake event が来ている状態になる (CRCT-333)
        nne::wec::SetWakeEventEnabled(nne::wec::WakeEvent_SDMMC3_DAT1, true);
        nne::wec::SetWakeEventLevel(nne::wec::WakeEvent_SDMMC3_DAT1, nne::wec::WakeEventLevel_Low);
        break;

    case nn::gpio::driver::WakePinDebugMode_NoWake :
        // wake pin を disable にする設定を書き込む
        for (int i = 0; i < sizeof(detail::wakePinDisableWithoutPmicConfigList) / sizeof(detail::wakePinDisableWithoutPmicConfigList[0]); i++)
        {
            SetWakeEventLevel(detail::wakePinDisableWithoutPmicConfigList[i].wakeEventId, detail::wakePinDisableWithoutPmicConfigList[i].level);
            SetWakeEventEnabled(detail::wakePinDisableWithoutPmicConfigList[i].wakeEventId, detail::wakePinDisableWithoutPmicConfigList[i].isWakeEventEnable);
        }
        break;

    default: NN_UNEXPECTED_DEFAULT;
    }


#else
    NN_DETAIL_GPIO_TRACE("No wake pin debug settings\n");
#endif
}

}}}} // nnd::gpio::tegra::detail
