﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/os/os_InterruptEventCommon.h>

#include "gpioTegra_Pad.h"

namespace{

const int NumberOfGpioPerOneDriverBitWidth = 5;
const int NumberOfDriverBitWidth = 3;

const int NumberOfPortPerOneDriverBitWidth = 2;
const int NumberOfPortPerOneDriver = (1u << NumberOfPortPerOneDriverBitWidth);

const int NumberOfbitPerOnePortBitWidth = 3;
const int NumberOfbitPerOnePort = (1u << NumberOfbitPerOnePortBitWidth);
}

namespace nnd { namespace gpio { namespace tegra { namespace detail {

// GPIO コントローラー番号の列挙体
enum GpioDriver
{
    GpioDriver_GPIO1 = 0,
    GpioDriver_GPIO2,
    GpioDriver_GPIO3,
    GpioDriver_GPIO4,
    GpioDriver_GPIO5,
    GpioDriver_GPIO6,
    GpioDriver_GPIO7,
    GpioDriver_GPIO8,
    GpioDriver_NumOfGpioDriver,
};

// 割り込み番号の定義
// (Tegra_K1_TRM_DP06905001v02p.pdf p.23)
const int PeripheralInterruptOffset = 32;
const int Interrupt_Gpio1 = PeripheralInterruptOffset + 32;
const int Interrupt_Gpio2 = PeripheralInterruptOffset + 33;
const int Interrupt_Gpio3 = PeripheralInterruptOffset + 34;
const int Interrupt_Gpio4 = PeripheralInterruptOffset + 35;
const int Interrupt_Gpio5 = PeripheralInterruptOffset + 55;
const int Interrupt_Gpio6 = PeripheralInterruptOffset + 87;
const int Interrupt_Gpio7 = PeripheralInterruptOffset + 89;
const int Interrupt_Gpio8 = PeripheralInterruptOffset + 125;
const nn::os::InterruptName InterruptNameTable[GpioDriver_NumOfGpioDriver] =
{
    Interrupt_Gpio1,
    Interrupt_Gpio2,
    Interrupt_Gpio3,
    Interrupt_Gpio4,
    Interrupt_Gpio5,
    Interrupt_Gpio6,
    Interrupt_Gpio7,
    Interrupt_Gpio8,
};

// GPIO コントローラー(レジスタ)の列挙体
enum GpioRegisterType
{
    GpioRegisterType_GPIO_CNF = 0,
    GpioRegisterType_GPIO_OE,
    GpioRegisterType_GPIO_OUT,
    GpioRegisterType_GPIO_IN,
    GpioRegisterType_GPIO_INT_STA,
    GpioRegisterType_GPIO_INT_ENB,
    GpioRegisterType_GPIO_INT_LVL,
    GpioRegisterType_GPIO_INT_CLR,
    GpioRegisterType_GPIO_DB_CTRL,
    GpioRegisterType_GPIO_DB_CNT,
};

// GPIO パッドがサポートしている機能の一覧
enum AccessType
{
    AccessType_ExistPad         = (1 << 0),
    AccessType_InterruptMode    = (1 << 1),
    AccessType_InterruptEnable  = (1 << 2),
    AccessType_DataInput        = (1 << 3),
    AccessType_DataOutput       = (1 << 4),
    AccessType_DebounceFunction = (1 << 5),
};

// Jetson の InterruptMode 設定用レジスタ(GPIO_INT_LVL)に設定する Bit 列(BIT_0 を起点としているので、シフトして使う)。
// (Tegra_K1_TRM_DP06905001v02p.pdf p.283)
enum InternalInterruptMode
{
    InternalInterruptMode_LowLevel    = (0 << 16) | (0 << 8) | (0 << 0),
    InternalInterruptMode_HighLevel   = (0 << 16) | (0 << 8) | (1 << 0),
    InternalInterruptMode_RisingEdge  = (0 << 16) | (1 << 8) | (1 << 0),
    InternalInterruptMode_FallingEdge = (0 << 16) | (1 << 8) | (0 << 0),
    InternalInterruptMode_AnyEdge     = (1 << 16) | (1 << 8) | (0 << 0),
    InternalInterruptMode_BitMask     = (1 << 16) | (1 << 8) | (1 << 0),
};

// Tegra の GPIO パッドが持っている機能
#if defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1)
const nn::Bit8 GpioType = AccessType_ExistPad | AccessType_InterruptMode | AccessType_InterruptEnable |
                      AccessType_DataInput | AccessType_DataOutput;
#elif defined(NN_BUILD_CONFIG_SOC_TEGRA_X1)
const nn::Bit8 GpioType = AccessType_ExistPad | AccessType_InterruptMode | AccessType_InterruptEnable |
                      AccessType_DataInput | AccessType_DataOutput | AccessType_DebounceFunction;
#endif

// 内部でサポートしていないパッドは割込みが使用できないようにする
const nn::Bit8 GpioWitoutInterruptType = AccessType_ExistPad | AccessType_DataInput | AccessType_DataOutput;


inline int ConvertInternalGpioPadNumberToDriverNumber(InternalGpioPadNumber padNumber)
{
    return (padNumber >> NumberOfGpioPerOneDriverBitWidth);
}

inline int ConvertInternalGpioPadNumberToPortNumber(InternalGpioPadNumber padNumber)
{
    return (padNumber >> NumberOfDriverBitWidth);
}

inline int ConvertInternalGpioPadNumberToBitPosition(InternalGpioPadNumber padNumber)
{
    return padNumber & (NumberOfbitPerOnePort - 1u);
}

inline int ConvertPortNumberToAddressOffset(int number)
{
    return number & (NumberOfPortPerOneDriver - 1u);
}

// Per-Pin Mask Write をするためのアドレスとビットのオフセット値
const uintptr_t MaskWriteAddressOffset = 0x0080;
const int       MaskWriteBitOffset = 8;

// 各GPIO レジスタのアクセスアドレスを計算する関数 (Tegra_K1_TRM_DP06905001v02p.pdf p.278)
inline nn::Bit32* GetGpioRegAccessAddress(GpioRegisterType regType, InternalGpioPadNumber padNumber, uintptr_t gpioVirtualAddress)
{
    int controllerNum = ConvertInternalGpioPadNumberToDriverNumber(padNumber);
    int portNum  = ConvertInternalGpioPadNumberToPortNumber(padNumber);
    int offset   = ConvertPortNumberToAddressOffset(portNum);

    // Debaunce 系のレジスタだけ特定のレジスタの Upper Offset を使っている
    switch(regType)
    {
    // このレジスタは GpioRegisterType_GPIO_IN の Upper Offset を使っているので、MaskedWrite を行うので、GPIO_IN のアドレスをそのまま返す
    case GpioRegisterType_GPIO_DB_CTRL:
        return reinterpret_cast<nn::Bit32*>(gpioVirtualAddress + (controllerNum) * 0x0100 + (GpioRegisterType_GPIO_IN) * 0x0010 + (offset) * 0x004);

    // このレジスタは GpioRegisterType_GPIO_INT_CLR の Upper Offset を使っているので、GPIO_INT_CLR のアドレス + Offset を返す
    case GpioRegisterType_GPIO_DB_CNT:
        return reinterpret_cast<nn::Bit32*>(gpioVirtualAddress + (controllerNum) * 0x0100 + (GpioRegisterType_GPIO_INT_CLR) * 0x0010 + (offset) * 0x004
                                                                                                                                + MaskWriteAddressOffset);
    default:
        return reinterpret_cast<nn::Bit32*>(gpioVirtualAddress + (controllerNum) * 0x0100 + (regType) * 0x0010 + (offset) * 0x004);
    }
}

template <typename T>
inline T ReadBitsFromPort(GpioRegisterType regType, int portNumber, uintptr_t gpioVirtualAddress)
{
    // DB_CTRL だけ offset を足した値を返す必要がある
    if (regType == GpioRegisterType_GPIO_DB_CTRL)
    {
        auto address = GetGpioRegAccessAddress(regType, static_cast<InternalGpioPadNumber>(portNumber * 8), gpioVirtualAddress);
        return *reinterpret_cast<T*>((reinterpret_cast<uintptr_t>(address) + MaskWriteAddressOffset));
    }
    else
    {
        // 1 ポートあたり 8 ピンなので、ポート番号に 8 をかけた先頭のピンを渡す
        return static_cast<T>(*GetGpioRegAccessAddress(regType, static_cast<InternalGpioPadNumber>(portNumber * 8), gpioVirtualAddress));
    }
}

template <typename T>
inline void WriteBitsFromPort(T value, GpioRegisterType regType, int portNumber, uintptr_t gpioVirtualAddress)
{
    // DB_CTRL だけ offset を足した値のアドレスに MaskedWrite する必要がある
    if (regType == GpioRegisterType_GPIO_DB_CTRL)
    {
        auto address = reinterpret_cast<uintptr_t>(GetGpioRegAccessAddress(regType, static_cast<InternalGpioPadNumber>(portNumber * 8), gpioVirtualAddress))
            + MaskWriteAddressOffset;

        // このレジスタは MaskWrite する必要がある。
        *reinterpret_cast<volatile nn::Bit32*>(address) = value | (0xff << 8);
    }
    else
    {
        // 1 ポートあたり 8 ピンなので、ポート番号に 8 をかけた先頭のピンを渡す
        volatile nn::Bit32* address = GetGpioRegAccessAddress(regType, static_cast<InternalGpioPadNumber>(portNumber * 8), gpioVirtualAddress);
        *address = value;
    }
}

// Tegra の GPIO レジスタに備わっている MaskedWrite 機能を使って特定ピン(ビット)だけ設定する関数
// Read-Modify-Write していないので、必ず MaskedWrite 機能を持ったレジスタに使用すること。
inline void SetBitForTegraMaskedWrite(int value, int pos, nn::Bit32* gpioVirtualAddress)
{
    uintptr_t address = reinterpret_cast<uintptr_t>(gpioVirtualAddress) + MaskWriteAddressOffset;

    *reinterpret_cast<volatile nn::Bit32*>(address) = (1 << ( MaskWriteBitOffset + pos)) | (value << pos);
}

}}}} // nnd::gpio::tegra::detail
