﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/os/os_InterruptEvent.h>

#include <nn/gpio/gpio_Type.h>
#include <nn/gpio/driver/gpio_IGpioDriver.h>

#include "gpioTegra_Pad.h"
#include "gpioTegra_RegAccessor.h"
#include "gpioTegra_SuspendHandler.h"

//---------------------------------------------------------------------------

namespace nnd { namespace gpio { namespace tegra { namespace detail {

class DriverImpl :
    public nn::gpio::driver::IGpioDriver
{
    NN_DDSF_CAST_SAFE_DECL;

    NN_DISALLOW_COPY(DriverImpl);
    NN_DISALLOW_MOVE(DriverImpl);

public:
    DriverImpl() NN_NOEXCEPT :
        m_GpioBaseAddress(0),
        m_SuspendHandler(this)
    {}

    // IGpioDriver: ドライバ初期化
    virtual void InitializeDriver() NN_NOEXCEPT NN_OVERRIDE;

    // IGpioDriver: パッド初期化・終了
    virtual nn::Result InitializePad(nn::gpio::driver::Pad* pPad) NN_NOEXCEPT NN_OVERRIDE;
    virtual void FinalizePad(nn::gpio::driver::Pad* pPad) NN_NOEXCEPT NN_OVERRIDE;

    // IGpioDriver: パッド制御
    virtual nn::Result GetDirection(nn::gpio::Direction* pOut, nn::gpio::driver::Pad* pPad) const NN_NOEXCEPT NN_OVERRIDE;
    virtual nn::Result SetDirection(nn::gpio::driver::Pad* pPad, nn::gpio::Direction direction) NN_NOEXCEPT NN_OVERRIDE;
    virtual nn::Result GetValue(nn::gpio::GpioValue* pOut, nn::gpio::driver::Pad* pPad) const NN_NOEXCEPT NN_OVERRIDE;
    virtual nn::Result SetValue(nn::gpio::driver::Pad* pPad, nn::gpio::GpioValue value) NN_NOEXCEPT NN_OVERRIDE;

    // IGpioDriver: 割り込みハンドリング
    virtual nn::Result GetInterruptMode(nn::gpio::InterruptMode* pOut, nn::gpio::driver::Pad* pPad) const NN_NOEXCEPT NN_OVERRIDE;
    virtual nn::Result SetInterruptEnabled(nn::gpio::driver::Pad* pPad, bool enable) NN_NOEXCEPT NN_OVERRIDE;
    virtual nn::Result SetInterruptMode(nn::gpio::driver::Pad* pPad, nn::gpio::InterruptMode mode) NN_NOEXCEPT NN_OVERRIDE;
    virtual nn::Result GetInterruptStatus(nn::gpio::InterruptStatus* pOut, nn::gpio::driver::Pad* pPad) const NN_NOEXCEPT NN_OVERRIDE;
    virtual nn::Result ClearInterruptStatus(nn::gpio::driver::Pad* pPad) NN_NOEXCEPT NN_OVERRIDE;
    virtual nn::os::SdkMutex& GetInterruptControlMutex(const nn::gpio::driver::Pad& pad) const NN_NOEXCEPT NN_OVERRIDE
    {
        // GPIO サブシステムがパッドの割り込み設定の操作をする際に使用する mutex 。とりあえずドライバ全体で 1 つ
        NN_UNUSED(pad);
        return m_InterruptControlMutex;
    }

    // IGpioDriver: デバウンス設定
    virtual nn::Result GetDebounceEnabled(bool* pOut, nn::gpio::driver::Pad* pPad) const NN_NOEXCEPT NN_OVERRIDE;
    virtual nn::Result SetDebounceEnabled(nn::gpio::driver::Pad* pPad, bool enable) NN_NOEXCEPT NN_OVERRIDE;
    virtual nn::Result GetDebounceTime(int* pOutMsec, nn::gpio::driver::Pad* pPad) const NN_NOEXCEPT NN_OVERRIDE;
    virtual nn::Result SetDebounceTime(nn::gpio::driver::Pad* pPad, int msecTime) NN_NOEXCEPT NN_OVERRIDE;

    // IGpioDriver: サスペンド・レジューム
    virtual nn::Result SetValueForSleepState(nn::gpio::driver::Pad* pPad, nn::gpio::GpioValue value) NN_NOEXCEPT NN_OVERRIDE;
    virtual nn::Result IsWakeEventActive(bool* pOutIsActive, nn::gpio::driver::Pad* pPad) const NN_NOEXCEPT NN_OVERRIDE;
    virtual nn::Result SetWakeEventActiveFlagSetForDebug(nn::gpio::driver::Pad* pPad, bool enable) NN_NOEXCEPT NN_OVERRIDE;
    virtual nn::Result SetWakePinDebugMode(nn::gpio::driver::WakePinDebugMode mode) NN_NOEXCEPT NN_OVERRIDE;
    virtual nn::Result Suspend() NN_NOEXCEPT NN_OVERRIDE;
    virtual nn::Result SuspendLow() NN_NOEXCEPT NN_OVERRIDE;
    virtual nn::Result Resume() NN_NOEXCEPT NN_OVERRIDE;
    virtual nn::Result ResumeLow() NN_NOEXCEPT NN_OVERRIDE;

    // For DeviceCodeNodeParser
    void AddPad(PadTegra& pad) NN_NOEXCEPT
    {
        RegisterDevice(&pad);
    }
    PadTegra* FindIdenticalPad(int padNumber, const PadInfo& padInfo) NN_NOEXCEPT
    {
        PadTegra* pPad = nullptr;
        ForEachDevice(
            [&](nn::ddsf::IDevice* pDevice) NN_NOEXCEPT -> bool
            {
                NN_SDK_ASSERT_NOT_NULL(pDevice);
                auto& pad = pDevice->SafeCastTo<PadTegra>();
                if ( pad.GetPadNumber() == padNumber && pad.GetInfo() == padInfo )
                {
                    pPad = &pad;
                    return false; // STOP
                }
                return true;
            }
        );
        return pPad;
    }

private:
    class InterruptEventHandler :
        public nn::ddsf::IEventHandler
    {
    public:
        void Initialize(DriverImpl* pDriver, int port) NN_NOEXCEPT;

        virtual void HandleEvent() NN_NOEXCEPT NN_OVERRIDE;

    private:
        void CheckHandleInterrupt(PadTegra* pPad) NN_NOEXCEPT;

    private:
        DriverImpl* m_pDriver{ nullptr };
        int m_Port;
        nn::os::InterruptEventType m_Event;
    };

private:

    uintptr_t GetGpioBaseAddress() const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_EQUAL(m_GpioBaseAddress, 0);
        return m_GpioBaseAddress;
    }

    static inline PadTegra& GetPadTegra(nn::gpio::driver::Pad* pBase) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pBase);
        return static_cast<PadTegra&>(*pBase);
    }
    static inline const PadInfo& GetInfo(nn::gpio::driver::Pad* pBase) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pBase);
        return static_cast<PadTegra&>(*pBase).GetInfo();
    }
    static inline PadStatus& GetStatus(nn::gpio::driver::Pad* pBase) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pBase);
        return static_cast<PadTegra&>(*pBase).GetStatus();
    }

    void AddInterruptBoundPad(PadTegra* pPad) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pPad);
        if ( !pPad->IsLinkedToInterruptBoundPadList() )
        {
            m_InterruptBoundPadList.push_back(*pPad);
        }
    }
    void RemoveInterruptBoundPad(PadTegra* pPad) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pPad);
        if ( pPad->IsLinkedToInterruptBoundPadList() )
        {
            m_InterruptBoundPadList.erase(m_InterruptBoundPadList.iterator_to(*pPad));
        }
    }

private:
    uintptr_t m_GpioBaseAddress{ 0 };
    SuspendHandler m_SuspendHandler;
    PadTegra::InterruptBoundList m_InterruptBoundPadList;

    InterruptEventHandler m_InterruptEventHandler[GpioDriver_NumOfGpioDriver];
    mutable nn::os::SdkMutex m_InterruptControlMutex;
};

}}}} // nnd::gpio::tegra::detail
