﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/util/util_BitPack.h>
#include <nn/os/os_InterruptEvent.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/dd.h>
#include <nn/gpio/gpio_Result.h>
#include <nn/gpio/detail/gpio_Log.h>
#include <nn/gpio/driver/gpio_DriverService.h>

#include "gpioTegra_DdUtil.h"
#include "gpioTegra_RegAccessor.h"

#include "gpioTegra_DriverImpl.h"

NN_DDSF_CAST_SAFE_DEFINE(nnd::gpio::tegra::detail::PadTegra, nn::gpio::driver::Pad); // TORIAEZU: PadTegra.cpp がないのでここに
NN_DDSF_CAST_SAFE_DEFINE(nnd::gpio::tegra::detail::DriverImpl, nn::gpio::driver::IGpioDriver);

namespace nnd { namespace gpio { namespace tegra { namespace detail {

void DriverImpl::InitializeDriver() NN_NOEXCEPT
{
    const nn::dd::PhysicalAddress   GpioPhysicalAddress = 0x06000d000ull;
    const size_t                    GpioAddressSize = 0x1000;
    m_GpioBaseAddress = nn::dd::QueryIoMappingAddress(GpioPhysicalAddress, GpioAddressSize);
    if ( m_GpioBaseAddress == 0 )
    {
        // physicalAddress が指す I/O アドレスがマッピングされていない
        NN_ABORT("I/O registers for 0x%llx are not mapped. Make sure the capability setting is properly set for this process.\n", GpioPhysicalAddress);
    }

    // GPIO コントローラーすべての割り込みを Initialize し、MultiWaitHolder に登録する。
    int index = 0;
    for ( auto&& handler : m_InterruptEventHandler )
    {
        handler.Initialize(this, index);
        nn::gpio::driver::RegisterInterruptHandler(&handler);
        ++index;
    }

    m_SuspendHandler.Initialize(m_GpioBaseAddress);
}

void DriverImpl::InterruptEventHandler::Initialize(DriverImpl* pDriver, int port) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDriver);
    m_Port = port;
    m_pDriver = pDriver;
    nn::os::InitializeInterruptEvent(&m_Event, InterruptNameTable[port], nn::os::EventClearMode_AutoClear);
    IEventHandler::Initialize(&m_Event);
}


nn::Result DriverImpl::InitializePad(nn::gpio::driver::Pad* pPad) NN_NOEXCEPT
{
    // 内部で使用する padNumber へキャスト
    InternalGpioPadNumber internalPadNum = pPad->GetPadNumber();

    // pOutSession 内の GPIO 番号を アクセスするアドレス・ビットへ変換
    nn::Bit32* accessAddress = GetGpioRegAccessAddress(GpioRegisterType_GPIO_CNF, internalPadNum, GetGpioBaseAddress());
    int bitPosition = ConvertInternalGpioPadNumberToBitPosition(internalPadNum);

    // GPIO_CNF へ Per-Pin Mask Write で値を書きこみ、GPIO としてセットする。
    //(Tegra_K1_TRM_DP06905001v02p.pdf p.278)
    SetBitForTegraMaskedWrite(1, bitPosition, accessAddress);
    DummyRead(accessAddress);

    NN_RESULT_SUCCESS;
}

void DriverImpl::FinalizePad(nn::gpio::driver::Pad* pPad) NN_NOEXCEPT
{
    NN_UNUSED(pPad);
}

nn::Result DriverImpl::GetDirection(nn::gpio::Direction* pOut, nn::gpio::driver::Pad* pPad) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOut);
    NN_SDK_REQUIRES_NOT_NULL(pPad);

    // 内部で使用する padNumber へキャスト
    InternalGpioPadNumber internalPadNum = pPad->GetPadNumber();

    // InternalGpioPadNumber を アクセスするアドレス・ビットへ変換
    int bitPosition = ConvertInternalGpioPadNumberToBitPosition(internalPadNum);
    nn::Bit32*   accessAddress = GetGpioRegAccessAddress(GpioRegisterType_GPIO_OE, internalPadNum, GetGpioBaseAddress());

    if ( GetBit<nn::Bit32>(accessAddress, bitPosition) )
    {
        *pOut = nn::gpio::Direction_Output;
    }
    else
    {
        *pOut = nn::gpio::Direction_Input;
    }

    NN_RESULT_SUCCESS;
}

nn::Result DriverImpl::SetDirection(nn::gpio::driver::Pad* pPad, nn::gpio::Direction direction) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pPad);

    // 内部で使用する padNumber へキャスト
    InternalGpioPadNumber internalPadNum = pPad->GetPadNumber();

    // pOutSession 内の GPIO 番号を アクセスするアドレス・ビットへ変換
    int bitPosition      = ConvertInternalGpioPadNumberToBitPosition(internalPadNum);
    nn::Bit32*   accessAddress = GetGpioRegAccessAddress(GpioRegisterType_GPIO_OE, internalPadNum, GetGpioBaseAddress());

    // GPIO_OE を output の時は DRIVENに、input の時は TRISTATE にする
    // (Tegra_K1_TRM_DP06905001v02p.pdf p.268 の回路図参照)
    SetBitForTegraMaskedWrite(direction, bitPosition, accessAddress);
    DummyRead(accessAddress);

    NN_RESULT_SUCCESS;
}

nn::Result DriverImpl::GetValue(nn::gpio::GpioValue* pOut, nn::gpio::driver::Pad* pPad) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOut);
    NN_SDK_REQUIRES_NOT_NULL(pPad);

    // 内部で使用する padNumber へキャスト
    InternalGpioPadNumber internalPadNum = pPad->GetPadNumber();

    // InternalGpioPadNumber を アクセスするアドレス・ビットへ変換
    int bitPosition = ConvertInternalGpioPadNumberToBitPosition(internalPadNum);
    nn::Bit32*   accessAddress = GetGpioRegAccessAddress(GpioRegisterType_GPIO_IN, internalPadNum, GetGpioBaseAddress());

    if ( GetBit<nn::Bit32>(accessAddress, bitPosition) )
    {
        *pOut = nn::gpio::GpioValue_High;
    }
    else
    {
        *pOut = nn::gpio::GpioValue_Low;
    }

    NN_RESULT_SUCCESS;
}

nn::Result DriverImpl::SetValue(nn::gpio::driver::Pad* pPad, nn::gpio::GpioValue value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pPad);

    // 内部で使用する padNumber へキャスト
    InternalGpioPadNumber internalPadNum = pPad->GetPadNumber();

    // pOutSession 内の GPIO 番号を アクセスするアドレス・ビットへ変換
    int          bitPosition = ConvertInternalGpioPadNumberToBitPosition(internalPadNum);
    nn::Bit32*   accessAddress = GetGpioRegAccessAddress(GpioRegisterType_GPIO_OUT, internalPadNum, GetGpioBaseAddress());

    // GPIO_OUT へ Per-Pin Mask Write で値を書きこみ、出力を変える
    SetBitForTegraMaskedWrite(value, bitPosition, accessAddress);
    DummyRead(accessAddress);

    NN_RESULT_SUCCESS;
}

nn::Result DriverImpl::GetInterruptMode(nn::gpio::InterruptMode* pOut, nn::gpio::driver::Pad* pPad) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOut);
    NN_SDK_REQUIRES_NOT_NULL(pPad);

    // 内部で使用する padNumber へキャスト
    InternalGpioPadNumber internalPadNum = pPad->GetPadNumber();

    // InternalGpioPadNumber を アクセスするアドレス・ビットへ変換
    int bitPosition      = ConvertInternalGpioPadNumberToBitPosition(internalPadNum);
    nn::Bit32*   accessAddress = GetGpioRegAccessAddress(GpioRegisterType_GPIO_INT_LVL, internalPadNum, GetGpioBaseAddress());

    // レジスタに書かれた InterruptMode を bitPosition 分シフトした値で InterruptMode を判別
    nn::Bit32 shiftedInterruptModeRegister = *accessAddress >> bitPosition;

    switch(shiftedInterruptModeRegister & InternalInterruptMode_BitMask)
    {
    case InternalInterruptMode_LowLevel:
        *pOut = nn::gpio::InterruptMode_LowLevel;
        break;

    case InternalInterruptMode_HighLevel:
        *pOut = nn::gpio::InterruptMode_HighLevel;
        break;

    case InternalInterruptMode_RisingEdge:
        *pOut = nn::gpio::InterruptMode_RisingEdge;
        break;

    case InternalInterruptMode_FallingEdge:
        *pOut = nn::gpio::InterruptMode_FallingEdge;
        break;

    case InternalInterruptMode_AnyEdge:
        *pOut = nn::gpio::InterruptMode_AnyEdge;
        break;

    default: NN_UNEXPECTED_DEFAULT;
    }

    NN_RESULT_SUCCESS;
}

nn::Result DriverImpl::SetInterruptMode(nn::gpio::driver::Pad* pPad, nn::gpio::InterruptMode mode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pPad);

    // 内部で使用する padNumber へキャスト
    InternalGpioPadNumber internalPadNum = pPad->GetPadNumber();

    // InternalGpioPadNumber を アクセスするアドレス・ビットへ変換(ここは Read-Modify-Write)
    int bitPosition = ConvertInternalGpioPadNumberToBitPosition(internalPadNum);
    nn::Bit32*   accessAddress = GetGpioRegAccessAddress(GpioRegisterType_GPIO_INT_LVL, internalPadNum, GetGpioBaseAddress());

    // 各割り込みモードに対応した Bit 列をシフトして設定
    switch ( mode )
    {
        case nn::gpio::InterruptMode_LowLevel:
            WriteMasked32(accessAddress,
                (InternalInterruptMode_LowLevel << bitPosition),
                (InternalInterruptMode_BitMask << bitPosition));
            break;

        case nn::gpio::InterruptMode_HighLevel:
            WriteMasked32(accessAddress,
                (InternalInterruptMode_HighLevel << bitPosition),
                (InternalInterruptMode_BitMask << bitPosition));
            break;

        case nn::gpio::InterruptMode_RisingEdge:
            WriteMasked32(accessAddress,
                (InternalInterruptMode_RisingEdge << bitPosition),
                (InternalInterruptMode_BitMask << bitPosition));
            break;

        case nn::gpio::InterruptMode_FallingEdge:
            WriteMasked32(accessAddress,
                (InternalInterruptMode_FallingEdge << bitPosition),
                (InternalInterruptMode_BitMask << bitPosition));
            break;

        case nn::gpio::InterruptMode_AnyEdge:
            WriteMasked32(accessAddress,
                (InternalInterruptMode_AnyEdge << bitPosition),
                (InternalInterruptMode_BitMask << bitPosition));
            break;

        default:
            NN_SDK_ASSERT("GPIO ライブラリが対応していない割り込み方法が指定されています");
            break;
    }
    DummyRead(accessAddress);

    NN_RESULT_SUCCESS;
}

nn::Result DriverImpl::SetInterruptEnabled(nn::gpio::driver::Pad* pPad, bool enable) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pPad);

    // 内部で使用する padNumber へキャスト
    InternalGpioPadNumber internalPadNum = pPad->GetPadNumber();

    // InternalGpioPadNumber を アクセスするアドレス・ビットへ変換
    int bitPosition      = ConvertInternalGpioPadNumberToBitPosition(internalPadNum);
    nn::Bit32*   accessAddress = GetGpioRegAccessAddress(GpioRegisterType_GPIO_INT_ENB, internalPadNum, GetGpioBaseAddress());

    // GPIO_INT_ENB を叩いて割り込みを設定する
    if(enable)
    {
        AddInterruptBoundPad(&GetPadTegra(pPad));
        SetBitForTegraMaskedWrite(1, bitPosition, accessAddress);
    }
    else
    {
        SetBitForTegraMaskedWrite(0, bitPosition, accessAddress);
        RemoveInterruptBoundPad(&GetPadTegra(pPad));
    }
    DummyRead(accessAddress);

    NN_RESULT_SUCCESS;
}

nn::Result DriverImpl::GetInterruptStatus(nn::gpio::InterruptStatus* pOut, nn::gpio::driver::Pad* pPad) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOut);
    NN_SDK_REQUIRES_NOT_NULL(pPad);

    // 内部で使用する padNumber へキャスト
    InternalGpioPadNumber internalPadNum = pPad->GetPadNumber();

    // InternalGpioPadNumber を アクセスするアドレス・ビットへ変換
    int bitPosition = ConvertInternalGpioPadNumberToBitPosition(internalPadNum);
    nn::Bit32*   accessAddress = GetGpioRegAccessAddress(GpioRegisterType_GPIO_INT_STA, internalPadNum, GetGpioBaseAddress());

    // GPIO_INT_STA を叩いて割り込みが発生しているかどうかを返す。
    if ( GetBit<nn::Bit32>(accessAddress, bitPosition) )
    {
        *pOut = nn::gpio::InterruptStatus_Active;
    }
    else
    {
        *pOut = nn::gpio::InterruptStatus_Inactive;
    }

    NN_RESULT_SUCCESS;
}

nn::Result DriverImpl::ClearInterruptStatus(nn::gpio::driver::Pad* pPad) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pPad);

    // 内部で使用する padNumber へキャスト
    InternalGpioPadNumber internalPadNum = pPad->GetPadNumber();

    // InternalGpioPadNumber を アクセスするアドレス・ビットへ変換
    int bitPosition = ConvertInternalGpioPadNumberToBitPosition(internalPadNum);
    nn::Bit32*   accessAddress = GetGpioRegAccessAddress(GpioRegisterType_GPIO_INT_CLR, internalPadNum, GetGpioBaseAddress());

    // GPIO_INT_CLR を叩いて割り込みをクリアする
    SetBit<nn::Bit32>(accessAddress, 1, bitPosition);
    DummyRead(accessAddress);

    NN_RESULT_SUCCESS;
}

nn::Result DriverImpl::GetDebounceEnabled(bool* pOut, nn::gpio::driver::Pad* pPad) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOut);
    NN_SDK_REQUIRES_NOT_NULL(pPad);

    // 内部で使用する padNumber へキャスト
    InternalGpioPadNumber internalPadNum = pPad->GetPadNumber();

    // pSession 内の GPIO 番号を アクセスするアドレス・ビットへ変換
    int          bitPosition = ConvertInternalGpioPadNumberToBitPosition(internalPadNum);
    nn::Bit32*   accessAddress = GetGpioRegAccessAddress(GpioRegisterType_GPIO_DB_CTRL, internalPadNum, GetGpioBaseAddress());

    *pOut = GetBit<nn::Bit32>(accessAddress, bitPosition);

    NN_RESULT_SUCCESS;
}

nn::Result DriverImpl::SetDebounceEnabled(nn::gpio::driver::Pad* pPad, bool isEnable) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pPad);

    // 内部で使用する padNumber へキャスト
    InternalGpioPadNumber internalPadNum = pPad->GetPadNumber();

    // pSession 内の GPIO 番号を アクセスするアドレス・ビットへ変換
    int          bitPosition   = ConvertInternalGpioPadNumberToBitPosition(internalPadNum);
    nn::Bit32*   accessAddress = GetGpioRegAccessAddress(GpioRegisterType_GPIO_DB_CTRL, internalPadNum, GetGpioBaseAddress());

    SetBitForTegraMaskedWrite(isEnable ? 1 : 0, bitPosition, accessAddress);
    DummyRead(accessAddress);

    NN_RESULT_SUCCESS;
}

nn::Result DriverImpl::GetDebounceTime(int* pOutMsec, nn::gpio::driver::Pad* pPad) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutMsec);
    NN_SDK_REQUIRES_NOT_NULL(pPad);

    // 内部で使用する padNumber へキャスト
    InternalGpioPadNumber internalPadNum = pPad->GetPadNumber();

    // pSession 内の GPIO 番号を アクセスするアドレスへ変換
    nn::Bit32*   accessAddress       = GetGpioRegAccessAddress(GpioRegisterType_GPIO_DB_CNT, internalPadNum, GetGpioBaseAddress());

    *pOutMsec = *accessAddress & 0xff;

    NN_RESULT_SUCCESS;
}

nn::Result DriverImpl::SetDebounceTime(nn::gpio::driver::Pad* pPad, int msecTime) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pPad);
    NN_SDK_REQUIRES(0 <= msecTime && msecTime <= 128);

    // TODO: こんなワークアラウンドが要らない汎用的な方法を考える
#if defined NN_BUILD_CONFIG_HARDWARE_NX
    InternalGpioPadNumber padNumber = pPad->GetPadNumber();
    NN_UNUSED(padNumber); // Release ビルド対策

    // チャタリング防止時間の設定は特定ピンだけが設定可能
    NN_SDK_ASSERT(
        padNumber == 38  || // ExtconDetS
        padNumber == 62  || // ExtconDetU
        padNumber == 174 || // CodecHpDetIrq
        padNumber == 190 || // ButtonVolUp
        padNumber == 191 || // ButtonVolDn
        padNumber == 201 || // SdCd
        padNumber == 203,   // SdWp for testing
        "This pad (%d) cannot change decounde time. Please contact the person in charge of gpio library\n", padNumber
    );
#endif

    // 内部で使用する padNumber へキャスト
    InternalGpioPadNumber internalPadNum = pPad->GetPadNumber();

    // pSession 内の GPIO 番号を アクセスするアドレスへ変換
    nn::Bit32*   accessAddress = GetGpioRegAccessAddress(GpioRegisterType_GPIO_DB_CNT, internalPadNum, GetGpioBaseAddress());

    // Debounce 時間を書き込む(このピンが属するポートすべてに影響する)
    WriteMasked32(accessAddress, msecTime, 0xff);
    DummyRead(accessAddress);

    NN_RESULT_SUCCESS;
}

nn::Result DriverImpl::SetValueForSleepState(nn::gpio::driver::Pad* pPad, nn::gpio::GpioValue value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pPad);
    m_SuspendHandler.SetValueForSleepState(&GetPadTegra(pPad), value);
    NN_RESULT_SUCCESS;
}
nn::Result DriverImpl::IsWakeEventActive(bool* pOutIsActive, nn::gpio::driver::Pad* pPad) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pPad);
    return m_SuspendHandler.IsWakeEventActive(pOutIsActive, &GetPadTegra(pPad));
}
nn::Result DriverImpl::SetWakeEventActiveFlagSetForDebug(nn::gpio::driver::Pad* pPad, bool enable) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pPad);
    return m_SuspendHandler.SetWakeEventActiveFlagSetForDebug(&GetPadTegra(pPad), enable);
}
nn::Result DriverImpl::SetWakePinDebugMode(nn::gpio::driver::WakePinDebugMode mode) NN_NOEXCEPT
{
    m_SuspendHandler.SetWakePinDebugMode(mode);
    NN_RESULT_SUCCESS;
}
nn::Result DriverImpl::Suspend() NN_NOEXCEPT
{
    m_SuspendHandler.Suspend();
    NN_RESULT_SUCCESS;
}
nn::Result DriverImpl::SuspendLow() NN_NOEXCEPT
{
    m_SuspendHandler.SuspendLow();
    NN_RESULT_SUCCESS;
}
nn::Result DriverImpl::Resume() NN_NOEXCEPT
{
    m_SuspendHandler.Resume();
    NN_RESULT_SUCCESS;
}
nn::Result DriverImpl::ResumeLow() NN_NOEXCEPT
{
    m_SuspendHandler.ResumeLow();
    NN_RESULT_SUCCESS;
}

void DriverImpl::InterruptEventHandler::HandleEvent() NN_NOEXCEPT
{
    for ( auto first = m_pDriver->m_InterruptBoundPadList.begin(), end = m_pDriver->m_InterruptBoundPadList.end(); first != end; )
    {
        auto&& pad = *first;
        ++first; // CheckHandleInterrupt() 内で pad を m_InterruptBoundPadList から抜く可能性があるため、先にイテレートしておかないとハマる
        CheckHandleInterrupt(&pad);
    }
}

void DriverImpl::InterruptEventHandler::CheckHandleInterrupt(PadTegra* pPad) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_pDriver->GetInterruptControlMutex(*pPad))> lock(m_pDriver->GetInterruptControlMutex(*pPad));

    auto baseAddress = m_pDriver->GetGpioBaseAddress();
    auto padNumber = pPad->GetPadNumber();
    nn::Bit32* staAddress = GetGpioRegAccessAddress(GpioRegisterType_GPIO_INT_STA, static_cast<InternalGpioPadNumber>(padNumber), baseAddress);
    nn::Bit32* enbAddress = GetGpioRegAccessAddress(GpioRegisterType_GPIO_INT_ENB, static_cast<InternalGpioPadNumber>(padNumber), baseAddress);
    int bitPosition = ConvertInternalGpioPadNumberToBitPosition(static_cast<InternalGpioPadNumber>(padNumber));

    // INT_STA が立っており、かつ割り込みが enable
    if ( GetBit(staAddress, bitPosition) && GetBit(enbAddress, bitPosition) )
    {
        // 1. 対象のピンの割込みを無効にする
        SetBitForTegraMaskedWrite(0, bitPosition, enbAddress);
        DummyRead(enbAddress);
        pPad->SetInterruptEnabled(false);
        m_pDriver->RemoveInterruptBoundPad(pPad);

        // 2. シグナルされた割り込みイベントをクリアする
        nn::os::ClearInterruptEvent(&m_Event);

        // 3. ピンに紐づいているセッション側のイベントをシグナルする
        pPad->SignalInterruptBoundEvent();
    }
}

}}}} // nnd::gpio::tegra::detail
