﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   デバイスドライバ向けユーティリティ関数とクラス
 */



#pragma once

#include <nn/nn_BitTypes.h>
#include <nn/os.h>
#include <nn/os/os_InterruptEvent.h>

namespace nnd { namespace gpio { namespace tegra { namespace detail {

// address に指定したレジスタの読み出しを実行します。
// TegraX1 では レジスタの書き込み完了に使用します。
inline void DummyRead(volatile nn::Bit32* address)
{
    (void)(*address);
}

inline void WriteMasked32( volatile nn::Bit32* address, nn::Bit32 value, nn::Bit32 mask)
{
    *address = (*address & ~mask) | (value & mask);
}

template <typename UInt>
inline void SetBit(volatile UInt* address, int value, int pos)
{
    auto tmp = *address;
    if (value)
    {
        tmp |= (static_cast<UInt>(1) << pos);
    }
    else
    {
        tmp &= ~(static_cast<UInt>(1) << pos);
    }
    *address = tmp;
}

template <typename UInt>
inline bool GetBit(volatile UInt* address, int pos)
{
    return (*address >> pos) & 1;
}

/*!
@brief 最下位から連続する 0 のビットの数を返します。(TODO:Alice に Util が追加されたら置き換える)

@param[in]  x   指定する値です。
@return     結果を返します。
*/
inline int cntt0OnBit32( nn::Bit32 x ) NN_NOEXCEPT
{
    x = (x & ~( x - 1 )) - 1;

    x = (x & 0x55555555) + (x >> 1 & 0x55555555);
    x = (x & 0x33333333) + (x >> 2 & 0x33333333);
    x = (x & 0x0f0f0f0f) + (x >> 4 & 0x0f0f0f0f);
    x = (x & 0x00ff00ff) + (x >> 8 & 0x00ff00ff);
    return (x & 0x0000ffff) + (x >>16 & 0x0000ffff);
}


class InternalInterruptEvent
{
public:
    void Initialize(nn::os::InterruptName name, bool isManual) NN_NOEXCEPT
    {
        nn::os::InitializeInterruptEvent(&m_Event, name, isManual? nn::os::EventClearMode_ManualClear: nn::os::EventClearMode_AutoClear);
    }
    void Finalize() NN_NOEXCEPT
    {
        nn::os::FinalizeInterruptEvent(&m_Event);
    }
    void Clear() NN_NOEXCEPT
    {
        nn::os::ClearInterruptEvent(&m_Event);
    }
    void Wait() NN_NOEXCEPT
    {
        nn::os::WaitInterruptEvent(&m_Event);
    }
    bool TryWait() NN_NOEXCEPT
    {
        return nn::os::TryWaitInterruptEvent(&m_Event);
    }
    bool TimedWait(int timeoutUsec) NN_NOEXCEPT
    {
        return nn::os::TimedWaitInterruptEvent(&m_Event, nn::TimeSpan::FromMicroSeconds(timeoutUsec));
    }
    nn::os::InterruptEventType& GetInstance() NN_NOEXCEPT
    {
        return m_Event;
    }

private:
    nn::os::InterruptEventType m_Event;
};

}}}} // nnd::gpio::tegra::detail
