﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   ftm のコントローラに関する定義です。
 */

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/i2c/i2c.h>
#include "ftm_Driver.h"
#include "ftm_Driver-3bd56.h"
#include "ftm_Driver-4cd60d.h"
#include "ftm_Util.h"

namespace nnd { namespace ftm { namespace detail {

namespace {

    IController* g_pController;
    Ftm4cd60dController g_Ftm4cd60dController;
    //Ftm3bd56Controller g_Ftm3bd56Controller; // 今後 Ftm3bd56 が使用されることはない

} // namespace

IController::IController() NN_NOEXCEPT
    : m_MaxTouchNumber(0)
    , m_MaxEventReportCount(0)
    , m_EventReportByteSize(0)
{
}

IController::~IController() NN_NOEXCEPT
{
}

void IController::SetI2cSession(::nn::i2c::I2cSession session) NN_NOEXCEPT
{
    m_Session = session;
}

int IController::GetMaxTouchNumber() const NN_NOEXCEPT
{
    return this->m_MaxTouchNumber;
}

int IController::GetMaxEventReportCount() const NN_NOEXCEPT
{
    return this->m_MaxEventReportCount;
}

size_t IController::GetEventReportByteSize() const NN_NOEXCEPT
{
    return this->m_EventReportByteSize;
}

::nn::Result IController::ResetDevice() const NN_NOEXCEPT
{
    NN_RESULT_SUCCESS;
}

::nn::Result IController::SleepInDevice() const NN_NOEXCEPT
{
    NN_RESULT_SUCCESS;
}

::nn::Result IController::SleepOutDevice() const NN_NOEXCEPT
{
    NN_RESULT_SUCCESS;
}

::nn::Result IController::BindInterrupt() const NN_NOEXCEPT
{
    NN_RESULT_SUCCESS;
}

::nn::Result IController::ActivateSensing() const NN_NOEXCEPT
{
    NN_RESULT_SUCCESS;
}

::nn::Result IController::DeactivateSensing() const NN_NOEXCEPT
{
    NN_RESULT_SUCCESS;
}

::nn::Result IController::ReadLeftEventCount(uint32_t* pOutLeftCount) const NN_NOEXCEPT
{
    NN_UNUSED(pOutLeftCount);
    NN_RESULT_SUCCESS;
}

::nn::Result IController::ReadEventReports(
    char* pOutReadData, uint32_t* pOutReadCount, bool* pOutIsOverflow, uint32_t readCount) const NN_NOEXCEPT
{
    NN_UNUSED(pOutReadData);
    NN_UNUSED(pOutReadCount);
    NN_UNUSED(pOutIsOverflow);
    NN_UNUSED(readCount);
    NN_RESULT_SUCCESS;
}

::nn::Result IController::ReadEventReports(
    char* pOutReadData, uint32_t* pOutReadCount, uint32_t* pOutLeftCount, bool* pOutIsOverflow, uint32_t readCount) const NN_NOEXCEPT
{
    NN_UNUSED(pOutReadData);
    NN_UNUSED(pOutReadCount);
    NN_UNUSED(pOutLeftCount);
    NN_UNUSED(pOutIsOverflow);
    NN_UNUSED(readCount);
    NN_RESULT_SUCCESS;
}

void IController::ParseEventReports(
    EventReport* pOutEventReport, const char* pRawData, uint32_t parseCount) const NN_NOEXCEPT
{
    NN_UNUSED(pOutEventReport);
    NN_UNUSED(pRawData);
    NN_UNUSED(parseCount);
}

::nn::Result IController::UpdateFirmware(FirmwareInputFunctionPointer const pFunction, void* const pParameter, size_t fileSize) NN_NOEXCEPT
{
    NN_UNUSED(pFunction);
    NN_UNUSED(pParameter);
    NN_UNUSED(fileSize);
    NN_RESULT_SUCCESS;
}

::nn::Result IController::EraseFirmware() const NN_NOEXCEPT
{
    NN_RESULT_SUCCESS;
}

::nn::Result IController::ReadVersionAddressRegister(uint16_t* pOutAddress) const NN_NOEXCEPT
{
    const uint8_t HwCommandReadVersionAddress[] = {0xD0, 0x00, 0x60};
    char address[3];

    uint8_t commandList[::nn::i2c::CommandListLengthCountMax];
    ::nn::i2c::CommandListFormatter commandListFormatter(commandList, sizeof(commandList));
    ::std::memset(commandList, 0, sizeof(commandList));

    commandListFormatter.EnqueueSendCommand(I2cTransStart, HwCommandReadVersionAddress, sizeof(HwCommandReadVersionAddress));
    commandListFormatter.EnqueueReceiveCommand(I2cTransStop, sizeof(address));
    NN_RESULT_DO(::nn::i2c::ExecuteCommandList(
        address, sizeof(address), m_Session, commandList, commandListFormatter.GetCurrentLength()));

    // [0:0] Dummy
    // [2:1] Address of register of version information
    *pOutAddress = (address[2] << 8) | address[1];

    NN_RESULT_SUCCESS;
}

::nn::Result IController::ReadVersionRegister(FirmwareVersion* pOutFirmwareVersion, uint16_t address) const NN_NOEXCEPT
{
    const uint8_t ReadVersionCmd[] =
    {
        0xD0,
        static_cast<uint8_t>((address & 0xFF00) >> 8),
        static_cast<uint8_t>(address & 0x00FF)
    };
    char version[9];

    uint8_t commandList[::nn::i2c::CommandListLengthCountMax];
    ::nn::i2c::CommandListFormatter commandListFormatter(commandList, sizeof(commandList));
    ::std::memset(commandList, 0, sizeof(commandList));

    commandListFormatter.EnqueueSendCommand(I2cTransStart, ReadVersionCmd, sizeof(ReadVersionCmd));
    commandListFormatter.EnqueueReceiveCommand(I2cTransStop, sizeof(version));
    NN_RESULT_DO(::nn::i2c::ExecuteCommandList(
        version, sizeof(version), m_Session, commandList, commandListFormatter.GetCurrentLength()));

    // [0:0] Dummy
    // [4:1] Release Version
    // [6:5] Program Version
    // [8:7] Config Version
    pOutFirmwareVersion->release =
        (version[1] << 24) | (version[2] << 16) | (version[3] << 8) | version[4];
    pOutFirmwareVersion->program = (version[6] << 8) | version[5];
    pOutFirmwareVersion->configuration = (version[8] << 8) | version[7];

    NN_RESULT_SUCCESS;
}

::nn::Result IController::ReadFirmwareVersion(FirmwareVersion* pOutFirmwareVersion) const NN_NOEXCEPT
{
    uint16_t address;
    NN_RESULT_DO(this->ReadVersionAddressRegister(&address));
    NN_RESULT_DO(this->ReadVersionRegister(pOutFirmwareVersion, address));

    NN_RESULT_SUCCESS;
}

::nn::Result IController::RunAutoTune() const NN_NOEXCEPT
{
    NN_RESULT_SUCCESS;
}

::nn::Result IController::RunItoTest(ItoEventReport* pOutItoEventReport) const NN_NOEXCEPT
{
    NN_UNUSED(pOutItoEventReport);
    NN_RESULT_SUCCESS;
}

::nn::Result IController::ReadMutualCompensationData(MutualCompensationData* pOutCompensationData) const NN_NOEXCEPT
{
    NN_UNUSED(pOutCompensationData);
    NN_RESULT_SUCCESS;
}

::nn::Result IController::ReadSelfCompensationData(SelfCompensationData* pOutCompensationData) const NN_NOEXCEPT
{
    NN_UNUSED(pOutCompensationData);
    NN_RESULT_SUCCESS;
}

::nn::Result IController::ReadMutualRawData(MutualRawData* pOutRawData) const NN_NOEXCEPT
{
    NN_UNUSED(pOutRawData);
    NN_RESULT_SUCCESS;
}

::nn::Result IController::ReadSelfRawData(SelfRawData* pOutRawData) const NN_NOEXCEPT
{
    NN_UNUSED(pOutRawData);
    NN_RESULT_SUCCESS;
}

::nn::Result IController::RequestGpioState(GpioInformEventReport* pOutGpioInformEventReport) const NN_NOEXCEPT
{
    NN_UNUSED(pOutGpioInformEventReport);
    NN_RESULT_SUCCESS;
}

void InitializeController(::nn::i2c::I2cSession session) NN_NOEXCEPT
{
    // 元々ここで Chip ID を読み出して Ftm3bd56 か Ftm4cd60d かを区別していた
    // しかし FW が書かれていない状態で Initialize() を通す必要が出たため
    // Ftm4cd60d で固定するよう修正した（Ftm3bd56 は致命的なバグがあり今後使用されることはない）
    g_pController = &g_Ftm4cd60dController;
    g_pController->SetI2cSession(session);
}

void FinalizeController() NN_NOEXCEPT
{
    if (g_pController != nullptr)
    {
        g_pController = nullptr;
    }
}

IController* GetController() NN_NOEXCEPT
{
    return g_pController;
}

}}} // namespace nnd::ftm::detail
