﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       タッチコントローラ ftm4cd60d のファームウェアアップデータ用の定義です。
 */

#include <nn/i2c/i2c.h>
#include "ftm_Driver-4cd60d.updater.h"
#include "ftm_Util.h"

namespace nnd { namespace ftm { namespace detail {

namespace {

const uint32_t I2cRetryCountMax         = 3; // リトライの最大回数
const uint32_t I2cAccessRetryIntervalMs = 5; // リトライ時のアクセス間隔

} // namespace

Ftm4cd60dUpdater::Ftm4cd60dUpdater() NN_NOEXCEPT
{
    m_FirmwareInputCallback.pFunction  = nullptr;
    m_FirmwareInputCallback.pParameter = nullptr;
}

void Ftm4cd60dUpdater::SetI2cSession(::nn::i2c::I2cSession session) NN_NOEXCEPT
{
    m_Session = session;
}

void Ftm4cd60dUpdater::RegisterFirmwareInputCallback(
    FirmwareInputFunctionPointer const pFunction, void* const pParameter) NN_NOEXCEPT
{
    m_FirmwareInputCallback.pFunction  = pFunction;
    m_FirmwareInputCallback.pParameter = pParameter;
}

void Ftm4cd60dUpdater::UnregisterFirmwareInputCallback() NN_NOEXCEPT
{
    m_FirmwareInputCallback.pFunction  = nullptr;
    m_FirmwareInputCallback.pParameter = nullptr;
}

::nn::Result Ftm4cd60dUpdater::Proceed(size_t fileSize) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(
        m_FirmwareInputCallback.pFunction != nullptr,
        "Please call nnd::ftm::RegisterFirmwareInputFunction()\n");

    // ファイルサイズと署名が適切か確認する
    NN_RESULT_DO(this->ConfirmFileSize(fileSize));
    NN_RESULT_DO(this->ConfirmSignature());

    // 書き込みデータの CRC チェックを有効にするのは Erase 前である必要がある
    // https://redbtsred.nintendo.co.jp/redmine/issues/3503
    NN_RESULT_DO(this->EnableWarmBootCrcCheck());

    // フラッシュ内のファームウェアを消去する
    NN_RESULT_DO(this->EraseFirmware());

    // ファームウェアの書き込み（ファームウェアは Program 部分と Configration 部分から成る）
    NN_RESULT_DO(this->WriteProgram());
    NN_RESULT_DO(this->WriteConfiguration());

    NN_RESULT_SUCCESS;
}

::nn::Result Ftm4cd60dUpdater::ConfirmFileSize(size_t fileSize) const NN_NOEXCEPT
{
    // File header 32 byte + Flash content 256 kbyte
    // https://redbtsred.nintendo.co.jp/redmine/issues/3683
    const int FirmwareSize = 32 + (256 * 1024);
    NN_RESULT_THROW_UNLESS(fileSize == FirmwareSize, ResultInvalidArgument());
    NN_RESULT_SUCCESS;
}

::nn::Result Ftm4cd60dUpdater::ConfirmSignature() const NN_NOEXCEPT
{
    // https://redbtsred.nintendo.co.jp/redmine/issues/3683
    const size_t SignatureSize = 4;
    const char Signature[SignatureSize] = {0x55, 0x55, 0xAA, 0xAA};
    char buffer[SignatureSize];

    NN_RESULT_DO(this->ReadFirmwareData(buffer, 0, SignatureSize));

    NN_RESULT_THROW_UNLESS(
        ::std::memcmp(buffer, Signature, SignatureSize) == 0, ResultInvalidArgument());

    NN_RESULT_SUCCESS;
}

::nn::Result Ftm4cd60dUpdater::EnableWarmBootCrcCheck() const NN_NOEXCEPT
{
    const uint8_t HwCommandEnableCrcCheck[] = {0xB6, 0x00, 0x1E, 0x38};

    NN_RESULT_DO(this->WriteRegister(HwCommandEnableCrcCheck, sizeof(HwCommandEnableCrcCheck)));
    NN_RESULT_SUCCESS;
}

::nn::Result Ftm4cd60dUpdater::EraseFirmware() const NN_NOEXCEPT
{
    // Erase 操作を受け付けるように前もってアンロックする必要がある
    NN_RESULT_DO(this->UnlockFlash());

    // Erase コマンドの発行
    NN_RESULT_DO(this->IssueEraseFlashCommand());

    // Erase コマンド発行後は完了を待つ必要がある
    NN_RESULT_DO(this->WaitForCompletionOfErasing());

    NN_RESULT_SUCCESS;
}

::nn::Result Ftm4cd60dUpdater::UnlockFlash() const NN_NOEXCEPT
{
    const uint8_t HwCommandUnlockFlash[]      = {0xF7, 0x74, 0x45};
    const uint8_t HwCommandPrepareForUpdate[] = {0xFA, 0x72, 0x03};

    NN_RESULT_DO(this->WriteRegister(HwCommandUnlockFlash, sizeof(HwCommandUnlockFlash)));
    NN_RESULT_DO(this->WriteRegister(HwCommandPrepareForUpdate, sizeof(HwCommandPrepareForUpdate)));
    NN_RESULT_SUCCESS;
}

::nn::Result Ftm4cd60dUpdater::IssueEraseFlashCommand() const NN_NOEXCEPT
{
    const uint8_t HwCommandEraseFlash[] = {0xFA, 0x02, 0xC0};

    NN_RESULT_DO(this->WriteRegister(HwCommandEraseFlash, sizeof(HwCommandEraseFlash)));
    NN_RESULT_SUCCESS;
}

::nn::Result Ftm4cd60dUpdater::WaitForCompletionOfErasing() const NN_NOEXCEPT
{
    // Erase のタイムアウトは 80ms までみる（仕様上は 40ms 以内に完了する）
    // https://redbtsred.nintendo.co.jp/redmine/issues/3412
    const uint32_t EraseTimeOutMs = 80;

    const char CompleteStateMask  = 0x80;
    bool isTimeout = false;
    ::nn::os::Tick tick = ::nn::os::GetSystemTick();
    while (isTimeout == false)
    {
        ::nn::TimeSpan timeSpan = ::nn::os::ConvertToTimeSpan(::nn::os::GetSystemTick() - tick);
        isTimeout = (timeSpan.GetMilliSeconds() > EraseTimeOutMs) ? true : false;

        char eraseState;
        NN_RESULT_DO(this->ReadEraseState(&eraseState, sizeof(eraseState)));

        if ((eraseState & CompleteStateMask) == 0)
        {
            NN_RESULT_SUCCESS;
        }
    }

    NN_RESULT_THROW(ResultTimeoutEraseFlash());
}

::nn::Result Ftm4cd60dUpdater::ReadEraseState(char* pOutEraseState, size_t size) const NN_NOEXCEPT
{
    const uint8_t HwCommandReadEraseState[] = {0xF9, 0x02};
    NN_RESULT_DO(this->ReadRegister(pOutEraseState, size, HwCommandReadEraseState, sizeof(HwCommandReadEraseState)));

    NN_RESULT_SUCCESS;
}

::nn::Result Ftm4cd60dUpdater::WriteProgram() const NN_NOEXCEPT
{
    const uint32_t ProgramDataOffset  = 0x20;         // ファームウェア内のオフセット
    const uint32_t ProgramDataSizeMax = 252 * 1024;   // データの最大サイズ
    const uint16_t ProgramAddress     = 0x0000;       // 転送先フラッシュ内のアドレス

    // 実際にデータが書かれている部分のサイズを調べてその部分だけを書き込む
    uint32_t dataSize;
    NN_RESULT_DO(this->GetActualProgramDataSize(&dataSize, ProgramDataOffset, ProgramDataSizeMax));
    NN_RESULT_DO(this->WriteOnFlashMemory(ProgramDataOffset, dataSize, ProgramAddress));

    NN_RESULT_SUCCESS;
}

::nn::Result Ftm4cd60dUpdater::WriteConfiguration() const NN_NOEXCEPT
{
    const uint32_t ConfigurationDataOffset  = 0x3F020;  // ファームウェア内のオフセット
    const uint32_t ConfigurationDataSizeMax = 2 * 1024; // データの最大サイズ
    const uint16_t ConfigurationAddress     = 0xFC00;   // 転送先フラッシュ内のアドレス

    // Configuration データは不使用領域がないので実サイズを調べる必要がない
    NN_RESULT_DO(this->WriteOnFlashMemory(ConfigurationDataOffset, ConfigurationDataSizeMax, ConfigurationAddress));

    NN_RESULT_SUCCESS;
}

::nn::Result Ftm4cd60dUpdater::GetActualProgramDataSize(uint32_t* pOutSize, uint32_t offset, uint32_t dataSizeMax) const NN_NOEXCEPT
{
    const size_t BufferSize = 512;
    char buffer[BufferSize];
    uint32_t dataSize;

    // バッファ内がすべて 0xFF で埋められているか判定
    auto IsAllF = [](const char* buffer, size_t BufferSize) -> bool
    {
        for (size_t i = 0; i < BufferSize; i++)
        {
            if (buffer[i] != 0xFF)
            {
                return false;
            }
        }
        return true;
    };

    // データ末尾から逆読みして 0xFF 以外の文字が出てくるところを見つける
    for (dataSize = dataSizeMax; dataSize > offset + BufferSize; dataSize -= BufferSize)
    {
        uint32_t dataOffset = offset + dataSize - BufferSize;
        NN_RESULT_DO(ReadFirmwareData(buffer, dataOffset, BufferSize));

        if (IsAllF(buffer, BufferSize) == false)
        {
            break;
        }
    }

    *pOutSize = dataSize;

    NN_RESULT_SUCCESS;
}

::nn::Result Ftm4cd60dUpdater::WriteOnFlashMemory(uint32_t offset, uint32_t dataSize, uint16_t address) const NN_NOEXCEPT
{
    const size_t  BufferSize = 512;
    const uint8_t SectorByteSize = 4;  // フラッシュの 1 セクタに書き込めるサイズ
    char buffer[BufferSize];

    // セクタの書き込み単位で RoundUp する
    dataSize = (dataSize + (SectorByteSize - 1)) & ~(SectorByteSize - 1);

    // バッファサイズに制約があるので数回に分けて転送する
    for (size_t writtenSize = 0; writtenSize < dataSize; writtenSize += BufferSize)
    {
        // ファームウェアデータを読み込む
        const uint32_t FileAddress = static_cast<int32_t>(writtenSize) + offset;
        NN_RESULT_DO(this->ReadFirmwareData(buffer, FileAddress, BufferSize));

        // データを IC 内の RAM に転送する
        const uint32_t LeftSize = dataSize - writtenSize;
        const uint32_t Size = LeftSize < BufferSize ? LeftSize : BufferSize;
        NN_RESULT_DO(this->LoadFirmwareIntoRam(buffer, Size));

        // IC 内 RAM 上のデータをフラッシュに書き込む
        const uint32_t SectorCount = (Size / SectorByteSize) - 1; // 実際の転送数より 1 減らす (https://redbtsred.nintendo.co.jp/redmine/issues/3692)
        const uint16_t WrittenSectorCount = static_cast<uint16_t>(writtenSize / SectorByteSize);
        NN_RESULT_DO(this->StartFlashDma(address + WrittenSectorCount, static_cast<uint16_t>(SectorCount)));
    }

    NN_RESULT_SUCCESS;
}

::nn::Result Ftm4cd60dUpdater::ReadFirmwareData(char* pOutBuffer, uint32_t address, size_t size) const NN_NOEXCEPT
{
    NN_RESULT_DO(m_FirmwareInputCallback.pFunction(
        pOutBuffer, m_FirmwareInputCallback.pParameter, static_cast<int64_t>(address), size));
    NN_RESULT_SUCCESS;
}

::nn::Result Ftm4cd60dUpdater::LoadFirmwareIntoRam(const char* pBuffer, uint32_t size) const NN_NOEXCEPT
{
    const uint8_t  HwCommandWriteDataConstant = 0xF8;
    const uint8_t  HeaderByteSize   = 3;   // 転送時データに付けるヘッダのサイズ
    const uint32_t SendUnitByteSize = 32;  // 一度に転送するデータのサイズ（任意）

    for (uint32_t index = 0; index < size; index += SendUnitByteSize)
    {
        uint8_t programCmd[HeaderByteSize + SendUnitByteSize];
        ::std::memset(programCmd, 0, sizeof(programCmd));

        programCmd[0] = HwCommandWriteDataConstant;
        programCmd[1] = static_cast<uint8_t>((index & 0xFF00) >> 8);      // RAM 上アドレス
        programCmd[2] = static_cast<uint8_t>((index & 0x00FF));           // RAM 上アドレス
        ::std::memcpy(&programCmd[3], &pBuffer[index], SendUnitByteSize); // データの実体

        NN_RESULT_DO(this->WriteRegister(programCmd, sizeof(programCmd)));
    }

    NN_RESULT_SUCCESS;
}

::nn::Result Ftm4cd60dUpdater::StartFlashDma(uint16_t offset, uint16_t sectorCount) const NN_NOEXCEPT
{
    // DMA 設定後に開始トリガをかけ完了を待つ
    NN_RESULT_DO(this->ConfigureFlashDma(offset, sectorCount));
    NN_RESULT_DO(this->IssueStartFlashDmaCommand());
    NN_RESULT_DO(this->WaitForCompletionOfDma());

    NN_RESULT_SUCCESS;
}

::nn::Result Ftm4cd60dUpdater::ConfigureFlashDma(uint16_t offset, uint16_t sectorCount) const NN_NOEXCEPT
{
    const uint8_t HwCommandConfigureDma[] = {0xFA, 0x06, 0x00, 0x00};

    // DMA の Config 情報を設定する
    const uint8_t configureCmd[] =
    {
        HwCommandConfigureDma[0],
        HwCommandConfigureDma[1],
        HwCommandConfigureDma[2],
        HwCommandConfigureDma[3],
        static_cast<uint8_t>(offset & 0x00FF),               // 書き込みを開始するセクタアドレス
        static_cast<uint8_t>((offset & 0xFF00) >> 8),        // 書き込みを開始するセクタアドレス
        static_cast<uint8_t>(sectorCount & 0x00FF),          // 書き込むセクタ数
        static_cast<uint8_t>((sectorCount & 0xFF00) >> 8),   // 書き込むセクタ数
        0                                                    // https://redbtsred.nintendo.co.jp/redmine/issues/3692
    };

    NN_RESULT_DO(this->WriteRegister(configureCmd, sizeof(configureCmd)));
    NN_RESULT_SUCCESS;
}

::nn::Result Ftm4cd60dUpdater::IssueStartFlashDmaCommand() const NN_NOEXCEPT
{
    const uint8_t HwCommandStartFlashDma[] = {0xFA, 0x05, 0xC0};

    NN_RESULT_DO(this->WriteRegister(HwCommandStartFlashDma, sizeof(HwCommandStartFlashDma)));
    NN_RESULT_SUCCESS;
}

::nn::Result Ftm4cd60dUpdater::WaitForCompletionOfDma() const NN_NOEXCEPT
{
    // FW Update のタイムアウトは 4000ms までみる
    // https://redbtsred.nintendo.co.jp/redmine/issues/3412
    const uint32_t UpdateTimeOutMs = 4000;

    const char CompleteStateMask  = 0x80;
    bool isTimeout = false;
    ::nn::os::Tick tick = ::nn::os::GetSystemTick();
    while (isTimeout == false)
    {
        ::nn::TimeSpan timeSpan = ::nn::os::ConvertToTimeSpan(::nn::os::GetSystemTick() - tick);
        isTimeout = (timeSpan.GetMilliSeconds() > UpdateTimeOutMs) ? true : false;

        char dmaState;
        NN_RESULT_DO(this->ReadDmaState(&dmaState, sizeof(dmaState)));

        if ((dmaState & CompleteStateMask) == 0)
        {
            NN_RESULT_SUCCESS;
        }
    }

    NN_RESULT_THROW(ResultTimeoutDma());
}

::nn::Result Ftm4cd60dUpdater::ReadDmaState(char* pOutDmaState, size_t size) const NN_NOEXCEPT
{
    const uint8_t HwCommandReadDmaState[] = {0xF9, 0x05};
    NN_RESULT_DO(this->ReadRegister(pOutDmaState, size, HwCommandReadDmaState, sizeof(HwCommandReadDmaState)));

    NN_RESULT_SUCCESS;
}

::nn::Result Ftm4cd60dUpdater::WriteRegister(const void* pInData, size_t dataBytes) const NN_NOEXCEPT
{
    ::nn::Result result;
    for (uint32_t retry = 0; retry < I2cRetryCountMax; retry++)
    {
        result = ::nn::i2c::Send(m_Session, pInData, dataBytes, I2cTransStartStop);

        if (::nn::i2c::ResultBusBusy::Includes(result))
        {
            ::nn::os::SleepThread(
                ::nn::TimeSpan::FromMilliSeconds(I2cAccessRetryIntervalMs));
            continue;
        }

        break;
    }

    return result;
}

::nn::Result Ftm4cd60dUpdater::ReadRegister(
    void* pBuffer, size_t receiveDataBytes, const nn::Bit8* pInData, size_t sendDataBytes) const NN_NOEXCEPT
{
    ::nn::Result result;
    for (uint32_t retry = 0; retry < I2cRetryCountMax; retry++)
    {
        uint8_t commandList[::nn::i2c::CommandListLengthCountMax];
        ::nn::i2c::CommandListFormatter commandListFormatter(commandList, sizeof(commandList));
        ::std::memset(commandList, 0, sizeof(commandList));

        commandListFormatter.EnqueueSendCommand(I2cTransStart, pInData, sendDataBytes);
        commandListFormatter.EnqueueReceiveCommand(I2cTransStop, receiveDataBytes);

        result = ::nn::i2c::ExecuteCommandList(
            pBuffer, receiveDataBytes, m_Session, commandList, commandListFormatter.GetCurrentLength());

        if (::nn::i2c::ResultBusBusy::Includes(result))
        {
            ::nn::os::SleepThread(
                ::nn::TimeSpan::FromMilliSeconds(I2cAccessRetryIntervalMs));
            continue;
        }

        break;
    }

    return result;
}

}}} // namespace nnd::ftm::detail
