﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   Jetson TK2 向けの ftm 関連のペリフェラル設定値です。
 */

#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/nn_SdkAssert.h>

#include <nn/i2c/i2c.h>
#include <nn/i2c/i2c_BusDev.h>
#include <nn/gpio/gpio.h>
#include <nn/gpio/gpio_PadAccessorDev.h>

#include "ftm_BusConfig.h"
#include "ftm_Util.h"
#include "ftm_Debug.h"

namespace nnd { namespace ftm { namespace detail {

namespace {

const int I2cBusIdx                 =  2; // I2C3
const int GpioPadNumberInterrupt    = 33; // GPIO3_PE.01
// JetsonTK2にResetピンは無い

const ::nn::Bit16                 I2cSlaveAddr      = 0x92 >> 1;    // タッチICのスレーブアドレス
const ::nn::i2c::AddressingMode   I2cAddressingMode = ::nn::i2c::AddressingMode_BitWidth7;
const ::nn::i2c::SpeedMode        I2cSpeedMode      = ::nn::i2c::SpeedMode_Fast;

// Low レベルになったとき割り込みが入るようにする
const ::nn::gpio::InterruptMode   GpioInterruptMode   = ::nn::gpio::InterruptMode::InterruptMode_LowLevel;

// バスの状態
enum class BusState : uint8_t
{
    Uninitialized, // 未初期化
    Initialized,   // 初期化済み
};
BusState g_BusState = BusState::Uninitialized;

} // namespace

void InitializeBus(::nn::i2c::I2cSession* pOutI2c, ::nn::gpio::GpioPadSession* pOutIrq, ::nn::gpio::GpioPadSession* pOutReset) NN_NOEXCEPT
{
    // すでに初期化済みの場合は何もしない
    if (g_BusState == BusState::Initialized)
    {
        return;
    }

    // I2C -------------------------
    NN_FTM_LOG("%s(%d)\t: Initialize I2C\n", __FUNCTION__, __LINE__);
    ::nn::i2c::Initialize();

    NN_FTM_LOG("%s(%d)\t: NN_SDK_ASSERT(I2cHasDevice(pOutI2c));\n", __FUNCTION__, __LINE__);

    NN_FTM_LOG("%s(%d)\t: ::nn::i2c::OpenSessionForDev\n", __FUNCTION__, __LINE__);
    ::nn::i2c::OpenSessionForDev(
        pOutI2c,
        I2cBusIdx,
        I2cSlaveAddr,
        I2cAddressingMode,
        I2cSpeedMode);

    // GPIO -------------------------
    NN_FTM_LOG("%s(%d)\t: Initialize GPIO\n", __FUNCTION__, __LINE__);
    ::nn::gpio::Initialize();

    // 割り込みピン
    ::nn::gpio::OpenSessionForDev(pOutIrq, GpioPadNumberInterrupt);
    ::nn::gpio::SetDirection(pOutIrq, ::nn::gpio::Direction_Input);
    ::nn::gpio::SetInterruptMode(pOutIrq, GpioInterruptMode);

    // リセットピン
    NN_UNUSED(pOutReset);
    // ::nn::gpio::OpenSessionForDev(pOutReset, );    // JetsonTK2 には Reset ピンは用意されていない

    g_BusState = BusState::Initialized;
}

void FinalizeBus(::nn::i2c::I2cSession* pI2c, ::nn::gpio::GpioPadSession* pIrq, ::nn::gpio::GpioPadSession* pReset) NN_NOEXCEPT
{
    // すでにファイナライズ済みの場合は何もしない
    if (g_BusState == BusState::Uninitialized)
    {
        return;
    }

    // GPIO -------------------------
    // 割り込みピン
    ::nn::gpio::CloseSession(pIrq);

    // リセットピン
    NN_UNUSED(pReset);
    // ::nn::gpio::CloseSession(pReset);    // JetsonTK2 には Reset ピンは用意されていない

    ::nn::gpio::Finalize();

    // I2C -------------------------
    ::nn::i2c::CloseSession(*pI2c);
    ::nn::i2c::Finalize();

    g_BusState = BusState::Uninitialized;
}

void SleepInBus(::nn::i2c::I2cSession* pI2c, ::nn::gpio::GpioPadSession* pIrq, ::nn::gpio::GpioPadSession* pReset) NN_NOEXCEPT
{
    FinalizeBus(pI2c, pIrq, pReset);
}

void SleepOutBus(::nn::i2c::I2cSession* pOutI2c, ::nn::gpio::GpioPadSession* pOutIrq, ::nn::gpio::GpioPadSession* pOutReset) NN_NOEXCEPT
{
    InitializeBus(pOutI2c, pOutIrq, pOutReset);
}

// JetsonTK2 では電源は常通となっているため電源投入処理不要
void SetPowerEnabled(bool enabled) NN_NOEXCEPT
{
    NN_UNUSED(enabled);
}

}}} // namespace nnd::ftm::detail
