﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/i2c/i2c.h>
#include <nn/i2c/i2c_BusDev.h>

#include "bh1730fvc_BusConfig.h"

#include "bh1730fvc_Debug.h"

namespace nnd {
namespace bh1730fvc {
namespace detail {

namespace{
const nn::i2c::I2cDevice    I2cDeviceName   = nn::i2c::I2cDevice_Bh1730;

}

void InitializeBus(nn::i2c::I2cSession* pOutI2cSession) NN_NOEXCEPT
{
    NND_BH1730FVC_DETAIL_LOG("START\n");

    // 電源制御
    SetBusPowerEnabled(true);

    // I2C -------------------------
    nn::i2c::Initialize();
    nn::i2c::OpenSession(pOutI2cSession, I2cDeviceName);

    // 2.9V ラインの上昇をトリガにする 1.8V ラインの上昇を待つため 1ms の待機が必要
    ::nn::os::SleepThread(::nn::TimeSpan::FromMilliSeconds(1));
}

void FinalizeBus(nn::i2c::I2cSession* pI2cSession) NN_NOEXCEPT
{
    NND_BH1730FVC_DETAIL_LOG("START\n");

    // I2C -------------------------
    nn::i2c::CloseSession(*pI2cSession);
    nn::i2c::Finalize();

    // 電源制御
    SetBusPowerEnabled(false);
}

/**
 * NX でのエラーハンドリングポリシー
 *
 * # 上位レイヤーに返す
 * ResultSuccess            : 成功
 * ResultBusBusy            : リトライで対策可能なため return
 * ResultNoAck              : 回路的な不具合と思われるが、システムの動作を止めるべきかどうかは上位レイヤの判断に委ねるため return
 *
 * # ドライバ内部で止める
 * ResultCommandListFull    : ドライバの実装不備のため Abort
 */
nn::Result CheckI2cError(nn::Result result) NN_NOEXCEPT
{
#if 0
    // I2C アクセスでエラーが発生しても Abort せずに result の詳細を表示する
    NN_BH1730FVC_LOG_RESULT(result);
    NN_RESULT_THROW(result);
#else
    NN_RESULT_TRY(result)
        NN_RESULT_CATCH(nn::i2c::ResultBusBusy)
        {
            NN_RESULT_THROW(result);
        }
        NN_RESULT_CATCH(nn::i2c::ResultNoAck)
        {
            NN_RESULT_THROW(result);
        }
        NN_RESULT_CATCH_ALL
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }
    NN_RESULT_END_TRY

    NN_RESULT_SUCCESS;
#endif
}

// falseの場合はタイムアウト
bool IsReachTargetVoltage(::nn::i2c::I2cSession i2cSession) NN_NOEXCEPT
{
    // nx では nn::i2c::OpenSession() と ::nn::i2c::CloseSession(); により電源制御される
    NN_UNUSED(i2cSession);

    return true;
}


void SetBusPowerEnabled(bool enabled) NN_NOEXCEPT
{
    // nx では nn::i2c::OpenSession() と ::nn::i2c::CloseSession(); により電源制御される
    NN_UNUSED(enabled);
}

} // detail
} // bh1730fvc
} // nnd
