﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>

#include <nn/result/result_HandlingUtility.h>

#include <nn/i2c/i2c.h>
#include <nnd/bh1730fvc/bh1730fvc.h>

#include "detail/bh1730fvc_Driver.h"
#include "detail/bh1730fvc_Specification.h"

#include "detail/bh1730fvc_Debug.h"

namespace nnd {
namespace bh1730fvc {

namespace{
bool                g_IsInitialized  = false;
nn::i2c::I2cSession g_I2cSessions;
}

bool Initialize() NN_NOEXCEPT
{
    NND_BH1730FVC_LOG("START\n");
    if (g_IsInitialized)
    {
        NND_BH1730FVC_LOG("Already initialized.\n");
        return true;
    }

    if(!detail::Initialize(&g_I2cSessions))
    {
        return false;
    }

    g_IsInitialized = true;
    return true;
}


void Finalize() NN_NOEXCEPT
{
    NND_BH1730FVC_LOG("START\n");
    NN_SDK_REQUIRES(g_IsInitialized,"Please call nnd::bh1730fvc::Initialize()\n");

    detail::Finalize(&g_I2cSessions);

    g_IsInitialized = false;
}


nn::Result GetMeasurementMode(MeasurementMode* pOutMode) NN_NOEXCEPT
{
    NND_BH1730FVC_LOG("START\n");
    NN_SDK_REQUIRES(g_IsInitialized,"Please call nnd::bh1730fvc::Initialize()\n");
    NN_SDK_REQUIRES_NOT_NULL(pOutMode);

    NN_RESULT_DO((detail::GetMeasurementMode(g_I2cSessions, pOutMode)));
    NN_RESULT_SUCCESS;
}


nn::Result SetMeasurementMode(MeasurementMode mode) NN_NOEXCEPT
{
    NND_BH1730FVC_LOG("START\n");
    NN_SDK_REQUIRES(g_IsInitialized,"Please call nnd::bh1730fvc::Initialize()\n");

    NN_RESULT_DO(detail::SetMeasurementMode(g_I2cSessions, mode));
    NN_RESULT_SUCCESS;
}


nn::Result GetMeasurementValue(MeasurementValue* pOutValue) NN_NOEXCEPT
{
    NND_BH1730FVC_LOG("START\n");
    NN_SDK_REQUIRES(g_IsInitialized,"Please call nnd::bh1730fvc::Initialize()\n");
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    NN_RESULT_DO(detail::GetMeasurementValue(g_I2cSessions, pOutValue));
    NN_RESULT_SUCCESS;
}


float CalculateLux(int visible, int infrared, Gain gain, uint8_t cycle) NN_NOEXCEPT
{
    NND_BH1730FVC_LOG("START\n");
    NN_SDK_REQUIRES(g_IsInitialized,"Please call nnd::bh1730fvc::Initialize()\n");

    bool isOverflown;
    NN_UNUSED(isOverflown);
    return detail::CalculateLux(visible, infrared, gain, static_cast<nn::Bit8>(cycle), &isOverflown);
}

nn::Result IsLuxUpdated(bool* pOutUpdated) NN_NOEXCEPT
{
    NND_BH1730FVC_LOG("START\n");
    NN_SDK_REQUIRES(g_IsInitialized,"Please call nnd::bh1730fvc::Initialize()\n");
    NN_SDK_REQUIRES_NOT_NULL(pOutUpdated);

    NN_RESULT_DO(detail::IsLuxUpdated(g_I2cSessions, pOutUpdated));
    NN_RESULT_SUCCESS;
}


nn::Result ReadRevisionId(nn::Bit8* pOutId) NN_NOEXCEPT
{
    NND_BH1730FVC_LOG("START\n");
    NN_SDK_REQUIRES(g_IsInitialized,"Please call nnd::bh1730fvc::Initialize()\n");
    NN_SDK_REQUIRES_NOT_NULL(pOutId);

    NN_RESULT_DO(detail::ReadRevisionId(g_I2cSessions, pOutId));
    NN_RESULT_SUCCESS;
}


nn::Result GetIntegralCycle(int* pOutCycle) NN_NOEXCEPT
{
    NND_BH1730FVC_LOG("START\n");
    NN_SDK_REQUIRES(g_IsInitialized,"Please call nnd::bh1730fvc::Initialize()\n");
    NN_SDK_REQUIRES_NOT_NULL(pOutCycle);

    NN_RESULT_DO(detail::GetIntegralCycle(g_I2cSessions, pOutCycle));
    NN_RESULT_SUCCESS;
}


nn::Result SetIntegralCycle(int cycle) NN_NOEXCEPT
{
    NND_BH1730FVC_LOG("START\n");
    NN_SDK_REQUIRES(g_IsInitialized,"Please call nnd::bh1730fvc::Initialize()\n");
    MeasurementMode mode;
    NN_RESULT_DO(GetMeasurementMode(&mode));
    NN_SDK_REQUIRES(mode == MeasurementMode::Standby, "計測に関する設定は Standby モード時に実行してください。");
    NN_SDK_ASSERT_RANGE(cycle, detail::IntegralCycleMin, detail::IntegralCycleMax + 1);

    NN_RESULT_DO(detail::SetIntegralCycle(g_I2cSessions, cycle));
    NN_RESULT_SUCCESS;
}


nn::Result GetGain(Gain* pOutGain) NN_NOEXCEPT
{
    NND_BH1730FVC_LOG("START\n");
    NN_SDK_REQUIRES(g_IsInitialized,"Please call nnd::bh1730fvc::Initialize()\n");
    NN_SDK_REQUIRES_NOT_NULL(pOutGain);

    NN_RESULT_DO(detail::GetGain(g_I2cSessions, pOutGain));
    NN_RESULT_SUCCESS;
}


nn::Result SetGain(Gain gain) NN_NOEXCEPT
{
    NND_BH1730FVC_LOG("START\n");
    NN_SDK_REQUIRES(g_IsInitialized,"Please call nnd::bh1730fvc::Initialize()\n");
    MeasurementMode mode;
    NN_RESULT_DO(GetMeasurementMode(&mode));
    NN_SDK_REQUIRES(mode == MeasurementMode::Standby, "計測に関する設定は Standby モード時に実行してください。");

    NN_RESULT_DO(detail::SetGain(g_I2cSessions, gain));
    NN_RESULT_SUCCESS;
}


} // bh1730fvc
} // nnd
