﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file
 * @brief       tmp451 ライブラリのローカルとリモート計測共通の動作設定の API リストです。
 */

#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/os/os_SystemEvent.h>

#include <nnd/tmp451/tmp451.h>

#include "tmp451_Types.h"

namespace nnd {
namespace tmp451 {

//! @name   ローカルとリモート計測共通の動作設定
//! @{

/**
 * @brief       計測温度のレンジ設定を取得します。
 *
 * @param[out]  pOutRange   計測温度のレンジ設定
 *
 * @return      処理の結果を返します。
 * @retval      ResultSuccess
 *                  成功しました。
 *
 * @pre         初期化済み状態である必要があります。
 *
 * @details     計測可能な温度値のレンジ設定を取得します。
 */
nn::Result GetMeasurementRange(MeasurementRange* pOutRange) NN_NOEXCEPT;

/**
 * @brief       計測温度のレンジを設定します。
 *
 * @param[in]   range   計測温度のレンジ設定
 *
 * @return      処理の結果を返します。
 * @retval      ResultSuccess
 *                  成功しました。
 * @retval      ResultInvalidState
 *                  不正なステートであるため、処理を実行出来ませんでした。
 *                  設定したいモードに遷移するための前提状態を満たしていない可能性があります。
 *
 * @pre         Standby 状態である必要があります。
 *
 * @post        計測可能な温度値のレンジが変更されます。@n
 *              温度検知機能に対する限度値を設定している場合は、
 *              必ず再設定を行ってください。再設定を行わない場合、
 *              意図せぬ温度値で検知が行われる可能性があります。@n
 *              本関数が完了した次の計測から、変更が反映されます。
 *
 * @details     計測可能な温度値のレンジを設定します。
 */
nn::Result SetMeasurementRange(MeasurementRange range) NN_NOEXCEPT;

/**
 * @brief       計測頻度の設定を取得します。
 *
 * @param[out]  pOutN   計測頻度算出式の n の値
 *
 * @return      処理の結果を返します。
 * @retval      ResultSuccess
 *                  成功しました。
 *
 * @pre         初期化済み状態である必要があります。
 *
 * @details     設定可能なレートは 2^n  {n | -4 ≤ n ≤ 5} であり、
 *              本関数の返り値は式中の n で返します。
 */
nn::Result GetMeasurementRateBy2ToThePowerN(int* pOutN) NN_NOEXCEPT;

/**
 * @brief       計測頻度を設定します。
 *
 * @param[in]   n       計測頻度算出式の n の値
 *
 * @return      処理の結果を返します。
 * @retval      ResultSuccess
 *                  成功しました。
 * @retval      ResultInvalidState
 *                  不正なステートであるため、処理を実行出来ませんでした。
 *                  設定したいモードに遷移するための前提状態を満たしていない可能性があります。
 *
 * @pre         Standby 状態である必要があります。
 *
 * @post        本関数が完了した次の計測から、変更が反映されます。
 *
 * @details     設定可能なレートは 2^n  {n | -4 ≤ n ≤ 5} であり、
 *              本関数の引数には式中の n を指定します。
 */
nn::Result SetMeasurementRateBy2ToThePowerN(int n) NN_NOEXCEPT;

/**
 * @brief       バスのタイムアウト機能の有効化状態を取得します。
 *
 * @param[out]  pOutEnable  バスのタイムアウト機能の有効化状態
 *                          true：有効
 *                          false：無効
 *
 * @return      処理の結果を返します。
 * @retval      ResultSuccess
 *                  成功しました。
 *
 * @pre         初期化済み状態である必要があります。
 */
nn::Result GetBusTimeoutFunctionEnabled(bool* pOutEnable) NN_NOEXCEPT;

/**
 * @brief       バスのタイムアウト機能の有効化状態を設定します。
 *
 * @param[in]   enable  バスのタイムアウト機能の有効化状態
 *                      true：有効
 *                      false：無効
 *
 * @return      処理の結果を返します。
 * @retval      ResultSuccess
 *                  成功しました。
 * @retval      ResultInvalidState
 *                  不正なステートであるため、処理を実行出来ませんでした。
 *                  設定したいモードに遷移するための前提状態を満たしていない可能性があります。
 *
 * @pre         Standby 状態である必要があります。
 *
 * @post        本関数が完了した次の計測から、変更が反映されます。
 *
 * @details     有効にした場合、バスの START から STOP までの間で 25ms の間
 *              反応が無ければ、デバイスは一旦バスを開放した後に
 *              次の START の待ち状態に戻ります。
 */
nn::Result SetBusTimeoutFunctionEnabled(bool enable) NN_NOEXCEPT;

//! @}



} // tmp451
} // nnd
