﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file
 * @brief       tmp451 ライブラリの温度検知機能に関する動作設定の API リストです。
 */

#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/os/os_SystemEvent.h>

#include <nnd/tmp451/tmp451.h>

#include "tmp451_Types.h"

namespace nnd {
namespace tmp451 {

//! @name 温度検知機能に関する動作設定
//! @{

/**
 * @brief       シグナルピン 1 に対する温度検知機能の動作設定を取得します。
 *
 * @param[out]  pOutRemoteLimit     リモート計測点に対する上限値
 * @param[out]  pOutLocalLimit      ローカル計測点に対する上限値
 *
 * @return      処理の結果を返します。
 * @retval      ResultSuccess
 *                  成功しました。
 *
 * @pre         初期化済み状態である必要があります。
 *
 * @details     シグナルピン 1 に対する限度値の設定を取得します。@n
 *              シグナルピン 1 は THERM 機能による検知が行われます。@n
 *              デバイスの仕様により小数部の設定は非対応なため、
 *              取得できる小数部は常に 0 になります。
 */
nn::Result GetPin1DetectionSettings(Temperature* pOutRemoteLimit, Temperature* pOutLocalLimit) NN_NOEXCEPT;

/**
 * @brief       シグナルピン 1 に対する温度検知機能の動作設定を設定します。
 *
 * @param[in]   remoteLimit     リモート計測点に対する上限値
 * @param[in]   localLimit      ローカル計測点に対する上限値
 *
 * @return      処理の結果を返します。
 * @retval      ResultSuccess
 *                  成功しました。
 * @retval      ResultInvalidState
 *                  不正なステートであるため、処理を実行出来ませんでした。
 *                  設定したいモードに遷移するための前提状態を満たしていない可能性があります。
 *
 * @pre         Standby 状態である必要があります。
 *
 * @post        本関数が完了した次の計測から、変更が反映されます。
 *
 * @details     シグナルピン 1 に対する限度値を設定します。@n
 *              シグナルピン 1 は THERM 機能による検知が行われます。@n
 *              デバイスの仕様により小数部の設定は非対応なため、
 *              小数部の値は無視されます。
 */
nn::Result SetPin1DetectionSettings(Temperature remoteLimit, Temperature localLimit) NN_NOEXCEPT;

/**
 * @brief       シグナルピン 2 に対する温度検知機能の動作設定を取得します。
 *
 * @param[out]  pOutRemoteLowLimit      リモート計測点に対する下限値
 * @param[out]  pOutRemoteHighLimit     ローカル計測点に対する上限値
 * @param[out]  pOutLocalLowLimit       ローカル計測点に対する下限値
 * @param[out]  pOutLocalHighLimit      ローカル計測点に対する上限値
 * @param[out]  pOutDetection           温度検知機能のモード
 *
 * @return      処理の結果を返します。
 * @retval      ResultSuccess
 *                  成功しました。
 *
 * @pre         初期化済み状態である必要があります。
 *
 * @details     シグナルピン 2 に対する限度値の設定を取得します。@n
 *              デバイスの仕様によりローカルの上限値と下限値の
 *              小数部の設定は非対応なため、取得できる小数部は常に 0 になります。
 */
nn::Result GetPin2DetectionSettings(Temperature* pOutRemoteLowLimit, Temperature* pOutRemoteHighLimit,
                                    Temperature* pOutLocalLowLimit, Temperature* pOutLocalHighLimit,
                                    DetectionMode* pOutDetection) NN_NOEXCEPT;

/**
 * @brief       シグナルピン 2 に対する温度検知機能の動作設定を設定します。
 *
 * @param[in]   remoteLowLimit      リモート計測点に対する下限値
 * @param[in]   remoteHighLimit     ローカル計測点に対する上限値
 * @param[in]   locaLowlLimit       ローカル計測点に対する下限値
 * @param[in]   localHighLimit      ローカル計測点に対する上限値
 * @param[in]   mode                温度検知機能のモード
 *
 * @return      処理の結果を返します。
 * @retval      ResultSuccess
 *                  成功しました。
 * @retval      ResultInvalidState
 *                  不正なステートであるため、処理を実行出来ませんでした。
 *                  設定したいモードに遷移するための前提状態を満たしていない可能性があります。
 *
 * @pre         Standby 状態である必要があります。
 *
 * @post        本関数が完了した次の計測から、変更が反映されます。
 *
 * @details     シグナルピン 2 に対する限度値の設定をします。@n
 *              デバイスの仕様によりローカルの上限値と下限値の
 *              小数部の設定は非対応なため、それらの小数部の値は無視されます。
 */
nn::Result SetPin2DetectionSettings(Temperature remoteLowLimit, Temperature remoteHighLimit,
                                    Temperature locaLowlLimit, Temperature localHighLimit,
                                    DetectionMode mode) NN_NOEXCEPT;

/**
 * @brief       温度検知による割り込みを受け取るためのイベントオブジェクトを紐付けます。
 *
 * @param[in]   pEvent          登録するイベントオブジェクト
 * @param[in]   pin             割り込みを受け取りたいピン
 *
 * @return      処理の結果を返します。
 * @retval      ResultSuccess
 *                  成功しました。
 * @retval      ResultInvalidState
 *                  不正なステートであるため、処理を実行出来ませんでした。
 *                  設定したいモードに遷移するための前提状態を満たしていない可能性があります。
 * @retval      ResultAlreadyBound
 *                  すでにイベントが紐付けされています。
 *
 *
 * @pre         Standby 状態である必要があります。
 *
 * @post        デバイスからの割り込みを検知できる状態になります。
 *              (pEvent は本関数内で初期化されます。)
 *              pEvent が紐付けられた状態となり、デバイスから割り込みが発生すると
 *              pEvent がシグナルされます。
 *
 * @details     デバイスからの割り込みを検知するためのイベントオブジェクトを紐付けます。
 *              紐付できるイベントオブジェクトは 1 つまでです。
 */
nn::Result BindInterrupt(nn::os::SystemEventType* pEvent, SignalPin pin) NN_NOEXCEPT;


/**
 * @brief       温度検知による割り込みとイベントの紐付けを解除します。
 *
 * @param[in]   pin         割り込みの受け取りを解除したいピン
 *
 * @return      処理の結果を返します。
 * @retval      ResultSuccess
 *                  成功しました。
 * @retval      ResultInvalidState
 *                  不正なステートであるため、処理を実行出来ませんでした。
 *                  設定したいモードに遷移するための前提状態を満たしていない可能性があります。
 *
 * @pre         Standby 状態である必要があります。
 *
 * @post        デバイスからの割り込みを検知できない状態になります。
 *              紐付けられた SystemEvent が紐付解除状態となります。 @n
 *              また、紐付いていた SystemEvent は破棄され、未初期化状態となります。
 *
 * @details     デバイスからの割り込みとイベントの紐付けを解除します。
 */
nn::Result UnbindInterrupt(SignalPin pin) NN_NOEXCEPT;

/**
 * @brief       温度検知による割り込み許可状態を設定します。
 *
 * @param[in]   enable      指定するピンからの割り込み許可状態を設定します。
 *                          false：禁止
 *                          true：許可
 * @param[in]   pin         割り込みを有効化するピン
 *
 * @return      処理の結果を返します。
 * @retval      ResultSuccess
 *                  成功しました。
 * @retval      ResultInvalidState
 *                  不正なステートであるため、処理を実行出来ませんでした。
 *                  設定したいモードに遷移するための前提状態を満たしていない可能性があります。
 *
 * @post        本関数が完了した次の計測から、変更が反映されます。
 *
 * @pre         Standby 状態である必要があります。
 *
 * @details     デバイスからの割り込み許可状態を設定します。
 *              割り込み検出ステータスはクリアされます。
 */
nn::Result SetInterruptEnable(bool enable, SignalPin pin) NN_NOEXCEPT;

/**
 * @brief       THERM 機能のヒステリシス設定値を取得します。
 *
 * @param[out]  pOutHysterisis  ヒステリシス設定値を受け取るバッファ
 *
 * @return      処理の結果を返します。
 * @retval      ResultSuccess
 *                  成功しました。
 *
 * @pre         初期化済み状態である必要があります。
 *
 * @details     THERM 機能のヒステリシス設定値を取得します。@n
 *              取得した値は Pin1 と Pin2 に共通した設定値です。
 */
nn::Result GetThermHysterisis(uint8_t* pOutHysterisis) NN_NOEXCEPT;

/**
 * @brief       THERM 機能のヒステリシス設定値を設定します。
 *
 * @param[in]   hysterisis  設定するヒステリシス値
 *
 * @return      処理の結果を返します。
 * @retval      ResultSuccess
 *                  成功しました。
 * @retval      ResultInvalidState
 *                  不正なステートであるため、処理を実行出来ませんでした。
 *                  設定したいモードに遷移するための前提状態を満たしていない可能性があります。
 *
 * @pre         Standby 状態である必要があります。
 *
 * @post        本関数が完了した次の計測から、変更が反映されます。
 *
 * @details     THERM 機能のヒステリシス値を設定します。@n
 *              設定した値は Pin1 と Pin2 に共通した設定値になります。
 */
nn::Result SetThermHysterisis(uint8_t hysterisis) NN_NOEXCEPT;

/**
 * @brief       ALERT 機能のコンセクティブ設定値を取得します。
 *
 * @param[out]  pOutConsecutive コンセクティブ設定値を受け取るバッファ
 *
 * @pre         初期化済み状態である必要があります。
 *
 * @details     ALERT 機能のコンセクティブ設定値を取得します。@n
 */
nn::Result GetConsecutiveAlert(AlertConsecutive* pOutConsecutive) NN_NOEXCEPT;

/**
 * @brief       ALERT 機能のコンセクティブ設定値を設定します。
 *
 * @param[in]   consecutive 設定するコンセクティブ値
 *
 * @return      処理の結果を返します。
 * @retval      ResultSuccess
 *                  成功しました。
 * @retval      ResultInvalidState
 *                  不正なステートであるため、処理を実行出来ませんでした。
 *                  設定したいモードに遷移するための前提状態を満たしていない可能性があります。
 *
 * @pre         Standby 状態である必要があります。
 *
 * @post        本関数が完了した次の計測から、変更が反映されます。
 *
 * @details     ALERT 機能のコンセクティブ値を設定します。@n
 */
nn::Result SetConsecutiveAlert(AlertConsecutive consecutive) NN_NOEXCEPT;

//! @}



} // tmp451
} // nnd
