﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   UART ドライバライブラリのポートアクセス機能に関する型や定数の定義（C リンケージ向けインタフェース）。
 */

#pragma once

#include <nnc/nn_Common.h>
#include <nnc/os/os_SystemEvent.h>

//--------------------------------------------------------------------------
//  C/C++ 共通の宣言・定義
//--------------------------------------------------------------------------

#ifdef __cplusplus
extern "C"
{
#endif // ifdef __cplusplus

/**
 * @brief   ボーレートを表す型です。
 *
 * @see
 *  nn::uart::BaudRate()
 */
typedef enum nnuartBaudRate
{
    /**
     * @brief   57600 baud
     */
    nnuartBaudRate_57600 = 57600,

    /**
     * @brief   115200 baud
     */
    nnuartBaudRate_115200 = 115200,

    /**
     * @brief   1M baud
     */
    nnuartBaudRate_1M = 1000000,

    /**
     * @brief   3M baud
     */
    nnuartBaudRate_3M = 3000000
} nnuartBaudRate;

/**
 * @brief   フロー制御モードを表す型です。
 *
 * @see
 *  nn::uart::FlowControlMode()
 */
typedef enum nnuartFlowControlMode
{
    /**
     * @brief   フロー制御を行いません。
     */
    nnuartFlowControlMode_None,

    /**
     * @brief   CTS/RTS 信号ベースのハードウェアフロー制御を行います。
     */
    nnuartFlowControlMode_Hardware
} nnuartFlowControlMode;

/**
 * @brief   ポート設定オブジェクトです。
 *
 * @see
 *  nn::uart::PortConfigType()
 */
typedef struct nnuartPortConfigType
{
    nnuartBaudRate          _baudRate;
    nnuartFlowControlMode   _flowControlMode;
    char*                   _sendBuffer;
    size_t                  _sendBufferLength;
    char*                   _receiveBuffer;
    size_t                  _receiveBufferLength;
    bool                    _isInvertTx;
    bool                    _isInvertRx;
    bool                    _isInvertRts;
    bool                    _isInvertCts;
} nnuartPortConfigType;

/**
 * @brief   ポートのセッションパラメータです。メンバに直接アクセスしないでください。
 */
typedef struct nnuartPortSession
{
    void* _handle;
    nnosSystemEventType* _sendBufferEmptyEvent;
    nnosSystemEventType* _sendBufferReadyEvent;
    nnosSystemEventType* _receiveBufferReadyEvent;
    nnosSystemEventType* _receiveEndEvent;
} nnuartPortSession;


/**
 * @brief   ポートの状態変化の種類を表す型です。
 *
 * @platformbegin{NX}
 * - @ref nnuartPortEventType_ReceiveEnd は、連続したデータの受信後、データを 4 キャラクタ分の期間受信しなかった場合に発生します。
 *
 * @platformend
 */
typedef enum nnuartPortEventType
{
    /**
     * @brief   送信予約済のすべてのデータの送信が完了しました。
     */
    nnuartPortEventType_SendBufferEmpty,

    /**
     * @brief   送信バッファに閾値以上の空きがあります。
     */
    nnuartPortEventType_SendBufferReady,

    /**
     * @brief   受信バッファに閾値以上の受信済データがあります。
     */
    nnuartPortEventType_ReceiveBufferReady,

    /**
     * @brief   連続したデータの受信が終了しました。@n
     *          この状態変化は、1 バイト以上の連続したデータを受信した後、一定時間データを受信しなかった場合に受信の区切りとして発生します。
     */
    nnuartPortEventType_ReceiveEnd
} nnuartPortEventType;


#ifdef __cplusplus
}
#endif // ifdef __cplusplus
