﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   UART ドライバライブラリのポートアクセス機能の API 宣言（C リンケージ向けインタフェース）。
 */

#pragma once

#include <nnc/nn_Common.h>
#include <nnc/nn_Result.h>
#include <nnc/os/os_SystemEvent.h>

#include "uart_PortTypes.h"

//--------------------------------------------------------------------------
//  C/C++ 共通の宣言・定義
//--------------------------------------------------------------------------

#ifdef __cplusplus
extern "C"
{
#endif // ifdef __cplusplus

//! @name ポート情報の取得
//! @{

/**
 * @brief   指定したポート識別子が、ターゲット環境上で使用可能かを取得します。
 *
 * @param[in]   name        確認対象のポート識別子
 *
 * @return  ポートが使用可能かを返します。
 *
 * @details
 *  この関数は、@ref nn::uart::HasPort() の C リンケージ向けインタフェースです。
 *  詳細は C++ 向け版を参照してください。
 *
 * @see
 *  nn::uart::HasPort()
 */
bool nnuartHasPort(nnuartPortName name);

/**
 * @brief   指定したポートが、指定したボーレートをサポートしているかを取得します。
 *
 * @param[in]   name        ポート識別子
 * @param[in]   baudRate    ボーレート
 *
 * @return  サポート有無を返します。
 *
 * @details
 *  この関数は、@ref nn::uart::IsSupportedBaudRate() の C リンケージ向けインタフェースです。
 *  詳細は C++ 向け版を参照してください。
 *
 * @see
 *  nn::uart::IsSupportedBaudRate()
 */
bool nnuartIsSupportedBaudRate(nnuartPortName name, nnuartBaudRate baudRate);

/**
 * @brief   指定したポートが、指定したフロー制御モードをサポートしているかを取得します。
 *
 * @param[in]   name            ポート識別子
 * @param[in]   flowControlMode フロー制御モード
 *
 * @return  サポート有無を返します。
 *
 * @details
 *  この関数は、@ref nn::uart::IsSupportedFlowControlMode() の C リンケージ向けインタフェースです。
 *  詳細は C++ 向け版を参照してください。
 *
 * @see
 *  nn::uart::IsSupportedFlowControlMode()
 */
bool nnuartIsSupportedFlowControlMode(nnuartPortName name, nnuartFlowControlMode flowControlMode);

/**
 * @brief   指定したポートが、指定したポートイベントをサポートしているかを取得します。
 *
 * @param[in]   name            ポート識別子
 * @param[in]   portEvent       ポートイベント
 *
 * @return  サポート有無を返します。
 *
 * @details
 *  この関数は、@ref nn::uart::IsSupportedPortEvent() の C リンケージ向けインターフェースです。
 *  詳細は C++ 向け版を参照してください。
 *
 * @see
 *  nn::uart::IsSupportedPortEvent()
 */
bool nnuartIsSupportedPortEvent(nnuartPortName name, nnuartPortEventType portEvent);

//! @}

//! @name ポートの動作設定
//! @{

/**
 * @brief   ポート設定オブジェクトを初期化します。
 *
 * @param[out]  pOutPortConfig      ポート設定オブジェクト
 * @param[in]   baudRate            ボーレート値
 * @param[in]   sendBuffer          送信バッファへのポインタ
 * @param[in]   sendBufferLength    送信バッファサイズ [byte]
 * @param[in]   receiveBuffer       受信バッファへのポインタ
 * @param[in]   receiveBufferLength 受信バッファサイズ [byte]
 *
 * @details
 *  この関数は、@ref nn::uart::InitializePortConfig() の C リンケージ向けインタフェースです。
 *  詳細は C++ 向け版を参照してください。
 *
 * @see
 *  nn::uart::InitializePortConfig()
 */
void nnuartInitializePortConfig(nnuartPortConfigType* pOutPortConfig, nnuartBaudRate baudRate, char* sendBuffer, size_t sendBufferLength, char* receiveBuffer, size_t receiveBufferLength);

/**
 * @brief   ポート設定オブジェクトに対しフロー制御モードを設定します。
 *
 * @param[in]   pPortConfig         ポート設定オブジェクト
 * @param[in]   flowControlMode     フロー制御モード
 *
 * @details
 *  この関数は、@ref nn::uart::SetPortConfigFlowControlMode() の C リンケージ向けインタフェースです。
 *  詳細は C++ 向け版を参照してください。
 *
 * @see
 *  nn::uart::SetPortConfigFlowControlMode()
 */
void nnuartSetPortConfigFlowControlMode(nnuartPortConfigType* pPortConfig, nnuartFlowControlMode flowControlMode);

/**
* @brief   ポート設定オブジェクトに対し各ピンの Invert 状態を設定します。
*
* @param[in]   pPortConfig         ポート設定オブジェクト
* @param[in]   isInvertTx          TX の Invert 設定
* @param[in]   isInvertRx          RX の Invert 設定
* @param[in]   isInvertRts         RTS の Invert 設定
* @param[in]   isInvertCts         CTS の Invert 設定
*
* @details
*  ポート設定オブジェクトに対し各ピンの Invert 状態を設定します。
*/
void nnuartSetPortConfigInvertPins(nnuartPortConfigType* pPortConfig, bool isInvertTx, bool isInvertRx, bool isInvertRts, bool isInvertCts);

//! @}

//! @name ポートアクセス機能
//! @{

/**
 * @brief   指定したポートをオープンします。
 *
 * @param[out]  pOutSession オープンされたポートセッション
 * @param[in]   name        ポート識別子
 * @param[in]   pPortConfig ポート設定
 *
 * @return      処理の結果を返します。
 *
 * @details
 *  この関数は、@ref nn::uart::OpenPort() の C リンケージ向けインタフェースです。
 *  詳細は C++ 向け版を参照してください。
 *
 * @see
 *  nn::uart::OpenPort()
 */
bool nnuartOpenPort(nnuartPortSession* pOutSession, nnuartPortName name, const nnuartPortConfigType* pPortConfig);

/**
 * @brief   送信バッファに追記可能なデータのサイズを返します。
 *
 * @param[in]   pSession    ポートセッション
 *
 * @return      送信バッファに追記可能なデータのサイズ
 *
 * @details
 *  この関数は、@ref nn::uart::GetWritableLength() の C リンケージ向けインタフェースです。
 *  詳細は C++ 向け版を参照してください。
 *
 * @see
 *  nn::uart::GetWritableLength()
 */
size_t nnuartGetWritableLength(const nnuartPortSession* pSession);

/**
 * @brief   指定したデータ列を送信バッファに書き込みます。
 *
 * @param[out]  pOutDoneBytes   バッファに格納された送信予約済データ列の長さ [bytes]
 * @param[in]   pSession        ポートセッション
 * @param[in]   data            送信するデータ列
 * @param[in]   dataBytes       送信するデータ列の長さ [bytes]
 *
 * @details
 *  この関数は、@ref nn::uart::Send() の C リンケージ向けインタフェースです。
 *  詳細は C++ 向け版を参照してください。
 *
 * @see
 *  nn::uart::Send()
 */
void nnuartSend(size_t* pOutDoneBytes, nnuartPortSession* pSession, const void* data, size_t dataBytes);

/**
 * @brief   内部受信バッファに溜まっている受信済データ列のサイズを返します。
 *
 * @param[in]   pSession    ポートセッション
 *
 * @return      内部受信バッファに溜まっている受信済データ列のサイズ
 *
 * @details
 *  この関数は、@ref nn::uart::GetReadableLength() の C リンケージ向けインタフェースです。
 *  詳細は C++ 向け版を参照してください。
 *
 * @see
 *  nn::uart::GetReadableLength()
 */
size_t nnuartGetReadableLength(const nnuartPortSession* pSession);

/**
 * @brief   受信済データを読み出します。
 *
 * @param[out]  pOutDoneBytes バッファに格納された受信済データ列の長さ [bytes]
 * @param[out]  outData       受信済データを格納するバッファ。
 *                            dataBytes で指定した以上のサイズで確保されている必要があります。
 * @param[in]   dataBytes     読み出す受信済データ列の最大長 [bytes]
 * @param[in]   pSession      ポートセッション
 *
 * @return      処理の結果を返します。
 *
 * @details
 *  この関数は、@ref nn::uart::Receive() の C リンケージ向けインタフェースです。
 *  詳細は C++ 向け版を参照してください。
 *
 * @see
 *  nn::uart::Receive()
 */
nnResult nnuartReceive(size_t* pOutDoneBytes, void* outData, size_t dataBytes, nnuartPortSession* pSession);

/**
 * @brief   特定の状態変化に対してシステムイベントを関連づけます。
 *
 * @param[in]   pEvent      状態変化が発生した際にシグナル化するシステムイベント
 * @param[in]   pSession    ポートセッション
 * @param[in]   eventType   関連づける状態変化の種類
 * @param[in]   threshold   シグナル化を起こす閾値
 *
 * @details
 *  この関数は、@ref nn::uart::BindPortEvent() の C リンケージ向けインタフェースです。
 *  詳細は C++ 向け版を参照してください。
 *
 * @see
 *  nn::uart::BindPortEvent()
 */
bool nnuartBindPortEvent(nnosSystemEventType* pEvent, nnuartPortSession* pSession, nnuartPortEventType eventType, size_t threshold);

/**
 * @brief   指定したポートの状態変化とシステムイベントとの関連づけを解除します。
 *
 * @param[in]   pEvent      システムイベント
 * @param[in]   pSession    ポートセッション
 *
 * @details
 *  この関数は、@ref nn::uart::UnbindPortEvent() の C リンケージ向けインタフェースです。
 *  詳細は C++ 向け版を参照してください。
 *
 * @see
 *  nn::uart::UnbindPortEvent()
 */
bool nnuartUnbindPortEvent(nnosSystemEventType* pEvent, nnuartPortSession* pSession);

/**
 * @brief   指定したポートをクローズします。
 *
 * @param[in]   pSession    ポートセッション
 *
 * @details
 *  この関数は、@ref nn::uart::ClosePort() の C リンケージ向けインタフェースです。
 *  詳細は C++ 向け版を参照してください。
 *
 * @see
 *  nn::uart::ClosePort()
 */
void nnuartClosePort(nnuartPortSession* pSession);

//! @}

#ifdef __cplusplus
}
#endif // ifdef __cplusplus
