﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   xcd ライブラリの振動に関係する API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/xcd/xcd_Device.h>

namespace nn { namespace xcd {

/**
 * @brief   振動子が左手用か右手用かを表す列挙型です。
 */
enum  VibratorPosition
{
    VibratorPosition_Left,  //!< 左手用振動子
    VibratorPosition_Right, //!< 右手用振動子
};

/**
 * @brief   振動値を表す構造体です。
 */
struct VibrationValue
{
    float amplitudeLow;     //!< 低周波帯 (160 Hz 帯) の振幅値
    float frequencyLow;     //!< 低周波帯 (160 Hz 帯) の周波数値
    float amplitudeHigh;    //!< 高周波帯 (320 Hz 帯) の振幅値
    float frequencyHigh;    //!< 高周波帯 (320 Hz 帯) の周波数値
};

/**
 * @brief   コントローラ内部の振動用のバッファの状態を表す構造体です。
 */
struct VibratorBufferStatus
{
    int remainingDataLevel; //!< バッファ残量レベル (0 以上 7 以下の値)
    bool isBufferUnderrun;  //!< バッファアンダーランが発生した場合に true
};

/**
 * @brief   振動子のキャリブレーション値を表す構造体です。
 */
struct VibratorCalibrationValue
{
    uint32_t _dummy;             //!< 詳細未定
};

/**
 * @brief       振動子を動作させるためのアンプの電源を制御します。
 *
 * @return      実行結果を返します。
 * @retval      ResultSuccess               処理に成功しました。
 * @retval      ResultNotConnected          デバイスが接続されていません。
 * @retval      ResultVibratorNotExist      指定された振動子が存在しません。
 *
 * @param[in]   isEnabled       電源を ON にする場合は true を、OFF にする場合には false を指定します。
 * @param[in]   position        振動子の種類
 * @param[in]   handle          デバイスへのハンドル
 */
Result SetVibratorEnabled(bool isEnabled, VibratorPosition position, DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       振動子を動作させるためのアンプの電源状態を取得します。
 *
 * @return      実行結果を返します。
 * @retval      ResultSuccess               処理に成功しました。
 * @retval      ResultNotConnected          デバイスが接続されていません。
 * @retval      ResultVibratorNotExist      指定された振動子が存在しません。
 *
 * @param[out]  pOutValue       電源状態の格納先。電源が ON ならば true が格納されます。
 * @param[in]   position        振動子の種類
 * @param[in]   handle          デバイスへのハンドル
 */
Result IsVibratorEnabled(bool* pOutValue, VibratorPosition position, DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       コントローラ内部の振動用のバッファの状態を取得します。
 *
 * @return      実行結果を返します。
 * @retval      ResultSuccess               処理に成功しました。
 * @retval      ResultNotConnected          デバイスが接続されていません。
 * @retval      ResultVibratorNotExist      指定された振動子が存在しません。
 * @retval      ResultVibratorNotEnabled    指定された振動子を動作させるためのアンプの電源が ON になっていません。
 *
 * @param[out]  pOutValue       振動用のバッファの状態の格納先
 * @param[in]   position        振動子の種類
 * @param[in]   handle          デバイスへのハンドル
 */
Result GetVibratorBufferStatus(VibratorBufferStatus* pOutValue, VibratorPosition position, DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       次の送信タイミングまでに上位ライブラリが渡すべき振動値の個数を取得します。
 *
 * @return      実行結果を返します。
 * @retval      ResultSuccess               処理に成功しました。
 * @retval      ResultNotConnected          デバイスが接続されていません。
 * @retval      ResultVibratorNotExist      指定された振動子が存在しません。
 * @retval      ResultVibratorNotEnabled    指定された振動子を動作させるためのアンプの電源が ON になっていません。
 *
 * @param[out]  pOutValue       振動用のバッファの状態の格納先
 * @param[in]   position        振動子の種類
 * @param[in]   handle          デバイスへのハンドル
 */
Result GetRequiredVibrationValueCount(int* pOutValue, VibratorPosition position, DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       次の送信タイミングでコントローラに送信する振動値を指定します。
 *
 * @return      実行結果を返します。
 * @retval      ResultSuccess                   処理に成功しました。
 * @retval      ResultNotConnected              デバイスが接続されていません。
 * @retval      ResultVibratorNotExist          指定された振動子が存在しません。
 * @retval      ResultVibratorNotEnabled        指定された振動子を動作させるためのアンプの電源が ON になっていません。
 * @retval      ResultVibrationDataQueueFull    内部の振動データキューがいっぱいです。
 *
 * @param[in]   value           振動値
 * @param[in]   position        振動子の種類
 * @param[in]   handle          デバイスへのハンドル
 *
 * @details
 *  指定された振動値は内部のキューにいったん保存されたのち、
 *  次の送信タイミングでまとめてコントローラに送信されます。
 */
Result SendVibrationValue(const VibrationValue& value, VibratorPosition position, DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       コントローラで実際に発生する振動値を取得します。
 *
 * @return      実行結果を返します。
 * @retval      ResultSuccess               処理に成功しました。
 * @retval      ResultNotConnected          デバイスが接続されていません。
 * @retval      ResultVibratorNotExist      指定された振動子が存在しません。
 * @retval      ResultVibratorNotEnabled    指定された振動子を動作させるためのアンプの電源が ON になっていません。
 *
 * @param[out]  pOutValue       振動値の格納先
 * @param[in]   position        振動子の種類
 * @param[in]   handle          デバイスへのハンドル
 *
 * @details
 *  直前の送信タイミングで振動子に送られた AMFM 符号がコントローラ側でデコードされた際に
 *  発生すると想定される振動を振動値の形で取得します。
 */
Result GetActualVibrationValue(VibrationValue* pOutValue, VibratorPosition position, DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       接続時振動を開始します。
 *
 * @return      実行結果を返します。
 * @retval      ResultSuccess               処理に成功しました。
 * @retval      ResultNotConnected          デバイスが接続されていません。
 * @retval      ResultVibratorNotExist      指定された振動子が存在しません。
 *
 * @param[in]   handle          デバイスへのハンドル
 *
 * @details
 *  接続時振動の振動パターン再生を開始します。
 *  振動パターン再生中は、SendVibrationValue 関数で送信された振動値は無視されます。
 *  振動子用アンプの電源が ON になっていない場合には、この関数を呼ぶことで一時的に振動パターン再生中だけ
 *  振動子用アンプの電源が ON になります。
 */
Result StartVibrationOnConnect(DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       振動子のキャリブレーション値を設定します。
 *
 * @return      実行結果を返します。
 * @retval      ResultSuccess               処理に成功しました。
 * @retval      ResultNotConnected          デバイスが接続されていません。
 * @retval      ResultVibratorNotExist      指定された振動子が存在しません。
 *
 * @param[in]   value           振動子のキャリブレーション値
 * @param[in]   position        振動子の種類
 * @param[in]   handle          デバイスへのハンドル
 */
Result SetVibratorCalibrationValue(const VibratorCalibrationValue& value, VibratorPosition position, DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       コントローラに設定されているキャリブレーション値を取得します。
 *
 * @return      実行結果を返します。
 * @retval      ResultSuccess               処理に成功しました。
 * @retval      ResultNotConnected          デバイスが接続されていません。
 * @retval      ResultVibratorNotExist      指定された振動子が存在しません。
 * @retval      ResultVibratorNotEnabled    指定された振動子を動作させるためのアンプの電源が ON になっていません。
 *
 * @param[out]  pOutValue       振動子のキャリブレーション値の格納先
 * @param[in]   position        振動子の種類
 * @param[in]   handle          デバイスへのハンドル
 */
Result GetVibratorCalibrationValue(VibratorCalibrationValue* pOutValue, VibratorPosition position, DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       振動子に送る信号の振幅に制限をかけるか設定します。
 *
 * @return      実行結果を返します。
 * @retval      ResultSuccess               処理に成功しました。
 * @retval      ResultNotConnected          デバイスが接続されていません。
 * @retval      ResultVibratorNotExist      指定された振動子が存在しません。
 *
 * @param[in]   isEnabled       振幅制限をかける場合は true を、かけない場合には false を指定します。
 * @param[in]   position        振動子の種類
 * @param[in]   handle          デバイスへのハンドル
 */
Result SetVibratorAmplitudeLimitEnabled(bool isEnabled, VibratorPosition position, DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       振動子に送る信号の振幅に制限がかけられているか取得します。
 *
 * @return      実行結果を返します。
 * @retval      ResultSuccess               処理に成功しました。
 * @retval      ResultNotConnected          デバイスが接続されていません。
 * @retval      ResultVibratorNotExist      指定された振動子が存在しません。
 *
 * @param[out]  pOutValue       結果の格納先。振幅に制限がかけられているならば true が格納されます。
 * @param[in]   position        振動子の種類
 * @param[in]   handle          デバイスへのハンドル
 */
Result IsVibratorAmplitudeLimitEnabled(bool* pOutValue, VibratorPosition position, DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       振動子に送る信号の振幅に周波数に応じた調整係数をかける機能（イコライザー機能）を有効にするか設定します。
 *
 * @return      実行結果を返します。
 * @retval      ResultSuccess               処理に成功しました。
 * @retval      ResultNotConnected          デバイスが接続されていません。
 * @retval      ResultVibratorNotExist      指定された振動子が存在しません。
 *
 * @param[in]   isEnabled       イコライザー機能を有効にする場合は true を、しない場合には false を指定します。
 * @param[in]   position        振動子の種類
 * @param[in]   handle          デバイスへのハンドル
 */
Result SetVibratorEqualizerEnabled(bool isEnabled, VibratorPosition position, DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       振動子に送る信号の振幅に周波数に応じた調整係数をかける機能（イコライザー機能）が有効か取得します。
 *
 * @return      実行結果を返します。
 * @retval      ResultSuccess               処理に成功しました。
 * @retval      ResultNotConnected          デバイスが接続されていません。
 * @retval      ResultVibratorNotExist      指定された振動子が存在しません。
 *
 * @param[out]  pOutValue       結果の格納先。イコライザー機能が有効ならば true が格納されます。
 * @param[in]   position        振動子の種類
 * @param[in]   handle          デバイスへのハンドル
 */
Result IsVibratorEqualizerEnabled(bool* pOutValue, VibratorPosition position, DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       振動のマスターボリュームを設定します。本体 NAND への値の保存は行いません。
 *
 * @return      実行結果を返します。
 *
 * @param[in]   masterVolume    振動のマスターボリューム
 */
Result SetVibrationMasterVolume(float masterVolume) NN_NOEXCEPT;

/**
 * @brief       振動のマスターボリュームを設定した上で本体 NAND に値を保存します。
 *
 * @return      実行結果を返します。
 *
 * @param[in]   masterVolume    振動のマスターボリューム
 */
Result SetAndSaveVibrationMasterVolume(float masterVolume) NN_NOEXCEPT;

/**
 * @brief       本体 NAND からロード済みの振動のマスターボリュームを取得します。
 *
 * @return      実行結果を返します。
 *
 * @param[out]  pOutValue       振動のマスターボリュームの格納先
 */
Result GetVibrationMasterVolume(float* pOutValue) NN_NOEXCEPT;

/**
 * @brief       本体 NAND に保存されている値をロードした上で振動のマスターボリュームを取得します。
 *
 * @return      実行結果を返します。
 *
 * @param[out]  pOutValue       振動のマスターボリュームの格納先
 */
Result LoadAndGetVibrationMasterVolume(float* pOutValue) NN_NOEXCEPT;

}} // namespace nn::xcd
