﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       xcd ライブラリに関係する API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_TimeSpan.h>
#include <nn/xcd/xcd_Device.h>

namespace nn { namespace xcd {

//!< Usb Hid Report の最大長
const size_t UsbHidReportLengthMax = 64;

/**
 * @brief      USB Hid デバイスの情報を表す構造体です
 */
struct UsbHidDeviceInfo
{
    uint32_t vid;    //!< vid
    uint32_t pid;    //!< pid
    ::nn::TimeSpan interval; //!< interval
};

 /**
 * @brief      USB デバイスが Xcd でサポートしているデバイスかどうか評価します。
 *
 * @return     Xcd がサポートしているデバイスの場合に true を返します。
 */
bool IsUsbHidSupported(UsbHidDeviceInfo deviceInfo) NN_NOEXCEPT;

 /**
 * @brief      USB デバイスを追加します
 *
 * @details    USB デバイスを追加します。
 *             index は SetInputReport(), GetOutputReport() を行う時の識別子として使用されるため、デバイスが異なる場合重複してはいけません
 *
 * @param[in]  index                       デバイスの識別子
 * @param[in]  deviceInfo                  デバイスの情報
 *
 * @return     実行結果を返します。
 * @retval     ResultSuccess                処理に成功しました。
 * @retval     ResultNotConnected           デバイスが接続されていません。
 *
 * @pre        - xcd ライブラリは初期化済である必要があります。
 */
Result AddUsbHidDevice(int index, UsbHidDeviceInfo deviceInfo) NN_NOEXCEPT;

 /**
 * @brief      USB デバイスを削除します
 *
 * @details    デバイスが切断された際に、xcd に追加したデバイスを削除します
 *
 * @param[in]  index                       デバイスの識別子
 *
 * @return     実行結果を返します。
 * @retval     ResultSuccess                処理に成功しました。
 * @retval     ResultNotConnected           デバイスが接続されていません。
 *
 * @pre        - xcd ライブラリは初期化済である必要があります。
 */
Result RemoveUsbHidDevice(int index) NN_NOEXCEPT;

 /**
 * @brief      受信した InputReport を処理します
 *
 * @param[in]  index                       デバイスの識別子
 * @param[in]  pBuffer                     InputReport のバッファ
 * @param[in]  length                      InputReport のバッファ長
  *
 * @return     実行結果を返します。
 * @retval     ResultSuccess             処理に成功しました。
 * @retval     ResultNotConnected            デバイスが接続されていません。
 *
 * @pre        - xcd ライブラリは初期化済である必要があります。
  */
Result SetUsbHidInputReport(int index, uint8_t* pBuffer, size_t length) NN_NOEXCEPT;

 /**
 * @brief      送信する OutputReport を取得
 *
 * @details    送信する OutputReport がない場合は、返り値として 0 が返ります
 *
 * @param[in]  index                       デバイスの識別子
 * @param[out] pBuffer                     OutputReport のバッファ
 * @param[in]  length                      OutputReport のバッファ長
  *
 * @return     OutputReport の長さを返します。送信する OutputReport がない場合は 0 となります。
 *
 * @pre        - xcd ライブラリは初期化済である必要があります。
  */
size_t GetUsbHidOutputReport(int index, uint8_t* pOutBuffer, size_t length) NN_NOEXCEPT;

 /**
 * @brief      Pro Controller 有線 USB 通信機能を有効にします
 *
 * @param[in]  enabled                     Pro Controller 有線 USB 通信機能を有効にする場合は true を、しない場合には false を指定します。
 *
 */
void SetFullKeyUsbEnabled(bool enabled) NN_NOEXCEPT;

 /**
 * @brief      Pro Controller 有線 USB 通信機能の状態を取得します
 *
 * @param[out] pOutEnabled                 Pro Controller 有線 USB 通信機能の状態の格納先。有効な場合は true が格納されます。
 *
 */
void GetFullKeyUsbEnabled(bool* pOutEnabled) NN_NOEXCEPT;

/**
 * @brief       デバイスの USB 接続状態を取得します。
 *
 * @param[out]  pOutValue       USB 接続状態の格納先。USB 接続されている場合は true が格納されます。
 * @param[in]   handle          デバイスへのハンドル
 */
void GetUsbConnected(bool* pOutValue, DeviceHandle handle) NN_NOEXCEPT;

}} // namespace nn::xcd
