﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       デバイスの列挙などに関する API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/os/os_SystemEventTypes.h>
#include <nn/util/util_BitFlagSet.h>
#include <nn/xcd/xcd_Device.h>
#include <nn/xcd/xcd_Result.h>

namespace nn { namespace xcd {

/**
 * @brief       シリアルフラッシュからの読み込みを行います。
 *
 * @details     処理は非同期に行われます。
 *              処理が完了すると渡されたシステムイベントに対してシグナルされます。
 *              処理結果を取得するには、 GetSerialFlashResult() を使用してください。
 *
 *              読み込みまたは書き込み中に、同一のデバイスに対して操作を行うことはできません。
 *
 * @param[in]  address                       操作を行う対象のアドレス
 * @param[in]  pOutBuffer                    データを読み込む先のバッファ
 * @param[in]  size                          操作を行うサイズ
 * @param[in]  pEvent                        操作が完了したときに通知するイベント
 * @param[in]  handle                        情報を取得するデバイスへのハンドル
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                 処理に成功しました。}
 *   @handleresult{nn::xcd::ResultNotConnected,       指定のハンドルにデバイスが接続されていません。}
 *   @handleresult{nn::xcd::ResultSerialFlashOnWrite, シリアルフラッシュへの書き込み中です。}
 *   @handleresult{nn::xcd::ResultSerialFlashOnRead,  シリアルフラッシュからの読み込み中です。}
 * @endretresult
 *
 * @pre         - xcd ライブラリは初期化済である必要があります。
 *              - pOutBuffer != nullptr
 */
Result ReadSerialFlash(
    const uint32_t address,
    uint8_t* pOutBuffer,
    int size,
    nn::os::SystemEventType* pEvent,
    const DeviceHandle& handle) NN_NOEXCEPT;

/**
 * @brief       シリアルフラッシュへの書き込みを行います。
 *
 * @details     処理は非同期に行われます。
 *              処理が完了すると渡されたシステムイベントに対してシグナルされます。
 *              処理結果を取得するには、 GetSerialFlashResult() を使用してください。
 *
 *              読み込みまたは書き込み中に、同一のデバイスに対して操作を行うことはできません。
 *
 * @param[in]  address                       操作を行う対象のアドレス
 * @param[in]  pBuffer                       書き込みを行うデータ
 * @param[in]  size                          操作を行うサイズ
 * @param[in]  pEvent                        操作が完了したときに通知するイベント
 * @param[in]  handle                        情報を取得するデバイスへのハンドル
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                 処理に成功しました。}
 *   @handleresult{nn::xcd::ResultNotConnected,       指定のハンドルにデバイスが接続されていません。}
 *   @handleresult{nn::xcd::ResultSerialFlashOnWrite, シリアルフラッシュへの書き込み中です。}
 *   @handleresult{nn::xcd::ResultSerialFlashOnRead,  シリアルフラッシュからの読み込み中です。}
 * @endretresult
 *
 * @pre         - xcd ライブラリは初期化済である必要があります。
 *              - pBuffer != nullptr
 */
Result WriteSerialFlash(
    const uint32_t address,
    const uint8_t* pBuffer,
    int size,
    nn::os::SystemEventType* pEvent,
    DeviceHandle& handle) NN_NOEXCEPT;

/**
 * @brief       シリアルフラッシュに対して操作を行った結果を取得します。
 *
 * @param[in]  handle                        情報を取得するデバイスへのハンドル
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                     処理に成功しました。}
 *   @handleresult{nn::xcd::ResultNotConnected,           指定のハンドルにデバイスが接続されていません。}
 *   @handleresult{nn::xcd::ResultSerialFlashNoResult,    処理中もしくは処理開始前です。}
 *   @handleresult{nn::xcd::ResultSerialFlashVerifyError, ベリファイに失敗しました。}
 * @endretresult
 *
 * @pre         - xcd ライブラリは初期化済である必要があります。
 */
Result GetSerialFlashResult(const DeviceHandle& handle) NN_NOEXCEPT;

}} // namespace nn::xcd
