﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       xcd ライブラリに関係する API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/os/os_SystemEventTypes.h>
#include <nn/os/os_MultipleWait.h>
#include "xcd_Device.h"
#include "xcd_NfcTypes.h"

namespace nn { namespace xcd {

/**
 * @brief       Tera MCU ファームウェアのバージョンを取得します。
 *
 * @details     Tera MCU ファームウェアのバージョンを取得します。
 *              バージョンを取得するには、Tera MCU を起動させておく必要があります。
 *              この関数を呼ぶ前に、 @ref McuState_Nfc または @ref McuState_Ir を指定して @ref SetMcuState を呼び出しておいてください。
 *              一度バージョン取得が完了した後は、デバイスが切断するまで Tera MCU が起動していない状態でもバージョンを取得できます。
 *
 * @param[out]  pOutMcuVersionData          バージョンデータの取得先
 * @param[in]   handle                      バージョンを取得するデバイスへのハンドル
 *
 * @retresult
 *      @handleresult{nn::ResultSuccess,                        処理に成功しました。}
 *      @handleresult{nn::xcd::ResultNotConnected,              デバイスが接続されていません。}
 *      @handleresult{nn::xcd::ResultMcuVersionNotAvailable,    MCU のバージョン取得が完了していません。}
 *      @handleresult{nn::xcd::ResultMcuFirmwareCorrupted,      MCU ファームウェアが壊れているため、アップデートが必要です。}
 *      @handleresult{nn::xcd::ResultMcuHardwareError,          MCU ハードウェア異常のため、バージョンを取得できません。}
 * @endresult
 *
 * @pre
 *  - XCD ライブラリは初期化済である必要があります。
 *  - @a pOutMcuVersionData != nullptr
 */
Result GetMcuVersionForNfc(McuVersionDataForNfc* pOutMcuVersionData, DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       NFC 機能を使用できる電源状態であるか確認します。
 *
 * @details     NFC 機能を使用できる電源状態であるか確認します。
 *              電池残量不足と判定された場合、NFC を使用するためにデバイスを充電する必要があります。
 *
 * @param[in]   handle          確認するデバイスへのハンドル
 *
 * @retresult
 *      @handleresult{nn::ResultSuccess,                        NFC 機能を使用できる電源状態です。}
 *      @handleresult{nn::xcd::ResultNotConnected,              デバイスが接続されていません。}
 *      @handleresult{nn::xcd::ResultNotSupported,              NFC 非対応のデバイスです。}
 *      @handleresult{nn::xcd::ResultLowBattery,                デバイスの電池残量が不足しています。}
 * @endresult
 *
 * @pre
 *  - XCD ライブラリは初期化済である必要があります。
 */
Result CheckNfcDevicePower(DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief      NFC に関連する通知を受け取るイベントを設定します。
 *
 * @param[in]  pNfcEvent                   NFC に関連する通知をうけるイベント
 * @param[in]  pNfcDetectEvent             タグの検出・喪失の通知をうけるイベント
 * @param[in]  handle                      デバイスへのハンドル（ XCD ライブラリで初期化するものを使用する予定：要検討）
 *
 * @retresult
 *      @handleresult{nn::ResultSuccess,            処理に成功しました。}
 *      @handleresult{nn::xcd::ResultNotConnected,  デバイスが接続されていません。}
 * @endretresult
 *
 * @pre        - XCD ライブラリは初期化済である必要があります。
 *             - @a *pNfcEvent, @a *pNfcDetectEvent は未初期化状態
 *
 * @post       - @a *pNfcEvent, @a *pNfcDetectEvent は ManualClear モードで初期化状態
 *
 * @details    NFC に関連する通知を受け取るイベントを設定します。
 *             タグの検出時やタグデータの読み取り終了時などにシグナルされます。
 *             イベントは、不要になったタイミングで @ref nn::os::DestroySystemEvent を使用して破棄する必要があります。
 *             設定したイベントは、デバイスの切断、 @ref nn::xcd::McuState_Standby への遷移によって解除されます。これらの状況が発生した後は、NFC 機能の使用前にイベントの再設定が必要です。
 */
Result SetNfcEvent(nn::os::SystemEventType* pNfcEvent, nn::os::SystemEventType* pNfcDetectEvent, DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       NFC の状態を取得します。
 *
 * @param[out]  pOutValue                   現在のデバイスの状態
 * @param[in]   handle                      状態を取得するデバイスへのハンドル
 *
 * @retresult
 *      @handleresult{nn::ResultSuccess,                処理に成功しました。}
 *      @handleresult{nn::xcd::ResultNotConnected,      デバイスが接続されていません。}
 *      @handleresult{nn::xcd::ResultInvalidMcuState,   MCU が NFC を扱えるステートではありません。}
 * @endretresult
 *
 * @pre         - XCD ライブラリは初期化済である必要があります。
 *              - @a pOutValue != nullptr
 *
 * @details     NFC の状態を取得します。
 *              @ref NfcInfo で定義されたイベントの状態やタグのデータなどが取得できます。
 */
Result GetNfcInfo(NfcInfo* pOutValue, DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       NFC タグの検出を開始します。
 *
 * @param[in]   nfcDiscoveryParam           タグ検出用パラメータ
 * @param[in]   handle                      対象のデバイスへのハンドル
 *
 * @retresult
 *      @handleresult{nn::ResultSuccess,                処理に成功しました。}
 *      @handleresult{nn::xcd::ResultNotConnected,      デバイスが接続されていません。}
 *      @handleresult{nn::xcd::ResultInvalidMcuState,   MCU が NFC を扱えるステートではありません。}
 *      @handleresult{nn::xcd::ResultMcuBusy,           実行中の処理があるため、要求を受け付けられません。}
 * @endretresult
 *
 * @pre         - XCD ライブラリは初期化済である必要があります。
 *
 * @details     NFC タグの検出を開始します。
 *              タグが検出された際には、検出・喪失イベントによって通知されます。
 */
Result StartNfcDiscovery(const NfcDiscoveryParameter& nfcDiscoveryParam, DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       タグのポーリングを中止もしくは間欠動作を停止します。
 *
 * @param[in]   handle                      対象のデバイスへのハンドル
 *
 * @retresult
 *      @handleresult{nn::ResultSuccess,                処理に成功しました。}
 *      @handleresult{nn::xcd::ResultNotConnected,      デバイスが接続されていません。}
 *      @handleresult{nn::xcd::ResultInvalidMcuState,   MCU が NFC を扱えるステートではありません。}
 *      @handleresult{nn::xcd::ResultMcuBusy,           実行中の処理があるため、要求を受け付けられません。}
 * @endretresult
 *
 * @pre         - XCD ライブラリは初期化済である必要があります。
 *
 * @details     タグのポーリングを中止もしくは間欠動作を停止します。
 */
Result StopNfcDiscovery(DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       NTAG シリーズのタグの読み取りを行います。
 *
 * @param[in]   ntagReadParam               NTAG アクセス用のパラメータ
 * @param[in]   handle                      対象のデバイスへのハンドル
 *
 * @retresult
 *      @handleresult{nn::ResultSuccess,                処理に成功しました。}
 *      @handleresult{nn::xcd::ResultNotConnected,      デバイスが接続されていません。}
 *      @handleresult{nn::xcd::ResultInvalidMcuState,   MCU が NFC を扱えるステートではありません。}
 *      @handleresult{nn::xcd::ResultMcuBusy,           実行中の処理があるため、要求を受け付けられません。}
 * @endretresult
 *
 * @pre         - XCD ライブラリは初期化済である必要があります。
 *
 * @details     NXP 社のNTAG シリーズのタグの読み取りを行います。
 *              amiibo の読み取りにおいては、amiibo 用の引数を指定する必要があります。
 *              読み取りの結果は、イベント取得後の @ref GetNfcInfo() にて取得できます。
 */
Result StartNtagRead(const NtagReadParameter& ntagReadParam, DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       NTAG シリーズのタグへの書き込みを行います。
 *
 * @param[in]   ntagWriteParam              NTAG アクセス用のパラメータ
 * @param[in]   handle                      対象のデバイスへのハンドル
 *
 * @retresult
 *      @handleresult{nn::ResultSuccess,                処理に成功しました。}
 *      @handleresult{nn::xcd::ResultNotConnected,      デバイスが接続されていません。}
 *      @handleresult{nn::xcd::ResultInvalidMcuState,   MCU が NFC を扱えるステートではありません。}
 *      @handleresult{nn::xcd::ResultMcuBusy,           実行中の処理があるため、要求を受け付けられません。}
 * @endretresult
 *
 * @pre         - XCD ライブラリは初期化済である必要があります。
 *
 * @details     NXP 社の NTAG シリーズのタグへの書き込みを行います。
 *              amiibo の書き込みにおいては、amiibo 用の引数を指定する必要があります。
 *              書き込みの結果は、イベント取得後の @ref GetNfcInfo() にて取得できます。
 */
Result StartNtagWrite(const NtagWriteParameter& ntagWriteParam, DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       パススルーコマンドを発行します。
 *
 * @param[in]   passThruParam               パススルーコマンド用データ
 * @param[in]   handle                      対象のデバイスへのハンドル
 *
 * @retresult
 *      @handleresult{nn::ResultSuccess,                処理に成功しました。}
 *      @handleresult{nn::xcd::ResultNotConnected,      デバイスが接続されていません。}
 *      @handleresult{nn::xcd::ResultInvalidMcuState,   MCU が NFC を扱えるステートではありません。}
 *      @handleresult{nn::xcd::ResultMcuBusy,           実行中の処理があるため、要求を受け付けられません。}
 * @endretresult
 *
 * @pre         - XCD ライブラリは初期化済である必要があります。
 *
 * @details     パススルーコマンドを発行します。
 *              発行したパススルーコマンドの返答は @ref GetNfcInfo() で受け取ります。
 */
Result SendNfcRawData(const NfcPassThruParameter& passThruParam, DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       MIFARE 鍵を登録します。
 *
 * @param[in]   keyWriteParameter           MIFARE 鍵の書き込みパラメータ
 * @param[in]   handle                      対象のデバイスへのハンドル
 *
 * @retresult
 *      @handleresult{nn::ResultSuccess,                処理に成功しました。}
 *      @handleresult{nn::xcd::ResultNotConnected,      デバイスが接続されていません。}
 *      @handleresult{nn::xcd::ResultInvalidMcuState,   MCU が NFC を扱えるステートではありません。}
 *      @handleresult{nn::xcd::ResultMcuBusy,           実行中の処理があるため、要求を受け付けられません。}
 * @endretresult
 *
 * @pre         - XCD ライブラリは初期化済である必要があります。
 *
 * @details     MIFARE 鍵をデバイスに登録します。
 *              既に登録されている鍵がある場合、それらは新しい鍵の登録前にすべて消去されます。
 *
 *              鍵の登録が完了または失敗すると、 @ref SetNfcEvent() で登録したイベントがシグナルされます。
 *              処理結果は、イベント発生後に @ref GetNfcInfo() で取得できます。
 */
Result RegisterMifareKey(const MifareKeyWriteParameter& keyWriteParameter, DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       登録されている MIFARE 鍵を消去します。
 *
 * @param[in]   keyClearParameter           MIFARE 鍵の消去パラメータ
 * @param[in]   handle                      対象のデバイスへのハンドル
 *
 * @retresult
 *      @handleresult{nn::ResultSuccess,                処理に成功しました。}
 *      @handleresult{nn::xcd::ResultNotConnected,      デバイスが接続されていません。}
 *      @handleresult{nn::xcd::ResultInvalidMcuState,   MCU が NFC を扱えるステートではありません。}
 *      @handleresult{nn::xcd::ResultMcuBusy,           実行中の処理があるため、要求を受け付けられません。}
 * @endretresult
 *
 * @pre         - XCD ライブラリは初期化済である必要があります。
 *
 * @details     デバイスに登録されている MIFARE 鍵を消去します。
 *
 *              鍵の消去が完了または失敗すると、 @ref SetNfcEvent() で登録したイベントがシグナルされます。
 *              処理結果は、イベント発生後に @ref GetNfcInfo() で取得できます。
 */
Result ClearMifareKey(const MifareKeyClearParameter& keyClearParameter, DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       MIFARE タグの読み取りを行います。
 *
 * @param[in]   readParameter               MIFARE Read アクセス用のパラメータ
 * @param[in]   handle                      対象のデバイスへのハンドル
 *
 * @retresult
 *      @handleresult{nn::ResultSuccess,                処理に成功しました。}
 *      @handleresult{nn::xcd::ResultNotConnected,      デバイスが接続されていません。}
 *      @handleresult{nn::xcd::ResultInvalidMcuState,   MCU が NFC を扱えるステートではありません。}
 *      @handleresult{nn::xcd::ResultMcuBusy,           実行中の処理があるため、要求を受け付けられません。}
 * @endretresult
 *
 * @pre         - XCD ライブラリは初期化済である必要があります。
 *
 * @details     MIFARE のタグの読み取りを行います。
 *              読み取りが完了または失敗すると、 @ref SetNfcEvent() で登録したイベントがシグナルされます。
 *              読み取りの結果は、イベント発生後に @ref GetNfcInfo() で取得できます。
 */
Result StartMifareRead(const MifareReadParameter& readParameter, DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       MIFARE タグへの書き込みを行います。
 *
 * @param[in]   writeParameter              MIFARE Write アクセス用のパラメータ
 * @param[in]   handle                      対象のデバイスへのハンドル
 *
 * @retresult
 *      @handleresult{nn::ResultSuccess,                処理に成功しました。}
 *      @handleresult{nn::xcd::ResultNotConnected,      デバイスが接続されていません。}
 *      @handleresult{nn::xcd::ResultInvalidMcuState,   MCU が NFC を扱えるステートではありません。}
 *      @handleresult{nn::xcd::ResultMcuBusy,           実行中の処理があるため、要求を受け付けられません。}
 * @endretresult
 *
 * @pre         - XCD ライブラリは初期化済である必要があります。
 *
 * @details     MIFARE のタグへの書き込みを行います。
 *              書き込みが完了または失敗すると、 @ref SetNfcEvent() で登録したイベントがシグナルされます。
 *              書き込みの結果は、イベント発生後に @ref GetNfcInfo() で取得できます。
 */
Result StartMifareWrite(const MifareWriteParameter& writeParameter, DeviceHandle handle) NN_NOEXCEPT;

}} // namespace nn::xcd
