﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   xcd ライブラリの モーションIRカメラに関係する API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/xcd/xcd_Device.h>
#include "xcd_IrsensorTypes.h"

namespace nn { namespace xcd {

/**
 * @brief       モーションIRカメラを使用できる電源状態であるか確認します。
 *
 * @details     モーションIRカメラを使用できる電源状態であるか確認します。
 *              電池残量不足と判定された場合、モーションIRカメラを使用するためにデバイスを充電する必要があります。
 *
 * @param[in]   handle          確認するデバイスへのハンドル
 *
 * @retresult
 *      @handleresult{nn::ResultSuccess,                        モーションIRカメラを使用できる電源状態です。}
 *      @handleresult{nn::xcd::ResultNotConnected,              デバイスが接続されていません。}
 *      @handleresult{nn::xcd::ResultNotSupported,              モーションIRカメラ非対応のデバイスです。}
 *      @handleresult{nn::xcd::ResultLowBattery,                デバイスの電池残量が不足しています。}
 * @endresult
 *
 * @pre
 *              - XCD ライブラリは初期化済である必要があります。
 */
Result CheckIrDevicePower(DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief      デバイスから定期的な受け取るデータを モーションIRカメラが利用するフォーマットに指定します。
 *
 * @details    デバイスから定期的な受け取るデータを モーションIRカメラが利用するフォーマットに指定します。
 *
 * @param[in]  handle                                 データフォーマットを変更するデバイスへのハンドル
 *
 * @retresult
 *      @handleresult{nn::ResultSuccess,              処理に成功しました。}
 *      @handleresult{nn::xcd::ResultNotConnected,    デバイスが接続されていません。}
 * @endretresult
 *
 * @pre
 *             - xcd ライブラリは初期化済である必要があります。
 */
Result SetIrDataFormat(DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief      デバイスから定期的な受け取るデータを通常のコントローラが利用するフォーマットに戻します。
 *
 * @details    デバイスから定期的な受け取るデータを通常のコントローラが利用するフォーマットに戻します。
 *
 * @param[in]  handle                                 データフォーマットを変更するデバイスへのハンドル
 *
 * @retresult
 *      @handleresult{nn::ResultSuccess,              処理に成功しました。}
 *      @handleresult{nn::xcd::ResultNotConnected,    デバイスが接続されていません。}
 * @endretresult
 *
 * @pre
 *             - xcd ライブラリは初期化済である必要があります。
 */
Result ResetDataFormat(DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       Tera MCU の内部ステートを取得します。
 *
 * @details     Tera MCU の内部ステートを取得します。
 *
 * @param[out]  pOutMcuState                ステートの取得先
 * @param[in]   handle                      ステートを取得するデバイスへのハンドル
 *
 * @retresult
 *      @handleresult{nn::ResultSuccess,            処理に成功しました。}
 *      @handleresult{nn::xcd::ResultNotConnected,  デバイスが接続されていません。}
 * @endretresult
 *
 * @pre
 *              - XCD ライブラリは初期化済である必要があります。
 *              - pOutMcuState != nullptr
 */
Result GetIrMcuState(McuState* pOutMcuState, DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       Tera MCU の内部ステートを設定します。
 *
 * @details     Tera MCU の内部ステートを @a mcuState で設定したステートに遷移させます。
 *              事前に @ref SetDataFormat を呼び出してデータフォーマットを @ref PeriodicDataFormat_MCU に設定しておく必要があります。
 *
 * @param[in]   mcuState                    設定したいステート
 * @param[in]   handle                      ステートを設定するデバイスへのハンドル
 *
 * @retresult
 *      @handleresult{nn::ResultSuccess,            処理に成功しました。}
 *      @handleresult{nn::xcd::ResultNotConnected,  デバイスが接続されていません。}
 *      @handleresult{nn::xcd::ResultMcuBusy,       状態遷移処理中で MCU がビジー状態です。}
 * @endretresult
 *
 * @pre
 *              - XCD ライブラリは初期化済である必要があります。
 *              - @a mcuState は、 @ref McuState_Standby, @ref McuState_Background, @ref McuState_Nfc, @ref McuState_Ir のいずれかである必要があります。
 */
Result SetIrMcuState(McuState mcuState, DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       Tera MCU の内部ステートを設定します。
 *
 * @details     Tera MCU の内部ステートを @a mcuState で設定したステートに遷移させます。
 *              @ref SetMcuState とは異なり、事前に @ref SetDataFormat を呼び出しておく必要はありません。
 *
 * @param[in]   mcuState                    設定したいステート
 * @param[in]   handle                      ステートを設定するデバイスへのハンドル
 *
 * @retresult
 *      @handleresult{nn::ResultSuccess,            処理に成功しました。}
 *      @handleresult{nn::xcd::ResultNotConnected,  デバイスが接続されていません。}
 *      @handleresult{nn::xcd::ResultMcuBusy,       状態遷移処理中で MCU がビジー状態です。}
 * @endretresult
 *
 * @pre
 *  - XCD ライブラリは初期化済である必要があります。
 *  - @a mcuState は、 @ref McuState_Standby, @ref McuState_Background, @ref McuState_Nfc, @ref McuState_Ir のいずれかである必要があります。
 */
Result SetIrMcuStateImmediate(McuState mcuState, DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       モーションIRカメラの制御に必要な通知イベントを設定します。
 *
 * @param[in]   pIrSamplingEvent                モーションIRカメラのデータ受信の通知を受けるイベント
 * @param[in]   pIrCommandCompletionEvent       モーションIRカメラコマンドの送信予約完了通知を受けるイベント
 * @param[in]   handle                          デバイスへのハンドル
 *
 * @retresult
 *     @handleresult{nn::ResultSuccess,         処理に成功しました。}
 *     @handleresult{nn::ResultNotConnected,    デバイスが接続されていません。}
 * @endretresult
 *
 * @pre
 *              - XCD ライブラリは初期化済である必要があります。
 *              - pIrSamplingEvent != nullptr
 *              - *pIrSamplingEvent は初期化状態
 *              - *pIrSamplingEvent は非シグナル状態
 *              - pIrCommandCompletionEvent != nullptr
 *              - *pIrCommandCompletionEvent は初期化状態
 *              - *pIrCommandCompletionEvent は非シグナル状態
 *
 * @details     モーションIRカメラの各種制御が完了したときに、設定したイベントが通知されます。
 */
Result SetIrControlEvent(
    nn::os::SystemEventType* pIrSamplingEvent,
    nn::os::SystemEventType* pIrCommandCompletionEvent,
    DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       モーションIRカメラの制御に必要な通知イベントを解放します。
 *
 * @param[in]   handle                          デバイスへのハンドル
 *
 * @retresult
 *     @handleresult{nn::ResultSuccess,         処理に成功しました。}
 *     @handleresult{nn::ResultNotConnected,    デバイスが接続されていません。}
 * @endretresult
 *
 * @pre
 *              - XCD ライブラリは初期化済である必要があります。
 *
 * @details     モーションIRカメラの各種制御用の通知イベントをクリアして解放します。
 */
Result ReleaseIrControlEvent(DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       モーションIRカメラの処理方式を設定します。
 *
 * @param[in]   irProcessorType                 モーションIRカメラの処理方式
 * @param[in]   modeOffset                      モーションIRカメラの各処理方式内でのモード番号のオフセット
 * @param[in]   param                           TeraPlugin モードに渡すパラメータ
 * @param[in]   requiredVersion                 ライブラリが要求するバージョン
 * @param[in]   irCommandType                   使用するコマンドのタイプ
 * @param[in]   handle                          デバイスへのハンドル
 *
 * @retresult
 *     @handleresult{nn::ResultSuccess,         処理に成功しました。}
 *     @handleresult{nn::ResultNotConnected,    デバイスが接続されていません。}
 * @endretresult
 *
 * @pre
 *              - XCD ライブラリは初期化済である必要があります。
 *              - SetIrControlEvent を呼び出しておく必要があります。
 *
 * @details     モーションIRカメラの処理方式を設定します。
 *              MCU に対してコマンド発行が完了したら、IrCommandCompletionEvent がシグナルされます。
 *              設定の完了を知るためには nn::xcd::GetIrProcessorType で処理方式を取得する必要があります。
 */
Result SetIrProcessorType(IrProcessorType irProcessorType, int modeOffset, IrTeraPluginParameter param, IrMcuVersion requiredVersion, IrCommandType irCommandType, DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       モーションIRカメラの処理方式を取得します。
 *
 * @param[out]  pOutIrProcessorType             モーションIRカメラの処理方式
 * @param[out]  pOutRequiredVersion             現在の Mcu の互換動作バージョン
 * @param[in]   handle                          デバイスへのハンドル
 *
 * @retresult
 *     @handleresult{nn::ResultSuccess,         処理に成功しました。}
 *     @handleresult{nn::ResultNotConnected,    デバイスが接続されていません。}
 * @endretresult
 *
 * @pre
 *              - XCD ライブラリは初期化済である必要があります。
 *              - pOutIrProcessorType != nullptr
 *              - pOutCompatibleVersion != nullptr
 *              - SetIrControlEvent を呼び出しておく必要があります。
 *
 * @details     モーションIRカメラの処理方式を取得します。
 *              同時に MCU への処理方式取得リクエストを行いますので、最新の状態を取得するためには
 *              少なくとも BT の通信間隔を空けて、2回以上呼び出す必要があります。
 */
Result GetIrProcessorType(IrProcessorType* pOutIrProcessorType, IrMcuVersion* pOutCompatibleVersion, DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       モーションIRカメラのキャプチャを開始する。
 *
 * @param[in]   handle                          デバイスへのハンドル
 *
 * @retresult
 *     @handleresult{nn::ResultSuccess,         処理に成功しました。}
 *     @handleresult{nn::ResultNotConnected,    デバイスが接続されていません。}
 * @endretresult
 *
 * @pre
 *              - XCD ライブラリは初期化済である必要があります。
 *              - IrProcessorType が各 Processor に設定されている必要があります。
 *
 * @details     各モードのサンプリングを開始する非同期関数です。
 *              サンプリング後、 Setup 関数で設定されたワークバッファに、取得したデータをコピーします。
 *              コピーが完了したら、 SamplingEvent はシグナルされます。
 */
Result StartIrSampling(DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       モーションIRカメラのキャプチャを停止する。
 *
 * @param[in]   handle                          デバイスへのハンドル
 *
 * @retresult
 *     @handleresult{nn::ResultSuccess,         処理に成功しました。}
 *     @handleresult{nn::ResultNotConnected,    デバイスが接続されていません。}
 * @endretresult
 *
 * @pre
 *              - XCD ライブラリは初期化済である必要があります。
 *
 * @post
 *              - 各モードのサンプリング停止を保証します。
 *
 * @details     各モードのサンプリングを停止する同期関数です。
 *              停止が完了するまでロックします。
 */
Result StopIrSampling(DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       モーションIRカメラの各モードの終了処理を行います。
 *
 * @param[in]   handle                          デバイスへのハンドル
 *
 * @retresult
 *     @handleresult{nn::ResultSuccess,         処理に成功しました。}
 *     @handleresult{nn::ResultNotConnected,    デバイスが接続されていません。}
 * @endretresult
 *
 * @pre
 *              - XCD ライブラリは初期化済である必要があります。
 *              - StopIrSampling() によりサンプリングが停止している必要があります。
 *
 * @post
 *              - 各 Processor の Setup 関数で指定されたワークバッファの使用終了を保証します。
 *
 * @details     各モードのワークバッファの解放を行います。
 */
Result TeardownIrProcessor(DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       モーションIRカメラ利用時にポーリングモードを有効化します。
 *
 * @param[in]   handle                          デバイスへのハンドル
 *
 * @retresult
 *     @handleresult{nn::ResultSuccess,         処理に成功しました。}
 *     @handleresult{nn::ResultNotConnected,    デバイスが接続されていません。}
 * @endretresult
 *
 * @pre
 *              - XCD ライブラリは初期化済である必要があります。
 *
 * @post
 *              - BT MCU においてポーリングモードが有効化されます。
 *
 * @details     本体側からコントローラへのパケットが落ちた際に、BTMCU から TeraMCU に指定したパケットを送ってもらう、
 *              ポーリングモードを有効化します。 モードによっては有効化されないものもありますので、注意が必要です。
 */
Result EnableIrPollingMode(DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       モーションIRカメラ利用時にポーリングモードを無効化します。
 *
 * @param[in]   handle                          デバイスへのハンドル
 *
 * @retresult
 *     @handleresult{nn::ResultSuccess,         処理に成功しました。}
 *     @handleresult{nn::ResultNotConnected,    デバイスが接続されていません。}
 * @endretresult
 *
 * @pre
 *              - XCD ライブラリは初期化済である必要があります。
 *
 * @post
 *              - BT MCU においてポーリングモードが無効化されます。
 *
 * @details     ポーリングモードを強制的に無効化します。 コントローラが切断された場合も強制的に無効化されます。
 */
Result DisableIrPollingMode(DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       Moment モードの初期設定を行います。
 *
 * @param[in]   pIrCommonWorkBuffer             共通データのワークバッファ
 * @param[in]   pMomentWorkBuffer               Moment モードのワークバッファ
 * @param[in]   handle                          デバイスへのハンドル
 *
 * @retresult
 *     @handleresult{nn::ResultSuccess,         処理に成功しました。}
 *     @handleresult{nn::ResultNotConnected,    デバイスが接続されていません。}
 * @endretresult
 *
 * @pre
 *              - XCD ライブラリは初期化済である必要があります。
 *              - pIrCommonWorkBuffer != nullptr
 *              - pMomentWorkBuffer != nullptr
 *
 * @details     Moment モードのワークバッファを設定します。
 *              Moment モードのデータを MCU から読み込むための、ワークバッファを設定します。
 *              StartSampling でサンプリングを開始するまでに設定を完了しておく必要があります。
 */
Result SetupIrMomentProcessor(
    IrCommonData* pIrCommonWorkBuffer,
    IrMomentProcessorState* pMomentWorkBuffer,
    DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       Moment モードのデータを取得します。
 *
 * @param[out]  pOutIrCommonData                共通データ
 * @param[out]  pOutMomentData                  Moment モードのデータ
 * @param[out]  pOutCount                       取得したフレームの数
 * @param[in]   countMax                        取得するフレームの最大数
 * @param[in]   handle                          デバイスへのハンドル
 *
 * @retresult
 *     @handleresult{nn::ResultSuccess,         処理に成功しました。}
 *     @handleresult{nn::ResultNotConnected,    デバイスが接続されていません。}
 *     @handleresult{nn::ResultIrProcessorNotReady, 対応する IR プロセッサが未初期化です。}
 * @endretresult
 *
 * @pre
 *              - XCD ライブラリは初期化済である必要があります。
 *              - IrProcessorType が Moment に設定されている必要があります。
 *              - SetIrControlEvent を呼び出しておく必要があります。
 *              - pOutIrCommonData != nullptr
 *              - pOutMomentProcessorStates != nullptr
 *              - pOutCount != nullptr
 *
 * @details     Moment モードのデータを取得します。
 *              Moment モードに設定された状態で、データ受信のイベントがシグナルされた際に呼び出します。
 *              最新のものから過去に遡って利用可能な数だけ順に、
 *              指定のハンドルに対応する IR カメラとモーメントプロセッサによる処理結果を取得します。
 *              利用可能な処理結果の数より大きなバッファ（配列）が指定された場合、余った領域に対しては何も行いません。
 *              読み出し可能な処理結果の最大数は MomentProcessorStateCountMax 個です。
 *              利用可能な入力状態には読み出し済みのものも含まれます。
 */
Result GetIrMomentStates(
    IrCommonData* pOutIrCommonData,
    IrMomentProcessorState* pOutMomentProcessorStates,
    int* pOutCount,
    int countMax,
    DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       Clustering モードの初期設定を行います。
 *
 * @param[in]   pIrCommonWorkBuffer             共通データのワークバッファ
 * @param[in]   pClusteringWorkBuffer           Clustering モードのワークバッファ
 * @param[in]   handle                          デバイスへのハンドル
 *
 * @retresult
 *     @handleresult{nn::ResultSuccess,         処理に成功しました。}
 *     @handleresult{nn::ResultNotConnected,    デバイスが接続されていません。}
 * @endretresult
 *
 * @pre
 *              - XCD ライブラリは初期化済である必要があります。
 *              - pIrCommonWorkBuffer != nullptr
 *              - pClusteringWorkBuffer != nullptr
 *
 * @details     Clustering モードのワークバッファを設定します。
 *              Clustering モードのデータを MCU から読み込むための、ワークバッファを設定します。
 *              StartSampling でサンプリングを開始するまでに設定を完了しておく必要があります。
 */
Result SetupIrClusteringProcessor(
    IrCommonData* pIrCommonWorkBuffer,
    IrClusteringProcessorState* pOutClusteringWorkBuffer,
    DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       Clustering モードのデータを取得します。
 *
 * @param[out]  pOutIrCommonData                共通データ
 * @param[out]  pOutClusteringData              Clustering モードのデータ
 * @param[out]  pOutCount                       取得したフレームの数
 * @param[in]   countMax                        取得するフレームの最大数
 * @param[in]   handle                          デバイスへのハンドル
 *
 * @retresult
 *     @handleresult{nn::ResultSuccess,         処理に成功しました。}
 *     @handleresult{nn::ResultNotConnected,    デバイスが接続されていません。}
 *     @handleresult{nn::ResultIrProcessorNotReady, 対応する IR プロセッサが未初期化です。}
 * @endretresult
 *
 * @pre
 *              - XCD ライブラリは初期化済である必要があります。
 *              - IrProcessorType が Clustering に設定されている必要があります。
 *              - SetIrControlEvent を呼び出しておく必要があります。
 *              - pOutIrCommonData != nullptr
 *              - pOutClusteringProcessorStates != nullptr
 *              - pOutCount != nullptr
 *
 * @details     Clustering モードのデータを取得します。
 *              Clustering モードに設定された状態で、データ受信のイベントがシグナルされた際に呼び出します。
 *              最新のものから過去に遡って利用可能な数だけ順に、指定のハンドルに対応する IR カメラと
 *              クラスタリングプロセッサによる処理結果を取得します。
 *              利用可能な処理結果の数より大きなバッファ（配列）が指定された場合、余った領域に対しては何も行いません。
 *              読み出し可能な処理結果の最大数は ClusteringProcessorStateCountMax 個です。
 *              利用可能な入力状態には読み出し済みのものも含まれます。
 */
Result GetIrClusteringStates(
    IrCommonData* pOutIrCommonData,
    IrClusteringProcessorState* pOutClusteringProcessorStates,
    int* pOutCount,
    int countMax,
    DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       ImageTransfer モードの初期設定を行います。
 *
 * @param[in]   pIrCommonWorkBuffer             共通データのワークバッファ
 * @param[in]   pRawImageWorkBuffer             ImageTransfer モードで取得できる生データのワークバッファ
 * @param[in]   size                            ワークバッファのサイズの種類
 * @param[in]   handle                          デバイスへのハンドル
 *
 * @retresult
 *     @handleresult{nn::ResultSuccess,         処理に成功しました。}
 *     @handleresult{nn::ResultNotConnected,    デバイスが接続されていません。}
 * @endretresult
 *
 * @pre
 *              - XCD ライブラリは初期化済である必要があります。
 *              - pIrCommonWorkBuffer != nullptr
 *              - pImageTransferWorkBuffer != nullptr
 *
 * @details     ImageTransfer モードのワークバッファを設定します。
 *              サイズを指定して、モードのデータを MCU から読み込むための、ワークバッファを設定します。
 *              StartSampling でサンプリングを開始するまでに設定を完了しておく必要があります。
 */
Result SetupIrImageTransferProcessor(
    IrCommonData* pIrCommonWorkBuffer,
    IrImageTransferProcessorState* pImageTransferWorkBuffer,
    IrImageTransferProcessorFormat size,
    DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       ImageTransfer モードのデータを取得します。
 *
 * @param[out]  pOutIrCommonData                共通データ
 * @param[out]  pOutRawImage                    ImageTransfer モードで取得できる生データ
 * @param[in]   handle                          デバイスへのハンドル
 *
 * @retresult
 *     @handleresult{nn::ResultSuccess,         処理に成功しました。}
 *     @handleresult{nn::ResultNotConnected,    デバイスが接続されていません。}
 *     @handleresult{nn::ResultIrProcessorNotReady, 対応する IR プロセッサが未初期化です。}
 *     @handleresult{nn::ResultIrSamplingTooLate,   タイミングが遅すぎてデータが取得できません。}
 * @endretresult
 *
 * @pre
 *              - XCD ライブラリは初期化済である必要があります。
 *              - IrProcessorType が ImageTransfer に設定されている必要があります。
 *              - SetIrControlEvent を呼び出しておく必要があります。
 *              - pOutIrCommonData != nullptr
 *              - pOutImageTransferProcessorStates != nullptr
 *
 * @details     ImageTransfer モードのデータを取得します。
 *              ImageTransfer モードに設定された状態で、データ受信のイベントがシグナルされた際に呼び出します。
 *              1 フレーム分の画像が取得できます。
 *              SamplingEvent が通知された直後に実行してください。
 *              1 フレーム以上遅れると、バッファは再び内部で使用されるため、ResultIrSamplingTooLate のエラーが返ります。
 */
Result GetIrImageTransferState(
    IrCommonData* pOutIrCommonData,
    IrImageTransferProcessorState*
    pOutImageTransferProcessorState,
    DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       TeraPlugin モード用の初期設定を行います。
 *
 * @param[in]   pIrCommonWorkBuffer              共通データのワークバッファ
 * @param[in]   pTeraPluginWorkBuffer            TeraPlugin モード用の生データのワークバッファ
 * @param[in]   handle                           デバイスへのハンドル
 *
 * @retresult
 *     @handleresult{nn::ResultSuccess,          処理に成功しました。}
 *     @handleresult{nn::ResultNotConnected,     デバイスが接続されていません。}
 * @endretresult
 *
 * @pre
 *              - XCD ライブラリは初期化済である必要があります。
 *              - pIrCommonWorkBuffer != nullptr
 *              - pTeraPluginWorkBuffer != nullptr
 *
 * @details     TeraPlugin モードの初期設定を行います。
 *              TeraPlugin モードのデータを MCU から読み込むための、ワークバッファを設定します。
 *              StartSampling でサンプリングを開始するまでに設定を完了しておく必要があります。
 */
Result SetupIrTeraPluginProcessor(
    IrCommonData* pIrCommonWorkBuffer,
    IrTeraPluginProcessorState* pTeraPluginWorkBuffer,
    DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       TeraPlugin 用のデータを取得します。
 *
 * @param[out]  pOutIrCommonData                 共通データ
 * @param[out]  pOutTeraPluginData               TeraPlugin モード用の生データ
 * @param[in]   handle                           デバイスへのハンドル
 *
 * @retresult
 *     @handleresult{nn::ResultSuccess,          処理に成功しました。}
 *     @handleresult{nn::ResultNotConnected,     デバイスが接続されていません。}
 *     @handleresult{nn::ResultIrProcessorNotReady, 対応する IR プロセッサが未初期化です。}
 * @endretresult
 *
 * @pre
 *              - XCD ライブラリは初期化済である必要があります。
 *              - IrProcessorType が TeraPlugin に設定されている必要があります。
 *              - SetIrControlEvent を呼び出しておく必要があります。
 *              - pOutIrCommonData != nullptr
 *              - pOutTeraPluginProcessorState != nullptr
 *
 * @details     TeraPlugin モードの生データを取得します。
 *              詳細なデータのパースは呼び出し側で行います。
 */
Result GetIrTeraPluginState(
    IrCommonData* pOutIrCommonData,
    IrTeraPluginProcessorState* pOutTeraPluginProcessorState,
    DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       Dpd モードの初期設定を行います。
 *
 * @param[in]   pIrCommonWorkBuffer             共通データのワークバッファ
 * @param[in]   pDpdWorkBuffer                  Dpd モードのワークバッファ
 * @param[in]   handle                          デバイスへのハンドル
 *
 * @retresult
 *     @handleresult{nn::ResultSuccess,         処理に成功しました。}
 *     @handleresult{nn::ResultNotConnected,    デバイスが接続されていません。}
 * @endretresult
 *
 * @pre
 *              - XCD ライブラリは初期化済である必要があります。
 *              - pIrCommonWorkBuffer != nullptr
 *              - pDpdWorkBuffer != nullptr
 *
 * @details     Dpd モードのワークバッファを設定します。
 *              Dpd モードのデータを MCU から読み込むための、ワークバッファを設定します。
 *              StartSampling でサンプリングを開始するまでに設定を完了しておく必要があります。
 */
Result SetupIrDpdProcessor(
    IrCommonData* pIrCommonWorkBuffer,
    IrDpdProcessorState* pOutDpdWorkBuffer,
    DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       Dpd モードのデータを取得します。
 *
 * @param[out]  pOutIrCommonData                共通データ
 * @param[out]  pOutDpdData                     Dpd モードのデータ
 * @param[out]  pOutCount                       取得したフレームの数
 * @param[in]   countMax                        取得するフレームの最大数
 * @param[in]   handle                          デバイスへのハンドル
 *
 * @retresult
 *     @handleresult{nn::ResultSuccess,         処理に成功しました。}
 *     @handleresult{nn::ResultNotConnected,    デバイスが接続されていません。}
 *     @handleresult{nn::ResultIrProcessorNotReady, 対応する IR プロセッサが未初期化です。}
 * @endretresult
 *
 * @pre
 *              - XCD ライブラリは初期化済である必要があります。
 *              - IrProcessorType が Dpd に設定されている必要があります。
 *              - SetIrControlEvent を呼び出しておく必要があります。
 *              - pOutIrCommonData != nullptr
 *              - pOutDpdProcessorStates != nullptr
 *              - pOutCount != nullptr
 *
 * @details     Dpd モードのデータを取得します。
 *              Dpd モードに設定された状態で、データ受信のイベントがシグナルされた際に呼び出します。
 *              最新のものから過去に遡って利用可能な数だけ順に、指定のハンドルに対応する IR カメラと
 *              Dpd プロセッサによる処理結果を取得します。
 *              利用可能な処理結果の数より大きなバッファ（配列）が指定された場合、余った領域に対しては何も行いません。
 *              読み出し可能な処理結果の最大数は DpdProcessorStateCountMax 個です。
 *              利用可能な入力状態には読み出し済みのものも含まれます。
 */
Result GetIrDpdStates(
    IrCommonData* pOutIrCommonData,
    IrDpdProcessorState* pOutClusteringProcessorStates,
    int* pOutCount,
    int countMax,
    DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       モーションIRカメラのレジスタリードを開始します。
 *
 * @param[in]   irReadRegisterSetting            設定したいレジスタ内容
 * @param[in]   handle                           デバイスへのハンドル
 *
 * @retresult
 *     @handleresult{nn::ResultSuccess,          処理に成功しました。}
 *     @handleresult{nn::ResultNotConnected,     デバイスが接続されていません。}
 * @endretresult
 *
 * @pre
 *              - XCD ライブラリは初期化済である必要があります。
 *              - SetIrControlEvent を呼び出しておく必要があります。
 *
 * @details     モーションIRカメラのレジスタ読み込みを開始します。
 *              結果は、コマンド送信完了通知のイベントをシグナル後に GetReadRegisterState() 関数で取得します。
 */
Result StartIrReadRegister(const IrReadRegisterSetting& irReadRegisterSetting, DeviceHandle handle) NN_NOEXCEPT;

 /**
 * @brief       モーションIRカメラのレジスタ書き込みを開始します。
 *
 * @param[in]   irWriteRegisterSetting           設定したいレジスタデータ
 * @param[in]   handle                           デバイスへのハンドル
 *
 * @retresult
 *     @handleresult{nn::ResultSuccess,          処理に成功しました。}
 *     @handleresult{nn::ResultNotConnected,     デバイスが接続されていません。}
 * @endretresult
 *
 * @pre
 *              - XCD ライブラリは初期化済である必要があります。
 *              - SetIrControlEvent を呼び出しておく必要があります。
 *
 * @details     モーションIRカメラのレジスタ書き込みを開始します。
 *              結果（書き込み成否）は、コマンド送信完了通知のイベントをシグナル後に
 *              GetWriteRegisterState() 関数で取得します。
 */
Result StartIrWriteRegister(const IrWriteRegisterSetting& irWriteRegisterSetting, DeviceHandle handle) NN_NOEXCEPT;

 /**
 * @brief       モーションIRカメラのレジスタ書き込みの高速版を開始します。
 *
 * @param[in]   irWriteRegisterSettingEx         設定したいレジスタデータ(高速版)
 * @param[in]   handle                           デバイスへのハンドル
 *
 * @retresult
 *     @handleresult{nn::ResultSuccess,          処理に成功しました。}
 *     @handleresult{nn::ResultNotConnected,     デバイスが接続されていません。}
 * @endretresult
 *
 * @pre
 *              - XCD ライブラリは初期化済である必要があります。
 *              - SetIrControlEvent を呼び出しておく必要があります。
 *
 * @details     モーションIRカメラのレジスタ書き込みの高速版を開始します。
 *              結果（書き込み成否）は、コマンド送信完了通知のイベントをシグナル後に
 *              GetWriteRegisterState() 関数で取得します。
 */
Result StartIrWriteRegisterEx(const IrWriteRegisterSettingEx& irWriteRegisterSettingEx, DeviceHandle handle) NN_NOEXCEPT;

 /**
 * @brief       モーションIRカメラのレジスタ設定の読み込み結果を取得します。
 *
 * @param[out]  pOutIrReadRegisterState          レジスタ読み込みの結果
 * @param[in]   handle                           デバイスへのハンドル
 *
 * @retresult
 *     @handleresult{nn::ResultSuccess,          処理に成功しました。}
 *     @handleresult{nn::ResultNotConnected,     デバイスが接続されていません。}
 * @endretresult
 *
 * @pre
 *              - XCD ライブラリは初期化済である必要があります。
 *              - SetIrControlEvent を呼び出しておく必要があります。
 *              - pOutIrReadRegisterState != nullptr
 *
 * @details     モーションIRカメラのレジスタ設定の読み込み結果を取得します。
 *              本関数はコマンド送信完了通知用のイベントがシグナルされた後に呼ぶ必要があります。
 */
Result GetIrReadRegisterState(IrReadRegisterState* pOutIrReadRegisterState, DeviceHandle handle) NN_NOEXCEPT;

 /**
 * @brief       モーションIRカメラのレジスタ設定の書き込み結果を取得します。
 *
 * @param[out]  pOutIrWriteRegisterState         レジスタ書き込みの結果
 * @param[in]   handle                           デバイスへのハンドル
 *
 * @retresult
 *     @handleresult{nn::ResultSuccess,          処理に成功しました。}
 *     @handleresult{nn::ResultNotConnected,     デバイスが接続されていません。}
 * @endretresult
 *
 * @pre
 *              - XCD ライブラリは初期化済である必要があります。
 *              - SetIrControlEvent を呼び出しておく必要があります。
 *              - pOutIrWriteRegisterState != nullptr
 *
 * @details     モーションIRカメラのレジスタ設定の書き込み結果を取得します。
 *              本関数はコマンド送信完了通知用のイベントがシグナルされた後に呼ぶ必要があります。
 */
Result GetIrWriteRegisterState(IrWriteRegisterState* pOutIrWriteRegisterState, DeviceHandle handle) NN_NOEXCEPT;

}} // namespace nn::xcd

