﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       xcd ライブラリに関係する API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/os/os_Tick.h>
#include <nn/util/util_BitFlagSet.h>

#include <nn/xcd/xcd_BleDevice.h>
#include <nn/xcd/xcd_Device.h>

namespace nn { namespace xcd {


/**
 * @brief       Joy-Con 及びSwitch Proコントローラーのデジタルボタン定義です。
 */
struct PadButton
{
    typedef ::nn::util::BitFlagSet<32, PadButton>::Flag<0> A;      //!< Joy-Con 及び Switch Proコントローラーの a ボタン
    typedef ::nn::util::BitFlagSet<32, PadButton>::Flag<1> B;      //!< Joy-Con 及び Switch Proコントローラーの b ボタン
    typedef ::nn::util::BitFlagSet<32, PadButton>::Flag<2> X;      //!< Joy-Con 及び Switch Proコントローラーの x ボタン
    typedef ::nn::util::BitFlagSet<32, PadButton>::Flag<3> Y;      //!< Joy-Con 及び Switch Proコントローラーの y ボタン
    typedef ::nn::util::BitFlagSet<32, PadButton>::Flag<4> StickL; //!< Joy-Con 及び Switch Proコントローラーの L スティックボタン
    typedef ::nn::util::BitFlagSet<32, PadButton>::Flag<5> StickR; //!< Joy-Con 及び Switch Proコントローラーの R スティックボタン
    typedef ::nn::util::BitFlagSet<32, PadButton>::Flag<6> L;      //!< Joy-Con 及び Switch Proコントローラーの L ボタン
    typedef ::nn::util::BitFlagSet<32, PadButton>::Flag<7> R;      //!< Joy-Con 及び Switch Proコントローラーの R ボタン
    typedef ::nn::util::BitFlagSet<32, PadButton>::Flag<8> ZL;     //!< Joy-Con 及び Switch Proコントローラーの ZL ボタン
    typedef ::nn::util::BitFlagSet<32, PadButton>::Flag<9> ZR;     //!< Joy-Con 及び Switch Proコントローラーの ZR ボタン
    typedef ::nn::util::BitFlagSet<32, PadButton>::Flag<10> Start; //!< Joy-Con 及び Switch Proコントローラーの Start ボタン
    typedef ::nn::util::BitFlagSet<32, PadButton>::Flag<11> Select;//!< Joy-Con 及び Switch Proコントローラーの Select ボタン
    typedef ::nn::util::BitFlagSet<32, PadButton>::Flag<12> Left;  //!< Joy-Con 及び Switch Proコントローラーの 方向ボタン 左
    typedef ::nn::util::BitFlagSet<32, PadButton>::Flag<13> Up;    //!< Joy-Con 及び Switch Proコントローラーの 方向ボタン 上
    typedef ::nn::util::BitFlagSet<32, PadButton>::Flag<14> Right; //!< Joy-Con 及び Switch Proコントローラーの 方向ボタン 右
    typedef ::nn::util::BitFlagSet<32, PadButton>::Flag<15> Down;  //!< Joy-Con 及び Switch Proコントローラーの 方向ボタン 下
    typedef ::nn::util::BitFlagSet<32, PadButton>::Flag<16> SL;    //!< Joy-Con 及び Switch Proコントローラーの SL ボタン
    typedef ::nn::util::BitFlagSet<32, PadButton>::Flag<17> SR;    //!< Joy-Con 及び Switch Proコントローラーの SR ボタン
    typedef ::nn::util::BitFlagSet<32, PadButton>::Flag<18> Home;  //!< Joy-Con 及び Switch Proコントローラーの Home ボタン
    typedef ::nn::util::BitFlagSet<32, PadButton>::Flag<19> Shot;  //!< Joy-Con 及び Switch Proコントローラーの Shot ボタン
};


/**
 * @brief       Joy-Con , Switch Proコントローラーのデジタルボタンの集合を扱う型です。
 */
typedef ::nn::util::BitFlagSet<32, PadButton> PadButtonSet;


/**
 * @brief       アナログスティックの入力状態を表す構造体です。
 */
struct AnalogStickState
{
     int16_t x;  //!< アナログスティックの x 軸座標
     int16_t y;  //!< アナログスティックの y 軸座標
};

/**
 * @brief       遡って取得できる6軸センサーのサンプルの最大数
 */
const int SixAxisSensorSampleCountMax = 8;

/**
 * @brief       3軸をもつセンサーの値を表す構造体です。
 * @TODO: Deprecate this structure ?
 */
struct SensorState
{
    int32_t x;  //!< センサーの x 軸の値
    int32_t y;  //!< センサーの y 軸の値
    int32_t z;  //!< センサーの z 軸の値
};

/**
* @brief       Structure needed to store the raw data, before DScale processing
*/
struct SixAxisSensorRawParsing
{
    int64_t sampleNumber;   //!< サンプル番号
    SensorState data0;       //!< Accelerometer related data
    SensorState data1;       //!< Compressed gyro data and timestamps
};

/**
 * @brief       6軸センサーの値を表す構造体です。
 */
struct SixAxisSensorState
{
    int64_t sampleNumber; //!< サンプル番号
    nn::util::Float3 accelerometer;
    nn::util::Float3 gyroscope;
    float deltaTimeInSec;
};

/**
 * @brief       Joy-Con , Switch Proコントローラーのボタン, アナログスティック, 6軸センサーの入力状態を表す構造体です。
 */
struct PadState
{
    uint8_t sampleNumber;         //!< 8bit で表現されるサンプル番号
    PadButtonSet buttons;         //!< Joy-Con 及び Switch Proコントローラーのデジタルボタンの状態です。
    AnalogStickState analogStickL;//!< Joy-Con (L) のアナログスティック及び Switch Proコントローラーの左アナログスティックの状態です。
    AnalogStickState analogStickR;//!< Joy-Con (R) のアナログスティック及び Switch Proコントローラーの右アナログスティックの状態です。
    int8_t sensorSampleCount;     //!< センサーのサンプル数
    SensorState gyroscope[3];     //!< ジャイロセンサーの生値。1つのサンプルにつき最大3つの値が含まれる
    SensorState accelerometer[3]; //!< 加速度センサーの生値。1つのサンプルにつき最大3つの値が含まれる
};

/**
 * @brief       6軸センサーのキャリブレーション値を表す構造体です。
 */
struct SensorCalibrationValue
{
    SensorState accelerometerOrigin;        //!< 加速度センサーの原点値
    SensorState accelerometerSensitivity;   //!< 加速度センサーの感度
    SensorState gyroscopeOrigin;            //!< ジャイロセンサーの原点値
    SensorState gyroscopeSensitivity;       //!< ジャイロセンサーの感度
};

/**
 * @brief       アナログスティックのキャリブレーション値を表す構造体です。
 */
struct AnalogStickValidRange
{
    AnalogStickState origin;       //!< スティックの原点値
    AnalogStickState circuitMin;   //!< スティックの外周最小値
    AnalogStickState circuitMax;   //!< スティックの外周最大値
    int16_t          originPlay;   //!< 原点の遊び領域の大きさ
};

/**
 * @brief      アナログスティックのデバイス特性値を表す構造体です。
 */
struct AnalogStickDeviceParameter
{
    AnalogStickState minimumStrokePositive;    //!< 正方向の保証下限ストローク
    AnalogStickState minimumStrokeNegative;    //!< 負方向の保証下限ストローク
    AnalogStickState originRangeMin;           //!< 原点範囲の最小値
    AnalogStickState originRangeMax;           //!< 原点範囲の最小値
};

/**
 * @brief       設定可能な加速度センサーのFSRの値です。
 */
enum AccelerometerFsr : uint8_t
{
    AccelerometerFsr_2G = 2,       //!< ±2G
    AccelerometerFsr_4G = 4,       //!< ±4G
    AccelerometerFsr_8G = 8,       //!< ±8G
};

/**
 * @brief       設定可能なジャイロセンサーのFSRの値です。
 */
enum GyroscopeFsr : uint16_t
{
    GyroscopeFsr_250dps  = 250,     //!< ±250dps
    GyroscopeFsr_500dps  = 500,     //!< ±500dps
    GyroscopeFsr_1000dps = 1000,    //!< ±1000dps
    GyroscopeFsr_2000dps = 2000,    //!< ±2000dps
};

/**
 * @brief       キャリブレーション値の書き換え中の状況
 */
enum CalibrationUpdateStatus : uint16_t
{
    CalibrationUpdateStatus_None,         //!< キャリブレーション処理が行われていない、もしくは前の処理が成功しました
    CalibrationUpdateStatus_OnUpdate,     //!< キャリブレーション値の書き換え中です
    CalibrationUpdateStatus_OnReset,      //!< キャリブレーション値を工程の値に書き戻しています
    CalibrationUpdateStatus_VerifyError,  //!< キャリブレーション値の書き換え中に VerifyError が発生しました。
};

/**
 * @brief      ボタン/アナログスティック/6軸センサーの入力状態を取得します。
 *
 * @details    ボタン/アナログスティックの入力状態を取得します。
 *             DeviceStateに定義されている各デバイスの状態が即時が取得されます。
 *             6軸センサーは1回のデバイスとの通信において、最大3回分のサンプル値が取得されます
 *             初期化処理やモード変更などデバイスに対して特別な処理を行っている際は、6軸センサーの値のみ取得されないことがあります。
 *
 * @param[in]  handle                        状態を取得するデバイスへのハンドル
 * @param[out] pOutValue                     最新のボタン/アナログスティックの入力状態
 *
 * @return     実行結果を返します。
 * @retval     ResultSuccess                 処理に成功しました。
 * @retval     ResultNotConnected            デバイスが接続されていません。
 *
 * @pre        - xcd ライブラリは初期化済である必要があります。
 *             - pOutValue!= nullptr
 */
Result GetPadState(PadState* pOutValue, DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief      ボタン/アナログスティック/6軸センサーの入力状態を取得します。
 *
 * @details    ボタン/アナログスティックの入力状態を取得します。
 *             DeviceStateに定義されている各デバイスの状態が即時が取得されます。
 *             初期化処理やモード変更などデバイスに対して特別な処理を行っている際は、6軸センサーの値のみ取得されないことがあります。
 *
 * @param[in]  handle                        状態を取得するデバイスへのハンドル
 * @param[out] pOutValue                     最新のボタン/アナログスティックの入力状態
 *
 * @return     実行結果を返します。
 * @retval     ResultSuccess                 処理に成功しました。
 * @retval     ResultNotConnected            デバイスが接続されていません。
 *
 * @pre        - xcd ライブラリは初期化済である必要があります。
 *             - pOutValue!= nullptr
 */
Result GetPadState(PadState* pOutValue, BleConnectionHandle handle) NN_NOEXCEPT;

/**
 * @brief      ボタンが最後に押下された時間を取得します
 *
 * @details    ボタンが最後に押下されたタイミングのシステムチックを取得します。
 *             時間は現在のボタンの押下状態にかかわらず取得することができます
 *             接続時からボタンが一度も押下されていない場合は、ResultPadButtonNotTriggered() が返ります。
 *
 * @param[out] pOutValue                     最後にボタンが押下されてからの経過時間
 * @param[in]  buttonIndex                   押下時間を取得するボタンの PadButton における Index
 * @param[in]  handle                        状態を取得するデバイスへのハンドル
 *
 * @return     実行結果を返します。
 * @retval     ResultSuccess                 処理に成功しました。
 * @retval     ResultNotConnected            デバイスが接続されていません。
 * @retval     ResultPadButtonNotTriggered   ボタンが一度も押下されていません
 *
 * @pre        - xcd ライブラリは初期化済である必要があります。
 *             - pOutValue!= nullptr
 */
Result GetButtonTriggerElapsedTime(nn::os::Tick* pOutValue, int buttonIndex, DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief      6軸センサーの入力状態を取得します。
 *
 * @details    6軸センサーのサンプルを最大 SixAxisSensorSampleCountMax 分まで遡って取得することが可能です。
 *             6軸センサーのサンプルには samplingNumber が付与されます。
 *             パケットロストが発生した場合には、samplingNumber のみ Increment されます。
 *
 * @param[in]  handle                        状態を取得するデバイスへのハンドル
 * @param[out] pOutValue                     最新のボタン/アナログスティックの入力状態
 *
 * @return     実行結果を返します。
 * @retval     ResultSuccess                 処理に成功しました。
 * @retval     ResultNotConnected            デバイスが接続されていません。
 *
 * @pre        - xcd ライブラリは初期化済である必要があります。
 *             - pOutValue != nullptr
 *             - pOutCount != nullptr
 *             - length <= SixAxisSensorSampleCountMax && length > 0
 */
Result GetSensorStates(int* pOutCount, SixAxisSensorState* pOutValue, int length, DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief      6軸センサーの入力状態を取得します。
 *
 * @details    6軸センサーのサンプルを最大 SixAxisSensorSampleCountMax 分まで遡って取得することが可能です。
 *             6軸センサーのサンプルには samplingNumber が付与されます。
 *             パケットロストが発生した場合には、samplingNumber のみ Increment されます。
 *
 * @param[in]  handle                        状態を取得するデバイスへのハンドル
 * @param[out] pOutValue                     最新のボタン/アナログスティックの入力状態
 *
 * @return     実行結果を返します。
 * @retval     ResultSuccess                 処理に成功しました。
 * @retval     ResultNotConnected            デバイスが接続されていません。
 *
 * @pre        - xcd ライブラリは初期化済である必要があります。
 *             - pOutValue != nullptr
 *             - pOutCount != nullptr
 *             - length <= SixAxisSensorSampleCountMax && length > 0
 */
Result GetSensorStates(int* pOutCount, SixAxisSensorState* pOutValue, int length, BleConnectionHandle handle) NN_NOEXCEPT;

/**
 * @brief      左アナログスティックの有効範囲(上限値/下限値)を取得します。
 *
 * @details    指定のデバイスのアナログスティックの有効範囲(上限値/下限値)を取得します。
 *             アナログスティックの有効範囲はデバイスの種類ごとに異なります。
 *             上位ライブラリは有効範囲の値に合わせて適切な補正処理を行う必要があります。
 *             有効範囲の値は(0,0)を中心座標とした場合の、絶対値であらわされます。
 *
 * @param[out] pOutValue                     アナログスティックの上限値
 * @param[in]  handle                        値を取得するデバイスへのハンドル
 *
 * @return     実行結果を返します。
 * @retval     ResultSuccess                 処理に成功しました。
 * @retval     ResultNotConnected            デバイスが接続されていません。
 *
 * @pre        - xcd ライブラリは初期化済である必要があります。
 *             - pOutValue != nullptr
 */
Result GetLeftAnalogStickValidRange(AnalogStickValidRange* pOutValue, DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief      左アナログスティックの有効範囲(上限値/下限値)を更新します。
 *
 * @details    指定のデバイスのアナログスティックの有効範囲(上限値/下限値)を更新します
 *             アナログスティックの有効範囲の値はコントローラーのSerial Flash 上に格納されます。
 *             更新を行っても、製造工程で書き込まれたキャリブレーション値は更新されません
 *
 * @param[in]  value                         更新するアナログスティックの有効範囲の値
 * @param[in]  handle                        値を更新するデバイスへのハンドル
 *
 * @return     実行結果を返します。
 * @retval     ResultSuccess                 処理に成功しました。
 * @retval     ResultNotConnected            デバイスが接続されていません。
 * @retval     ResultDeviceOnUpdate          Serial Flash の更新処理中のため更新ができません。
 *
 * @pre        - xcd ライブラリは初期化済である必要があります。
 */
Result UpdateLeftAnalogStickValidRange(const AnalogStickValidRange& value, DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief      左アナログスティックの有効範囲(上限値/下限値)を工程検査で使用されたものにリセットします。
 *
 * @details    ユーザーの手元で更新されたアナログスティックの有効範囲(上限値/下限値)を破棄して
 *             製造工程で書き込まれたキャリブレーション値に戻します
 *
 * @param[in]  handle                        リセットするデバイスへのハンドル
 *
 * @return     実行結果を返します。
 * @retval     ResultSuccess                 処理に成功しました。
 * @retval     ResultNotConnected            デバイスが接続されていません。
 * @retval     ResultDeviceOnUpdate          Serial Flash の更新処理中のため更新ができません。
 *
 * @pre        - xcd ライブラリは初期化済である必要があります。
 */
Result ResetLeftAnalogStickValidRange(DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief      左アナログスティックの有効範囲(上限値/下限値)の更新処理の状況を取得します
 *
 * @details    キャリブレーション値の更新処理は時間がかかります。
 *             キャリブレーション値の更新処理の経過状況を取得します。
 *
 * @param[out] pOutValue                     シリアルフラッシュの更新処理の進行状況です
 * @param[in]  handle                        リセットするデバイスへのハンドル
 *
 * @return     実行結果を返します。
 * @retval     ResultSuccess                 処理に成功しました。更新処理は行われていません
 * @retval     ResultNotConnected            デバイスが接続されていません。
 *
 * @pre        - xcd ライブラリは初期化済である必要があります。
 */
Result GetLeftAnalogStickValidRangeUpdateStatus(CalibrationUpdateStatus* pOutValue, DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief      右アナログスティックの有効範囲(上限値/下限値)を取得します。
 *
 * @details    指定のデバイスのアナログスティックの有効範囲(上限値/下限値)を取得します。
 *             アナログスティックの有効範囲はデバイスの種類ごとに異なります。
 *             上位ライブラリは有効範囲の値に合わせて適切な補正処理を行う必要があります。
 *             有効範囲の値は(0,0)を中心座標とした場合の、絶対値であらわされます。
 *
 * @param[out] pOutValue                     アナログスティックの上限値
 * @param[in]  handle                        値を取得するデバイスへのハンドル
 *
 * @return     実行結果を返します。
 * @retval     ResultSuccess                 処理に成功しました。
 * @retval     ResultNotConnected            デバイスが接続されていません。
 *
 * @pre        - xcd ライブラリは初期化済である必要があります。
 *             - pOutValue != nullptr
 */
Result GetRightAnalogStickValidRange(AnalogStickValidRange* pOutValue, DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief      右アナログスティックの有効範囲(上限値/下限値)を更新します。
 *
 * @details    指定のデバイスのアナログスティックの有効範囲(上限値/下限値)を更新します
 *             アナログスティックの有効範囲の値はコントローラーのSerial Flash 上に格納されます。
 *             更新を行っても、製造工程で書き込まれたキャリブレーション値は更新されません
 *
 * @param[in]  value                         更新するアナログスティックの有効範囲の値
 * @param[in]  handle                        値を更新するデバイスへのハンドル
 *
 * @return     実行結果を返します。
 * @retval     ResultSuccess                 処理に成功しました。
 * @retval     ResultNotConnected            デバイスが接続されていません。
 * @retval     ResultDeviceOnUpdate          Serial Flash の更新処理中のため更新ができません。
 *
 * @pre        - xcd ライブラリは初期化済である必要があります。
 */
Result UpdateRightAnalogStickValidRange(const AnalogStickValidRange& value, DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief      右アナログスティックの有効範囲(上限値/下限値)を工程検査で使用されたものにリセットします。
 *
 * @details    ユーザーの手元で更新されたアナログスティックの有効範囲(上限値/下限値)を破棄して
 *             製造工程で書き込まれたキャリブレーション値に戻します
 *
 * @param[in]  handle                        リセットするデバイスへのハンドル
 *
 * @return     実行結果を返します。
 * @retval     ResultSuccess                 処理に成功しました。
 * @retval     ResultNotConnected            デバイスが接続されていません。
 * @retval     ResultDeviceOnUpdate          Serial Flash の更新処理中のため更新ができません。
 *
 * @pre        - xcd ライブラリは初期化済である必要があります。
 */
Result ResetRightAnalogStickValidRange(DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief      右アナログスティックの有効範囲(上限値/下限値)の更新処理の状況を取得します
 *
 * @details    キャリブレーション値の更新処理は時間がかかります。
 *             キャリブレーション値の更新処理の経過状況を取得します。
 *
 * @param[out] pOutValue                     シリアルフラッシュの更新処理の進行状況です
 * @param[in]  handle                        リセットするデバイスへのハンドル
 *
 * @return     実行結果を返します。
 * @retval     ResultSuccess                 処理に成功しました。更新処理は行われていません。
 * @retval     ResultNotConnected            デバイスが接続されていません。
 *
 * @pre        - xcd ライブラリは初期化済である必要があります。
 */
Result GetRightAnalogStickValidRangeUpdateStatus(CalibrationUpdateStatus* pOutValue, DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief      左アナログスティックのキャリブレーション値に利用するモデル毎の閾値情報を取得します
 *
 * @param[out] pOutValue                     モデル情報
 * @param[in]  handle                        デバイスへのハンドル
 *
 * @return     実行結果を返します。
 * @retval     ResultSuccess                 処理に成功しました。更新処理は行われていません。
 * @retval     ResultNotConnected            デバイスが接続されていません。
 *
 * @pre        - xcd ライブラリは初期化済である必要があります。
 */
Result GetLeftAnalogStickDeviceParameter(AnalogStickDeviceParameter* pOutValue, DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief      右アナログスティックのキャリブレーション値に利用するモデル毎の閾値情報を取得します
 *
 * @param[out] pOutValue                     モデル情報
 * @param[in]  handle                        デバイスへのハンドル
 *
 * @return     実行結果を返します。
 * @retval     ResultSuccess                 処理に成功しました。更新処理は行われていません。
 * @retval     ResultNotConnected            デバイスが接続されていません。
 *
 * @pre        - xcd ライブラリは初期化済である必要があります。
 */
Result GetRightAnalogStickDeviceParameter(AnalogStickDeviceParameter* pOutValue, DeviceHandle handle) NN_NOEXCEPT;

 /**
 * @brief      6軸センサーのキャリブレーション値を取得します。
 *
 * @details    6軸センサーのキャリブレーション値を取得します。
 *             キャリブレーション値は現在のFSRに合わせた値が返されます。
 *             FSRが変更になった場合はキャリブレーション値を再取得してください
 *             キャリブレーショ値は加速度/ジャイロ共に原点値及び感度から構成されます。
 *             Joy-Con , Switch Proコントローラーではそれぞれ以下の値が用いられます。
 * 　　　　　  加速度センサー：   原点値=0G   感度=1G      FSR:±2G
 *             ジャイロセンサー:  原点値=0rpm 感度=156rpm  FSR:±2000dps
 *
 * @param[out] pOutValue                     キャリブレーション値
 * @param[in]  handle                        デバイスへのハンドル
 *
 * @return     実行結果を返します。
 * @retval     ResultSuccess                 処理に成功しました。
 * @retval     ResultNotConnected            デバイスが接続されていません。
 *
 * @pre        - xcd ライブラリは初期化済である必要があります。
 *             - pOutValue != nullptr
 */
Result GetSensorCalibrationValue(SensorCalibrationValue* pOutValue, DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief      6軸センサーのキャリブレーション値を更新します。
 *
 * @details    指定のデバイスの6軸センサーのキャリブレーション値を更新します
 *             6軸センサーのキャリブレーション値はコントローラーのSerial Flash 上に格納されます。
 *             更新を行っても、製造工程で書き込まれたキャリブレーション値は更新されません
 *
 * @param[in]  value                         更新する6軸センサーのキャリブレーション値
 * @param[in]  handle                        値を更新するデバイスへのハンドル
 *
 * @return     実行結果を返します。
 * @retval     ResultSuccess                 処理に成功しました。
 * @retval     ResultNotConnected            デバイスが接続されていません。
 * @retval     ResultDeviceOnUpdate          Serial Flash の更新処理中のため更新ができません。
 *
 * @pre        - xcd ライブラリは初期化済である必要があります。
 */
Result UpdateSensorCalibrationValue(const SensorCalibrationValue& value, DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief      6軸センサーのキャリブレーション値を工程検査で使用されたものにリセットします。
 *
 * @details    ユーザーの手元で更新された6軸センサーのキャリブレーション値を破棄して
 *             製造工程で書き込まれたキャリブレーション値に戻します
 *
 * @param[in]  handle                        リセットするデバイスへのハンドル
 *
 * @return     実行結果を返します。
 * @retval     ResultSuccess                 処理に成功しました。
 * @retval     ResultNotConnected            デバイスが接続されていません。
 * @retval     ResultDeviceOnUpdate          Serial Flash の更新処理中のため更新ができません。
 *
 * @pre        - xcd ライブラリは初期化済である必要があります。
 */
Result ResetSensorCalibrationValue(DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief      6 軸センサーのキャリブレーション値の更新処理の状況を取得します
 *
 * @details    キャリブレーション値の更新処理は時間がかかります。
 *             キャリブレーション値の更新処理の経過状況を取得します。
 *
 * @param[out] pOutValue                     シリアルフラッシュの更新処理の進行状況です
 * @param[in]  handle                        リセットするデバイスへのハンドル
 *
 * @return     実行結果を返します。
 * @retval     ResultSuccess                 処理に成功しました。更新処理は行われていません。
 * @retval     ResultNotConnected            デバイスが接続されていません。
 *
 * @pre        - xcd ライブラリは初期化済である必要があります。
 */
Result GetSensorCalibrationValueUpdateStatus(CalibrationUpdateStatus* pOutValue, DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief      6 軸センサーのモデル毎の設置時の姿勢情報を取得します。
 *
 * @details    キャリブレーション中に利用する、モデル毎の設置時の姿勢情報を取得します
 *
 * @param[out] pOutValue                     モデル情報
 * @param[in]  handle                        デバイスへのハンドル
 *
 * @return     実行結果を返します。
 * @retval     ResultSuccess                 処理に成功しました。更新処理は行われていません。
 * @retval     ResultNotConnected            デバイスが接続されていません。
 *
 * @pre        - xcd ライブラリは初期化済である必要があります。
 */
Result GetSensorHorizontalOffset(SensorState* pOutValue, DeviceHandle handle) NN_NOEXCEPT;

 /**
 * @brief      加速度センサー/ジャイロセンサーのFSRを取得します。
 *
 * @details    加速度センサー/ジャイロセンサーの現在のFSRを取得します
 *             FSRを変更するとキャリブレーション値の解像度や感度の値が変わるため、GetSensorCalibrationValue()で再取得をしてください
 *             変更をする場合は、SetSensorFSR()を使用してください
 *
 * @param[out] pOutAccelerometerFsr          取得された加速度センサーのFSR
 * @param[out] pOutGyrosocopeFsr             取得されたジャイロセンサーのFSR
 * @param[out] pOutAccelerometerFsr          取得された加速度センサーのFSR
 * @param[in]  handle                        デバイスへのハンドル
 *
 * @return     実行結果を返します。
 * @retval     ResultSuccess                 処理に成功しました。
 * @retval     ResultNotConnected            デバイスが接続されていません。
 *
 * @pre        - xcd ライブラリは初期化済である必要があります。
 *             - pOutAccelerometerFSR != nullptr
 *             - pOutGyrosocopeFSR != nullptr
 */
Result GetSensorFsr(AccelerometerFsr* pOutAccelerometerFsr, GyroscopeFsr* pOutGyrosocopeFsr, DeviceHandle handle) NN_NOEXCEPT;

 /**
 * @brief      加速度センサー/ジャイロセンサーのFSRを設定します。
 *
 * @details    加速度センサー/ジャイロセンサーの現在のFSRを設定します
 *
 * @param[out] accelerometerFsr             設定する加速度センサーのFSR
 * @param[out] gyrosocopeFsr                設定するジャイロセンサーのFSR
 * @param[in]  handle                       デバイスへのハンドル
 *
 * @return     実行結果を返します。
 * @retval     ResultSuccess            処理に成功しました。
 * @retval     ResultNotConnected           デバイスが接続されていません。
 *
 * @pre        - xcd ライブラリは初期化済である必要があります。
 */
Result SetSensorFsr(AccelerometerFsr accelerometerFsr, GyroscopeFsr gyrosocopeFsr, DeviceHandle handle) NN_NOEXCEPT;

 /**
 * @brief      加速度センサー/ジャイロセンサーをスリープを有効/無効にします。
 *
 * @details    加速度センサー/ジャイロセンサーのスリープを有効/無効にします。
 *             センサーをスリープすると、消費電力を低減することができます。
 *
 * @param[in]  handle                        デバイスへのハンドル
 * @param[in]  sleep                         スリープの有効/無効
  *
 * @return     実行結果を返します。
 * @retval     ResultSuccess             処理に成功しました。
 * @retval     ResultNotConnected            デバイスが接続されていません。
 *
 * @pre        - xcd ライブラリは初期化済である必要があります。
  */
Result SleepSensor(bool sleep, DeviceHandle handle) NN_NOEXCEPT;

 /**
 * @brief      加速度センサー/ジャイロセンサーをスリープしているかどうかを取得します。
 *
 * @details    加速度センサー/ジャイロセンサーがスリープ中かどうかを取得します。
 *
 * @param[in]  handle                        デバイスへのハンドル
 * @param[out] pOutValue                     スリープの有効/無効
  *
 * @return     実行結果を返します。
 * @retval     ResultSuccess             処理に成功しました。
 * @retval     ResultNotConnected            デバイスが接続されていません。
 *
 * @pre        - xcd ライブラリは初期化済である必要があります。
 */
Result IsSensorSleep(bool* pOutValue, DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief      BLE デバイスのアナログスティックの有効範囲(上限値/下限値)を取得します。
 *
 * @details    指定のデバイスのアナログスティックの有効範囲(上限値/下限値)を取得します。
 *             アナログスティックの有効範囲はデバイスの種類ごとに異なります。
 *             上位ライブラリは有効範囲の値に合わせて適切な補正処理を行う必要があります。
 *             有効範囲の値は(0,0)を中心座標とした場合の、絶対値であらわされます。
 *
 * @param[out] pOutValue                     アナログスティックの上限値
 * @param[in]  handle                        値を取得するデバイスへのハンドル
 *
 * @return     実行結果を返します。
 * @retval     ResultSuccess                 処理に成功しました。
 * @retval     ResultNotConnected            デバイスが接続されていません。
 *
 * @pre        - xcd ライブラリは初期化済である必要があります。
 *             - pOutValue != nullptr
 */
Result GetAnalogStickValidRange(AnalogStickValidRange* pOutValue, BleConnectionHandle handle) NN_NOEXCEPT;

/**
 * @brief      BLE デバイスのアナログスティックのキャリブレーション値に利用するモデル毎の閾値情報を取得します
 *
 * @param[out] pOutValue                     モデル情報
 * @param[in]  handle                        デバイスへのハンドル
 *
 * @return     実行結果を返します。
 * @retval     ResultSuccess                 処理に成功しました。更新処理は行われていません。
 * @retval     ResultNotConnected            デバイスが接続されていません。
 *
 * @pre        - xcd ライブラリは初期化済である必要があります。
 */
Result GetAnalogStickDeviceParameter(AnalogStickDeviceParameter* pOutValue, BleConnectionHandle handle) NN_NOEXCEPT;

 /**
 * @brief      BLE デバイスの 6軸センサーのキャリブレーション値を取得します。
 *
 * @details    6軸センサーのキャリブレーション値を取得します。
 *             キャリブレーション値は現在のFSRに合わせた値が返されます。
 *             Palma ではそれぞれ以下の値が用いられます。
 * 　　　　　   加速度センサー：   原点値=0G   感度=1G      FSR:±2G
 *             ジャイロセンサー:  原点値=0rpm 感度=156rpm  FSR:±2000dps
 *
 * @param[out] pOutValue                     キャリブレーション値
 * @param[in]  handle                        デバイスへのハンドル
 *
 * @return     実行結果を返します。
 * @retval     ResultSuccess                 処理に成功しました。
 * @retval     ResultNotConnected            デバイスが接続されていません。
 *
 * @pre        - xcd ライブラリは初期化済である必要があります。
 *             - pOutValue != nullptr
 */
Result GetSensorCalibrationValue(SensorCalibrationValue* pOutValue, BleConnectionHandle handle) NN_NOEXCEPT;

/**
 * @brief      BLE デバイスの 6 軸センサーのモデル毎の設置時の姿勢情報を取得します。
 *
 * @details    キャリブレーション中に利用する、モデル毎の設置時の姿勢情報を取得します
 *
 * @param[out] pOutValue                     モデル情報
 * @param[in]  handle                        デバイスへのハンドル
 *
 * @return     実行結果を返します。
 * @retval     ResultSuccess                 処理に成功しました。更新処理は行われていません。
 * @retval     ResultNotConnected            デバイスが接続されていません。
 *
 * @pre        - xcd ライブラリは初期化済である必要があります。
 */
Result GetSensorHorizontalOffset(SensorState* pOutValue, BleConnectionHandle handle) NN_NOEXCEPT;

}} // namespace nn::xcd
