﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       xcd ライブラリに関係する API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/os.h>
#include <nn/xcd/xcd_Device.h>

namespace nn { namespace xcd {

/**
 * @brief       本体との通信方式です。
 */
enum InterfaceType : uint8_t
{
    InterfaceType_Bluetooth = 0,       //!< Bluetoothで通信しています。
    InterfaceType_Uart,                //!< UARTで通信しています。
    InterfaceType_Usb,                 //!< USBで通信しています。
    InterfaceType_Unknown,             //!< 不明なインタフェースで通信しています。
};

/**
 * @brief       デバイスの電池残量です。
 */
enum BatteryLevel : uint8_t
{
    BatteryLevel_None = 0,      //!< NONE
    BatteryLevel_CriticalLow,   //!< CRITICAL LOW
    BatteryLevel_Low,           //!< LOW
    BatteryLevel_Midium,        //!< MIDIUM
    BatteryLevel_High,          //!< HIGH
};

/**
 * @brief       Joy-Con コネクタの接続状態です。
 */
enum ConnectorState : uint8_t
{
    Connector_Detached = 0,           //!< 何も接続されていません
    Connector_AttachedToConsole,      //!< 本体に接続されています
    Connector_AttachedToGrip,         //!< 拡張グリップに接続されています
    Connector_AttachedToEXDevice,     //!< 拡張コントローラに接続されています
};

/**
 * @brief       デバイスから定期的な受信するデータのフォーマットです。
 */
struct DeviceStatus
{
    InterfaceType interfaceType;           //!< 本体との通信方式
    BatteryLevel batteryLevel;             //!< 電池残量
    bool charged;                          //!< 充電状態
    bool powered;                          //!< 給電状態
    bool cablePlugged;                     //!< 拡張グリップもしくは Switch ProコントローラーのUSBケーブルがUSBホストに接続されているかどうか
    ConnectorState connector;              //!< Joy-Con の有線コネクタの状態
    uint8_t        _AttachmentTypeTemp;    //!< Joy-Con の有線コネクタの状態の即値 (内部で使用するため、参照しないでください。)
    nn::os::Tick   _connectorTempGetTime;  //!< Joy-Con の有線コネクタの状態の即値を取得した時間 (内部で使用するため、参照しないでください。)
    bool isAttachmentReady;                //!< 拡張デバイスと UART 通信が可能かどうか
};

/**
 * @brief       デバイスの状態を取得します。
 *
 * @details     デバイスの状態を取得します。
 *              DeviceStatusに定義されている各デバイスの状態が即時が取得されます。
 *
 * @param[in]   handle                      状態を取得するデバイスへのハンドル
 * @param[out]  pDeviceStatus               現在のデバイスの状態
 *
 * @return      実行結果を返します。
 * @retval      ResultSuccess               処理に成功しました。
 * @retval      ResultNotConnected          デバイスが接続されていません。
 *
 * @pre         - xcd ライブラリは初期化済である必要があります。
 *              - pDeviceStatus != nullptr
 */
Result GetDeviceStatus(DeviceStatus* pDeviceStatus, DeviceHandle handle) NN_NOEXCEPT;


}} // namespace nn::xcd
