﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       BLE デバイスの列挙などに関する API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/bluetooth/bluetooth_AddressTypes.h>
#include <nn/hid/detail/hid_RxPacketHistoryTypes.h>
#include <nn/xcd/xcd_DeviceState.h>
#include <nn/os/os_SystemEventTypes.h>

namespace nn { namespace xcd {

/**
 * @brief xcd の中で扱われる BLE デバイス毎の接続ハンドルです
 */
typedef uint32_t BleConnectionHandle;

const BleConnectionHandle BleInvalidConnectionHandle = 0xFFFFFFFF; //!< 接続がない状態におけるハンドルの値です

const int BleDeviceCountMax = 2; //!< xcd がサポートする BLE デバイスの最大数です。最大数に達すると Scan, AutoConnect が停止されます

/**
 * @brief       xcd がサポートする BLE デバイスの接続状態を表す構造体です。
 */
struct BleDeviceList
{
    int32_t deviceCount; //!< 接続されているデバイスの数です。
    BleConnectionHandle handleList[BleDeviceCountMax]; //!< 接続されているデバイスのハンドルのリストです
};

/**
 * @brief       デバイスの種類です。
 */
enum BleDeviceType : uint8_t
{
    BleDeviceType_Unknown = 0,      //!< 動作対象外のデバイスです
    BleDeviceType_Palma,            //!< Palma
};

/**
 * @brief       デバイスの情報を定義する構造体です。
 */
struct BleDeviceInfo
{
    BleDeviceType deviceType;               //!< デバイスの種類です。
    ::nn::bluetooth::Address address;       //!< デバイスの識別子です。
};

/**
 * @brief       接続されているデバイスを列挙します
 *
 * @details     接続されているデバイスを列挙します。
 *              デバイスは本体に直接物理的に接続されている単位で列挙されます。
 *              あるデバイスの先に拡張機器が接続されている場合は、そのデバイスのいち入力情報として扱われます。
 *
 * @param[out]  pOutValue                    接続されているデバイスのリスト
 *
 * @return      実行結果を返します。
 * @retval      ResultSuccess                処理に成功しました。
 * @retval      ResultNotFound      　       デバイスは1台も接続されていません
 *
 * @pre         - xcd ライブラリは初期化済である必要があります。
 *              - pOutValue != nullptr
 */
Result ListDevices(BleDeviceList* pOutValue) NN_NOEXCEPT;

/**
 * @brief      無線接続されているデバイスを切断します。
 *
 * @details    無線接続されているデバイスを切断します。
 *             切断処理は非同期で行われ、切断処理が完了するとInitialize()で設定したnn::os::SystemEventTypeで通知を受け取ることができます。
 *
 * @param[in]  handle                        切断するデバイスへのハンドル
 *
 * @return     実行結果を返します。
 * @retval     ResultSuccess                  処理に成功しました。
 * @retval     ResultNotConnected             デバイスが接続されていません。
 * @retval     ResultNotWireless              指定のデバイスは無線接続されていないため切断できません。
 *
 * @pre        - xcd ライブラリは初期化済である必要があります。
 */
Result Detach(BleConnectionHandle handle) NN_NOEXCEPT;

/**
 * @brief      接続されているデバイスの情報を取得します。
 *
 * @details    デバイスの種類など接続されているデバイスの情報を取得します。
 *             取得されるデバイスの情報は静的なもので変化することはありません、
 *             電池残量のように動的に変化するコントローラの状態についてはGetDeviceStatus()を使用してください。
 *
 * @param[in]  handle                        情報を取得するデバイスへのハンドル
 * @param[out] pOutValue                     取得したデバイスの情報
 *
 * @return     実行結果を返します。
 * @retval     ResultSuccess                 処理に成功しました。
 * @retval     ResultNotConnected            デバイスが接続されていません。
 *
 * @pre         - xcd ライブラリは初期化済である必要があります。
 *              - pOutValue!= nullptr
 */
Result GetDeviceInfo(BleDeviceInfo* pOutValue, BleConnectionHandle handle) NN_NOEXCEPT;

/**
 * @brief       デバイスの状態を取得します。
 *
 * @details     デバイスの状態を取得します。
 *              DeviceStatusに定義されている各デバイスの状態が即時が取得されます。
 *
 * @param[in]   handle                      状態を取得するデバイスへのハンドル
 * @param[out]  pDeviceStatus               現在のデバイスの状態
 *
 * @return      実行結果を返します。
 * @retval      ResultSuccess               処理に成功しました。
 * @retval      ResultNotConnected          デバイスが接続されていません。
 *
 * @pre         - xcd ライブラリは初期化済である必要があります。
 *              - pDeviceStatus != nullptr
 */
Result GetDeviceStatus(DeviceStatus* pDeviceStatus, BleConnectionHandle handle) NN_NOEXCEPT;

/**
 * @brief      パケットの受信状況を取得します
 */
nn::hid::detail::RxPacketHistory GetRxPacketHistory(const BleConnectionHandle& handle) NN_NOEXCEPT;

}} // namespace nn::xcd
