﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   WLANに関する型や定数の宣言
 */

#pragma once

#include <nn/os.h>
#include <nn/nn_Common.h>
#include "wlan_Ssid.h"
#include "wlan_MacAddress.h"

namespace nn {
namespace wlan {

/*!
 * @brief ScanIEマッチングに用いるマッチング長の最大値です。
 */
static const uint32_t ScanIeMatchLengthMax = 255; // 802.11仕様上の最大値に設定

/*!
 * @brief 受信データマッチングに用いるマッチング長の最大値です。ローカル通信専用。
 */
static const uint32_t ReceivedDataMatchLengthMax = 20; // TORIAEZU


/*!
 * @brief ローカル通信Masterに接続できるClientの最大数です。
 */
static const uint8_t ConnectableClientsCountMax = 7;


/*!
 * @brief ローカル通信のMasterが送信するビーコンのサイズです。
 */
static const size_t BssIndicationBufferSize = 1300; // [MEMO] 1300 は暫定値。


/*!
 * @brief 無線デバイスで扱う可能性のある通信チャンネルの最大数です。
 */
static const uint8_t WirelessChannelsCountMax = 38;

/*!
 * @brief スキャンの際にスキャン対象として指定できるSSIDの最大数です。
 */
static const uint8_t ScanningSsidCountMax = 10;

/*!
 * @brief 受信したDataFrameを貯めておける最大数です。
 */
static const size_t DataFrameRxCapacityMax = 96;

/*!
 * @brief 受信エントリに登録できるEthertypeの最大数です。
 */
static const size_t EthertypeCountMax = 16;

/*!
 * @brief 受信したActionFrameを貯めておける最大数です。(暫定)
 */
static const size_t ActionFrameRxCapacityMax = 15;

/*!
 * @brief Action Frame用受信エントリに登録できるActionFrameTypeの最大数です。
 */
static const size_t ActionFrameTypeCountMax = 5;

/*!
 * @brief LocalMasterBssのビーコンと共に送信するAction Frameの最大サイズです。
 */
static const size_t BeaconActionFrameSizeMax = 1450;

/*!
 * @brief 単発で送信するAction Frameの最大サイズです。
 */
static const size_t OneShotActionFrameSizeMax = 1640;

/*!
 * @brief 定期送信Action Frameの送信間隔最大値です。(ms)
 */
static const size_t PeriodicActionFrameIntervalMax = 5000;

/*!
 * @brief すれちがい通信用ActionFrameに搭載可能なペイロード最大値
 */
static const size_t DetectActionFramePayloadMax = 1300;

/*!
 * @brief すれちがいスリープ時のActionFrameフィルタの最大数
 */
static const size_t DetectSleepFilterMax = 500;  // SIGLO-77819

/*!
 * @brief マルチキャストフィルタの最大数
 */
static const uint32_t MulticastFilterCountMax = 8;

/*!
 * @brief WiFiの通信チャンネルを示す列挙体です。2.4GHz帯、5GHz帯の両方を含んでいます。計38ch。
 */
enum WirelessChannel
{
    WirelessChannel_1ch = 1,       //!< 2.4GHz
    WirelessChannel_2ch = 2,
    WirelessChannel_3ch = 3,
    WirelessChannel_4ch = 4,
    WirelessChannel_5ch = 5,
    WirelessChannel_6ch = 6,
    WirelessChannel_7ch = 7,
    WirelessChannel_8ch = 8,
    WirelessChannel_9ch = 9,
    WirelessChannel_10ch = 10,
    WirelessChannel_11ch = 11,
    WirelessChannel_12ch = 12,
    WirelessChannel_13ch = 13,
    WirelessChannel_36ch = 36,      //!< 5GHz W52(DFS不要)
    WirelessChannel_40ch = 40,
    WirelessChannel_44ch = 44,
    WirelessChannel_48ch = 48,
    WirelessChannel_52ch = 52,      //!< 5GHz W53(DFS必要)
    WirelessChannel_56ch = 56,
    WirelessChannel_60ch = 60,
    WirelessChannel_64ch = 64,
    WirelessChannel_100ch = 100,    //!< 5GHz W56(DFS必要)
    WirelessChannel_104ch = 104,
    WirelessChannel_108ch = 108,
    WirelessChannel_112ch = 112,
    WirelessChannel_116ch = 116,
    WirelessChannel_120ch = 120,
    WirelessChannel_124ch = 124,
    WirelessChannel_128ch = 128,
    WirelessChannel_132ch = 132,
    WirelessChannel_136ch = 136,
    WirelessChannel_140ch = 140,
    WirelessChannel_144ch = 144,
    WirelessChannel_149ch = 149,    //!< 5GHz W58
    WirelessChannel_153ch = 153,
    WirelessChannel_157ch = 157,
    WirelessChannel_161ch = 161,
    WirelessChannel_165ch = 165,
};


/*!
 * @brief 通信レートを示す列挙体です。11gまでのレートセットです。
 */
enum RateSetLegacy
{
    RateSetLegacy_Default  = 0x0000,  //!< default or auto
    RateSetLegacy_1_0m     = 0x0001,  //!< 1Mbps    (802.11b)
    RateSetLegacy_2_0m     = 0x0002,  //!< 2Mbps    (802.11b)
    RateSetLegacy_5_5m     = 0x0004,  //!< 5.5Mbps  (802.11b)
    RateSetLegacy_11_0m    = 0x0008,  //!< 11Mbps   (802.11b)
    RateSetLegacy_6_0m     = 0x0010,  //!< 6Mbps    (802.11g)
    RateSetLegacy_9_0m     = 0x0020,  //!< 9Mbps    (802.11g)
    RateSetLegacy_12_0m    = 0x0040,  //!< 12Mbps   (802.11g)
    RateSetLegacy_18_0m    = 0x0080,  //!< 18Mbps   (802.11g)
    RateSetLegacy_24_0m    = 0x0100,  //!< 24Mbps   (802.11g)
    RateSetLegacy_36_0m    = 0x0200,  //!< 36Mbps   (802.11g)
    RateSetLegacy_48_0m    = 0x0400,  //!< 48Mbps   (802.11g)
    RateSetLegacy_54_0m    = 0x0800,  //!< 54Mbps   (802.11g)
    RateSetLegacy_11bMask  = ( RateSetLegacy_1_0m |
                               RateSetLegacy_2_0m |
                               RateSetLegacy_5_5m |
                               RateSetLegacy_11_0m ), //!< 802.11b
    RateSetLegacy_11gMask  = ( RateSetLegacy_1_0m  |
                               RateSetLegacy_2_0m  |
                               RateSetLegacy_5_5m  |
                               RateSetLegacy_11_0m |
                               RateSetLegacy_6_0m  |
                               RateSetLegacy_9_0m  |
                               RateSetLegacy_12_0m |
                               RateSetLegacy_18_0m |
                               RateSetLegacy_24_0m |
                               RateSetLegacy_36_0m |
                               RateSetLegacy_48_0m |
                               RateSetLegacy_54_0m ) //!< 802.11b/g
};


/*!
 * @brief IEEE802.11 Status Code を示す列挙体です。
 */
enum Dot11StatusCode
{
    Dot11StatusCode_Success                    = 0x0000, //!< 成功(0)
    Dot11StatusCode_UnspecifiedFailure         = 0x0001, //!< 不特定の失敗(1)
    Dot11StatusCode_UnsupportedCapabilities    = 0x000a, //!< 要求されたCapability情報をサポートできない。(10)
    Dot11StatusCode_AssocDeniedUnspec          = 0x000c, //!< 802.11標準の仕様外のために、Associationが拒否された。(12)
    Dot11StatusCode_UnsupportedAuthAlgorithm   = 0x000d, //!< 指定されたAuthenticationアルゴリズムをサポートしていない。(13)
    Dot11StatusCode_AuthSequenceError          = 0x000e, //!< Authenticationシーケンスエラー。(14)
    Dot11StatusCode_AuthChallengeFailed        = 0x000f, //!< Challenge Text失敗により認証が拒否された。(15)
    Dot11StatusCode_AuthFrameTimeout           = 0x0010, //!< AuthenticationタイムアウトによりAssociationが拒否された。(16)
    Dot11StatusCode_AssocDeniedNoResource      = 0x0011, //!< 無線アクセスポイントのリソースオーバーにより、拒否された。(17)
    Dot11StatusCode_AssocDeniedUnsupportedRate = 0x0012, //!< 要求したベーシックレートをサポートしていないため、拒否された。(18)
    Dot11StatusCode_Reserved                   = 0x0033, // 52-65535 : reserved
    Dot11StatusCode_Sized16                    = 0xffff  //
}; // IEEE802.11 Status Codes

/*!
 * @brief IEEE802.11 Reason Code を示す列挙体です。ローカル通信の Master が Client を切断する際に指定する必要があります。
 */
enum Dot11ReasonCode
{
    Dot11ReasonCode_Reserved                        = 0x0000,   //!< Reserved
    Dot11ReasonCode_UnspecifiedFailure              = 0x0001,   //!< 不特定の理由
    Dot11ReasonCode_AuthInvalid                     = 0x0002,   //!< 1つ前の Authentication はもう有効ではない
    Dot11ReasonCode_DeauthStaLeave                  = 0x0003,   //!< 送信している STA が IBSS または BSS を去ったために Deauthenticateした
    Dot11ReasonCode_DisassocInactivity              = 0x0004,   //!< 通信がないために Disassociate された
    Dot11ReasonCode_DisassocUnableToHandle          = 0x0005,   //!< AP が全ての Associate されている STA をハンドルできなくなったために Disassociate された
    Dot11ReasonCode_NonauthClass2Frame              = 0x0006,   //!< Class 2 フレームを Authenticate されていない STA から受信した
    Dot11ReasonCode_NonAssocClass3Frame             = 0x0007,   //!< Class 3 フレームを Associate されていない STA から受信した
    Dot11ReasonCode_DisassocStaLeaveLEV             = 0x0008,   //!< 送信している STA が IBSS または BSS を去ったために Disassociate した
    Dot11ReasonCode_AssocReqWithoutAuth             = 0x0009,   //!< (Re)Association要求しているSTA がその相手 とAuthenticateしていない
    Dot11ReasonCode_DisassocPowerCapability         = 0x000a,   //!< Power Capability IE が受け入れられないもののため、Disassociate された
    Dot11ReasonCode_DisassocUnacceptableSupportedCh = 0x000b,   //!< Supported Channels が受け入れられないもののため、Disassociate された
    Dot11ReasonCode_InvalidIe                       = 0x000d,   //!< 不正な IE
    Dot11ReasonCode_MicError                        = 0x000e,   //!< MIC エラー
    Dot11ReasonCode_4WayHandshakeTimeout            = 0x000f,   //!< 4 Way ハンドシェークがタイムアウトした
    Dot11ReasonCode_GroupKeyHandshakeTimeout        = 0x0010,   //!< Group Key の更新がタイムアウトした
    Dot11ReasonCode_4WayIeMismatch                  = 0x0011,   //!< 4 Way ハンドシェークのIEが (Re)Association Req/ Probe Resp/Beacon のものと違う
    Dot11ReasonCode_InvalidGroupCipher              = 0x0012,   //!< Group Cipher が正しくない
    Dot11ReasonCode_InvalidPairwiseCipher           = 0x0013,   //!< Pairwise Cipher が正しくない
    Dot11ReasonCode_InvalidAkmp                     = 0x0014,   //!< AKMP が正しくない
    Dot11ReasonCode_UnsupportedRsnIeVersion         = 0x0015,   //!< サポート外の RSN IE バージョン
    Dot11ReasonCode_InvalidRsnIeCapability          = 0x0016,   //!< 不正な RSN IE Capability
    Dot11ReasonCode_8021xAuthFailed                 = 0x0017,   //!< IEEE 802.1X 認証が失敗した
    Dot11ReasonCode_CipherSuiteRejected             = 0x0018,   //!< セキュリティポリシー毎の Cipher Suite が拒否された
    Dot11ReasonCode_Sized16                         = 0xffff    //
}; // IEEE802.11 Reacon Codes


/*!
 * @brief 無線セキュリティの種類を示す列挙体です。
 */
enum SecurityMode
{
    SecurityMode_Open,          //!< オープン接続。暗号化無し。
    SecurityMode_Wep64Open,     //!< WEP 64 (WEP 40) Open authentication
    SecurityMode_Wep64Shared,   //!< WEP 64 (WEP 40) Shared authentication
    SecurityMode_Wep128Open,    //!< WEP 128 (WEP 104) Open authentication
    SecurityMode_Wep128Shared,  //!< WEP 128 (WEP 104) Shared authentication
    SecurityMode_WpaTkip,       //!< WPA AES-TKIP
    SecurityMode_Wpa2Tkip,      //!< WPA2 AES-TKIP
    SecurityMode_WpaAes,        //!< WPA AES-CCMP
    SecurityMode_Wpa2Aes,       //!< WPA2 AES-CCMP
    SecurityMode_StaticAes      //!< AES固定鍵を使った暗号化です。ローカル通信でのみ使用します。
};

/*!
 * @brief マルチキャストフィルタのための設定情報を示す構造体です。
 */
struct MulticastFilterInfo
{
    uint8_t count;                               //!< Multicastフィルタの数です。
    MacAddress address[MulticastFilterCountMax]; //!< Multicastフィルターのデータです。
};

/*!
 * @brief 無線デバイスドライバの無線セキュリティを示す構造体です。
 *
   WPAの場合：
       PSK(事前共有鍵)をキーバッファにセットします。パスフレーズ入力(8～63文字のASCII文字)のみ対応です。
       パスワードをキーバッファにセットしないようご注意ください。

   Static AESの場合：
       16byte のStaticAESキーを1つ設定できます。

   鍵の入力の仕方
   - WEPの場合：
       鍵をASCIIコードに変換し、そのASCIIコードを文字列としてkey[]に入力して下さい。
       例）暗号鍵「password01234」-> ASCIIコード変換 -> 0x70 0x61 0x73 0x73 0x77 0x6f 0x72 0x64 0x30 0x31 0x32 0x33 0x34
           -> 文字列として入力: strcpy(&key[0], "70617373776f72643031323334");
   NOTE:WEP64の場合は、5文字の文字列が鍵となるので、必ずkey[]には10文字からなるASCIIコード文字列を入力して下さい。
        WEP128の場合は、13文字の文字列が鍵となるので、必ずkey[]には26文字からなるASCIIコード文字列を入力して下さい。
   - WPAの場合：
       PSK(パスフレーズ)をそのまま文字列でkey[]に入力して下さい。有効文字数は8～63文字です。
       または、PSK(16進数64文字)をそのまま文字列でkey[]に入力して下さい。有効文字は [0~9 a~f A~F] です。
       例）PSK(パスフレーズ)「password」->文字列として入力: strcpy(&key[0], "password");
           PSK(16進数64文字)「0123456789abcdef0123456789abcdef0123456789ABCDEF0123456789ABCDEF」
           ->文字列として入力: strcpy(&key[0], "0123456789abcdef0123456789abcdef0123456789abcdef0123456789abcdef");
   - StaticAESの場合：
       MASTER側、CLIENT側で全く同じ値をセットするようにして下さい。鍵長は16です。
       例）鍵: nn::Bit8 staticAesKey[] = {0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77...};
       -> そのまま入力: memcpy(&key[0], &staticAesKey[0], 16);

 */
struct Security
{
    SecurityMode privacyMode;       //!< セキュリティモードです。@ref SecurityMode の値から設定します。WPAの場合は、ペア暗号化のセキュリティモードを指定します。
    SecurityMode groupPrivacyMode;  //!< グループ暗号化のセキュリティモードです。 @ref SecurityMode の値から設定します。WPA使用時のみ有効です。
    uint32_t     keyIdx;            //!< 鍵番号です。WEPの場合のみ使用されます。0 - 3の範囲で指定して下さい。
    Bit8         key[65];           //!< 鍵本体です。Securityモードによって有効な文字数が異なりますので注意してください。（上記述参照）
};


/*!
 * @brief ビーコン受信通知の有無を示す列挙体です。
 */
enum BeaconIndication
{
    BeaconIndication_Enable,    //!< ビーコン受信通知あり
    BeaconIndication_Disable    //!< ビーコン受信通知なし
};


/*!
 * @brief ローカル通信のMASTERのBSSの設定情報を示す構造体です。
 */
struct MasterBssParameters
{
    /**
     * @brief   BSSを作成するチャンネルを指定します。1 - 11chの中から選択して下さい。@n
     *          -1を指定した場合、自動で最適なチャンネルが選ばれます。@n
     *          5GHzのチャンネルの使用はデバッグ用途のみとしてください。また、使用出来るチャンネルはW52の範囲(36ch, 40ch, 44ch, 48ch)のみです。
     */
    int16_t       channel;

    /**
     * @brief   ステルスSSID設定フラグです。 false:any true:hidden
     */
    bool          hiddenSsid;

    /**
     * @brief   通信がないClientを切断するまでの時間[sec]です。0～180の範囲で指定して下さい。@n
     *          ここで指定した時間になったらキープアライブフレームが送信され、それに応答がなかったら切断を行うという仕様になっています。@n
     *          0を指定した場合、キープアライブフレームの送信は行わず、また切断も行いません。(実際は非常に大きな値がセットされ、実質的に無効化しています。)
     */
    uint32_t      inactivePeriod;

    /**
     * @brief   （廃止予定：使わないでください）
     */
    bool          autoKeepAlive;

    /**
     * @brief   IEEE802.11 Supported Rates です。@ref RateSet の値を論理和で指定して使用します。(未実装です)
     */
    RateSetLegacy supportedRates;

    /**
     * @brief   IEEE802.11 Supported Rates の Basic Rates を示します。@ref RateSet の値を論理和で指定して使用します。(未実装です)
     */
    RateSetLegacy basicRates;

    /**
     * @brief   接続先のセキュリティプロファイルです。 @ref Security 参照。@n
     *          ローカル通信においては、SecurityMode_OpenまたはSecurityMode_StaticAesのみ有効です。@n
     *          さらにローカル通信では、privacyModeに入力して下さい。groupPrivacyModeへの入力は不要です。
     */
    Security      security;

    /**
     * @brief   使用する SSID です。
     */
    Ssid          ssid;

    /**
     * @brief    使用する Beacon Interval です。(未実装)
     */
    uint16_t      beaconInterval;
};


/*!
 * @brief スキャンタイプを示す列挙体です。
 */
enum ScanType
{
    ScanType_Active, //!< アクティブスキャン。ProbeRequestフレームを送信してスキャンを行います。
    ScanType_Passive //!< パッシブスキャン。無線フレームの送信を行わずにスキャンを行います。
};


/*!
 * @brief スキャンフィルタのための設定情報を示す構造体です。
 */
struct ScanIeMatchInfo
{
    Bit8 matchData[ScanIeMatchLengthMax];    //!< IE に含まれる照合対象のデータです。Length を除いた部分をセットしてください。
    uint8_t matchLength;                        //!< matchData に設定するデータ長です。
};


/*!
 * @brief 受信データフィルタのための設定情報を示す構造体です。
 */
struct ReceivedDataMatchInfo
{
    Bit8 matchData[ReceivedDataMatchLengthMax]; //!< 受信データに対する照合対象のデータです。EtherTypeフィールドの次に続くデータに対しマッチングを行います。
    uint8_t matchLength;                        //!< matchData に設定するデータ長です。
};

/*!
 * @brief 無線デバイスドライバのスキャンパラメータを示す構造体です。
 *
 * アクセスポイントまたは無線親機のスキャンを開始する際のスキャン動作を設定します。
 */
struct ScanParameters
{
    /**
     * @brief   パッシブスキャン または アクティブスキャンの指定を行います。
     */
    ScanType  scanType;

    /**
     * @brief   Scanを行う無線チャンネルのリストです。2.4GHz帯/5GHz帯両方のチャンネルを指定することが可能です。@n
     *          ここで指定した各チャンネル毎に、channelScanTimeずつスキャンが行われます。
     */
    int16_t   channelList[WirelessChannelsCountMax];

    /**
     * @brief   スキャンするチャンネルの数です。0を指定した場合、channelListの中味に関わらず、スキャン可能な全チャンネルに対しスキャンが行われます。@n
     *          0以外を指定した場合、channelListの先頭から指定した数のチャンネル分スキャンが行われます。実質、channelList[]の要素数を渡す値として利用されます。
     */
    uint8_t   channelCount;

    /**
     * @brief   1チャンネルあたりのScan時間です。ミリ秒単位で指定します。@n
     *          -1(負数)を指定した場合、無線ドライバーのデフォルトの値が適用されます。@n
     *          有効数値は -1, 10 <= x <= 3000 です。@n
     *          デフォルト値は、 @n
     *          アクティブスキャンかつ未接続状態の場合：40ミリ秒 @n
     *          アクティブスキャンかつ接続状態の場合：20ミリ秒 @n
     *          パッシブスキャンの場合、接続状態に関わらず：110ミリ秒 @n
     */
    int32_t   channelScanTime;

    /**
     * @brief   複数チャンネルのスキャンを行なう場合、次のチャンネルのスキャンを行なう前に、@n
     *          自機がいるチャンネル（home channel）に一旦戻って滞在する時間を指定出来ます。ミリ秒単位で指定します。@n
     *          APやローカル通信のMASTERと接続している場合のみ有効です。@n
     *          この値を適切に与えることで、既接続状態のスキャン中でもhome channelでの通信が可能となります。@n
     *          -1(負数)を指定した場合、無線ドライバーのデフォルトの値が適用されます。デフォルト値は、45ミリ秒です。@n
     *          有効数値は -1, 0 <= x <= 500 です。@n
     *          既接続スキャンを行う場合は、適切な値を設定してください。長時間のスキャンを行って、@n
     *          接続端末からのビーコンを受信できない時間が続けば、ビーコンロストによる切断が発生する可能性があります。
     */
    int32_t   homeChannelTime;

    /**
     * @brief   スキャン対象とするSSIDを指定します。NULLを指定した場合SSIDの区別なく全てのSSIDをスキャン対象とします。
     */
    Ssid*     ssidList;

    /**
     * @brief   スキャンするSSIDの数です。0を指定した場合、全てのSSIDをスキャンします。@n
     *          0以外を指定した場合、ssidListの先頭から指定した数のSsid分スキャンが行われます。最大10個までスキャン出来ます。@n
     *          実質、ssidList[]の要素数を渡す値として利用されます。@n
     *          @ref ssidList がNULLまたは、この値が0の場合、全てのSSIDがスキャン対象となります。
     */
    uint8_t   ssidCount;

    /**
     * @brief   Scan対象MACアドレスです。@n
     *          nn::wlan::MacAddress::CreateBroadcastMacAddress()で作成したMACアドレスなら全てのノードを対象とします。@n
     *          nn::wlan::MacAddress::CreateZeroMacAddress()で作成したMACアドレスを指定することはできません。
     */
    MacAddress  bssid;
};


/*!
 * @brief LocalMaster 時の各 Client、また InfraSta 及び SapClient の 接続情報を示す列挙体です。
 */
enum ConnectionState
{
    ConnectionState_Idle         = 0x00, //!< 初期状態  // ConnectionStatus の中でのみ使用されます。切断が発生した場合もこのステートになります。
    ConnectionState_Connected    = 0x01, //!< 接続状態
    ConnectionState_Disconnected = 0x02, //!< 切断状態  // ClientsStatus の中でのみ使用されます。Clientの接続が切れた場合にこの値が入ります。
    ConnectionState_Joined       = 0x03  //!< Join 状態
};


/*!
 * @brief 接続状態が変更した際、その要因を示す列挙体です。
 */
enum CauseOfInfo
{
    CauseOfInfo_NoInfo = 0x0,             //!< 原因なし。
    CauseOfInfo_NoBss,                    //!< NO_NETWORK_AVAIL  所望の BSS が見つからない。
    CauseOfInfo_BeaconLost,               //!< LOST_LINK         Beacon Lost。
    CauseOfInfo_DisconnectReq,            //!< DISCONNECT_CMD    コマンド要求による切断。 Status(0:成功)  MasterではReasonCodeが入る。
    CauseOfInfo_RecieveDisconnect,        //!< BSS_DISCONNECTED  接続先より切断フレームを受けたことによる切断。  ReasonCodeが入る。
    CauseOfInfo_ConnectRequest,           //!< InfraSta/LocalClient時：Connect コマンド要求により試行。Master時：Clientからの接続試行。結果、接続完了。
    CauseOfInfo_ConnectFailure,           //!< AUTH_FAILED/ASSOC_FAILED/NO_RESOURCES_AVAIL  Connect 試行の失敗。 StatusCodeが入る。
    CauseOfInfo_ConnectFailureAfterAssoc, //!< Infra専用。WPAでのAssociation後の認証フェーズでの失敗。 ReasonCodeが入る。
    CauseOfInfo_JoinRequest,              //!< Join コマンド要求により試行。結果、Join 完了。
    CauseOfInfo_JoinFailure,              //!< Join 試行の失敗。
    CauseOfInfo_ReceiveBcDeauth,          //!< Spectator 時に Broadcast Deauthentication フレームを受信。
    CauseOfInfo_DecryptionError,          //!< 接続は維持されているが、暗号通信に失敗。
    CauseOfInfo_ClientTimeout,            //!< Clientから応答が来なくなったので、通信不可能と判断。(ClientStatus用)
    CauseOfInfo_WpsTimeout,               //!< WPSにてタイムアウトが発生し、WPS失敗。
    CauseOfInfo_SleepWithoutWoWL,         //!< 接続中ですが、WoWLに入るための情報がセットされていないので、本体スリープ前に自発的に切断しました。
    CauseOfInfo_Unknown                   //!< CSERV_DISCONNECT/INVALID_PROFILE/DOT11H_CHANNEL_SWITCH/PROFILE_MISMATCH/ADHOC_MERGE
};


/*!
 * @brief InfraSta 及び LocalClientSta/LocalClientSpectator の Connect/Join/Disconnect イベント発生時、WLAN 内に格納されるデータを示す構造体です。@n
 * LocalMasterでは、channel, ssid, bssidに自身の情報が格納されます。それ以外の値は0が入っていますので参照しないで下さい。
 */
struct ConnectionStatus
{
    ConnectionState state;        //!< @ref ConnectionState の値から設定される。
    CauseOfInfo cause;            //!< @ref CauseOfInfo の値。
    int16_t channel;             //!< 接続先のチャンネル。 切断が起きた場合、0が入ります。LocalMasterではBSSを作成したチャンネルが入ります。
    Ssid ssid;                    //!< 接続先のSSID。LocalMasterでは自身のSSIDが入ります。ステルスSSID設定をしていてもここには生の値が入ります。
    MacAddress bssid;             //!< 接続先ネットワークの BSSID。LocalMasterでは自身のMACアドレスが入ります。
    uint16_t aid;                 //!< Asocciation ID。
    uint16_t statusReasonCode;    //!< @ref Dot11StatusCode (操作要求の結果) または @ref Dot11ReasonCode (被切断の理由)を統一したもの。
    uint16_t capabilityInfo;      //!< Capability Information(固定長)フィールド
    uint16_t beaconInterval;      //!< 接続先の Beacon Interval [ms](未実装です)
};


/*!
 * @brief LocalMaster の Connect/Disconnect イベント発生時、WLAN 内に格納されるデータを示す構造体です。
 */
struct ClientStatus
{
    ConnectionState state;        //!< @ref ConnectionState の値から設定される。
    CauseOfInfo cause;            //!< @ref CauseOfInfo の値。
    MacAddress clientMacAddress;  //!< InfraSta/LocalClient の MAC アドレス。
    uint16_t statusReasonCode;    //!< @ref Dot11StatusCode (操作要求の結果) または @ref Dot11ReasonCode (被切断の理由)を統一したもの。
    uint16_t capabilityInfo;      //!< Capability Information(固定長)フィールド
    int32_t  rssi;                //!< LocalMasterとの間のRSSI値
    nn::os::Tick updateTick;      //!< wlanプロセス内でClientStatusが更新された時のシステムチック値。
};


/*!
 * @brief ローカル通信の動作モードを示す列挙体です。
 */
enum LocalCommunicationMode
{
    LocalCommunicationMode_Master,          //!< Masterモード
    LocalCommunicationMode_ClientSpectator  //!< Client/Spectatorモード
};


/*!
 * @brief 無線デバイスドライバのClient切断要求用情報を示す構造体です。(LocalMaster専用)
 */
struct DisconnectClient
{
    MacAddress clientMacAddress; //!< 切断したいClientのMACアドレスです。
    Dot11ReasonCode reasonCode;  //!< IEEE802.11 Reacon Code( @ref Dot11ReasonCode )を指定します。
};


/*!
 * @brief マネージメントフレームのタイプを示す列挙体です。
 */
enum ManagementFrameType
{
    ManagementFrameType_Beacon      = 0x01, //!< Beacon フレーム
    ManagementFrameType_ProbeRes    = 0x02, //!< Probe response フレーム
    ManagementFrameType_AssocRes    = 0x04, //!< Assoc/Reassoc response フレーム
    ManagementFrameType_AuthRes     = 0x08, //!< Auth response フレーム
    ManagementFrameType_ProbeReq    = 0x10, //!< Probe request フレーム
    ManagementFrameType_AssocReq    = 0x20  //!< Assoc/Reassoc request フレーム
};


/*!
 * @brief Action Frameのタイプを示す列挙体です。
 */
enum ActionFrameType
{
    // ActionFrameTypeの1~3は旧すれちがい通信用に内部で既に予約済みです。
    // 旧すれちがい機能（3DSと同じ仕組みのもの）を開発予定でしたが、ActionFrameだけでデータ通信を行う新方式の実装が決まりました。
    // (SIGLO-72728)に詳細が載っています。
    // 旧すれちがい用Subtypeは一応残しつつ、空いていた7を新すれちがい用に使うことになりました。
    // 6は、ローカル通信用途で拡張されるかもしれないので空けています。
    ActionFrameType_Beacon         = 4,  //!< ローカル通信ビーコン用途用
    ActionFrameType_Local          = 5,  //!< ローカル通信用
    ActionFrameType_Reserved1      = 6,  //!< 用途未設定
    ActionFrameType_Detect = 7,  //!< 新すれちがい用
    ActionFrameType_Reserved3      = 8,  //!< 用途未設定
    ActionFrameType_End            = 9   //!< 範囲チェック用。使用しないでください。
};

/*!
 * @brief @ref StartScan 関数のスキャンバッファに入るデータ構造です。1つのBSS情報が格納されます。@n
 * VIEが付加されている場合、この直後からVIEのサイズとデータが続きます。
 * +--------------------+             -
 * |      BssInfo       |             |
 * |                    |             | size = sizeof(BssInfo)
 * |                    |             |
 * +--------------------+ <- IEが続く       -
 * |       1st IE       |             |
 * |        (id)        |             |
 * |       (length)     |             | size = id長さ(uint8_t) + length長(uint8_t) + データ長(length)
 * |        (data)      |             |
 * +--------------------+ <- 2つめのIE  -
 * |        2nd IE      |             |
 * |        (id)        |             |
 * |       (length)     |             | size = id長さ(uint8_t) + length長(uint8_t) + データ長(length)
 * |        (data)      |             |
 * +--------------------+             -
 *
 * total size = bssSize ( = sizeof(BssInfo)
 *                        + 1st IE size( sizeof(uint8_t) * 2 + length )
 *                        + 2nd IE size( sizeof(uint8_t) * 2 + length )
 *                      )
 *
 * BSS情報が複数ある場合、1つ目のBssInfoの直後に2つ目のBssInfoが続きます。つまり2つ目のBssInfoの先頭アドレスは、@n
 * 1つ目のBssInfoの先頭アドレス＋bssSizeとなります。
 * IE はId,Length,dataが入っています。一般的なInformation Elementフォーマットがそのまま入っています。
 */
struct BssInfo {
    uint32_t     bssSize;  // IEも含めたBSSのサイズ。上記のtotal sizeに当たります。
    uint32_t     ieCount;  // IEの数
    Bit8         bssid[nn::wlan::MacAddress::MacAddressSize];
    uint32_t     ssidLength;
    Bit8         ssid[nn::wlan::Ssid::SsidLengthMax];
    int16_t      channel;
    uint32_t     linkLevel;
    int16_t      rssi;
    uint16_t     beaconPeriod;
    Bit16        capability;
    int32_t      rateCount;  // Rates[]に入っているRateの数
    uint8_t      rates[16];  // support rate
};

/* !
 * @brief スキャンバッファの先頭に配置されるヘッダです。@n
 * @ref StartScan 関数で与えられたバッファの先頭に本ヘッダが配置されます。@n
 * その後、@ref BssInfo 構造体が続き、実際のBSS情報が配置されていきます。
 */
struct ScanResultHeader {
    uint32_t bufferLength; // スキャンバッファのサイズ
    uint32_t resultLength; // 実際に取得したスキャン結果の全体サイズ
    uint32_t bssCount;  // スキャン結果に入っているBSSの総数
};

/*!
 * @brief 通信品質(リンクレベル) を表す列挙型です。
 */
enum LinkLevel
{
    LinkLevel_0 = 0,   //!< 非常に通信品質が悪い、もしくは通信が成立していない
    LinkLevel_1,       //!< 通信品質が悪い
    LinkLevel_2,       //!< 通信品質があまり良くない
    LinkLevel_3        //!< 通信品質がよい
};

/*!
 * @brief WPAの暗号タイプです。802.11i に準拠した値です。
 *
 * サポートされているものは、WPA_CIPHER_TKIP と WPA_CIPHER_AES_CCMP のみです。
 */
enum WpaCipherSuiteType
{
    WPA_CIPHER_USE_GROUPKEY = 0,   //!< グループキー使用 (このタイプはサポートしていません。)
    WPA_CIPHER_WEP_40BIT    = 1,   //!< 40ビット鍵長のWEP(IVを含めると64ビット) (このタイプはサポートしていません。)
    WPA_CIPHER_TKIP         = 2,   //!< TKIP
    WPA_CIPHER_AES_OCB      = 3,   //!< AES OCB (このタイプはサポートしていません。)
    WPA_CIPHER_AES_CCMP     = 4,   //!< AES CCMP
    WPA_CIPHER_WEP_104BIT   = 5,   //!< 104ビット鍵長のWEP(IVを含めると128ビット) (このタイプはサポートしていません。)
    WPA_CIPHER_NONE         = 255
};

/*!
 * @brief WPAの鍵管理方式タイプです。802.11i に準拠した値です。
 *
 * サポートされているものは、AKM_WPA_PSK のみです。
 */
enum KeyManagementType
{
    AKM_NO_WPA  = 0,   //!< WPA なし
    AKM_WPA_EAP = 1,   //!< WPA-EAP (このタイプはサポートしていません。)
    AKM_WPA_PSK = 2    //!< WPA-PSK
};

/*!
 * @brief 受信アンテナ切り替えパターンです。
 */
enum RxAntennaPattern
{
    RxAntennaPattern_0    = 1,   //!< 0番の無線アンテナのみを使用します。Bluetoothと共用です。
    RxAntennaPattern_1    = 2,   //!< 1番の無線アンテナのみを使用します。Bluetoothとは共用せずWiFi専用のアンテナです。
    RxAntennaPattern_Both = 3    //!< 0番と1番両方のアンテナを使用します。
};

/*!
 * @brief WPSのPIN方式時に使用されるPINコードの長さです。
 */
static const int WpsPinLength = 8;

/*!
 * @brief WPSの実行方式です。
 */
enum WpsMethod
{
    WpsMethod_Pbc,  //!< WPS Push button方式
    WpsMethod_Pin   //!< WPS PIN 入力方式
};

/*!
 * @brief WoWLスリープから起床した際の起床要因です。
 */
enum WowlWakeReason
{
    WowlWakeReason_Nothing,       //!< 起床要因が記録されていません。ユーザー操作やシステムによる自発的起床が発生した場合にこの要因となります。
    WowlWakeReason_Linkdown,      //!< APとの接続が切れたことによる起床です。
    WowlWakeReason_Magicpacket,   //!< MagicPacket受信による起床です。（現在はこの要因で起床することはありません）
    WowlWakeReason_PatternData,   //!< 登録した特定パターンのデータ受信による起床です。（現在はこの要因で起床することはありません）
    WowlWakeReason_TcpSessionData //!< 登録したTCPセッションからのデータ受信による起床です。
};

/*!
 * @brief WoWLにおける各種機能です。(デバッグ用途)
 */
enum WowlFeatures
{
    WowlFeatures_ArpOffload     = (1 << 0),   //!< ARP offload
    WowlFeatures_TcpKeepAlive   = (1 << 1)    //!< APとの接続が切れたことによる起床です。
};

/* !
 * @brief Ipv4アドレス@n
 */
struct WlanIpv4Address {
    uint8_t addr[4];
};

/* !
 * @brief WoWLスリープからの要因別復帰回数（デバッグ用）@n
 */
struct WowlWakeCount {
    uint32_t system;      // ユーザー操作やシステム起因の起床
    uint32_t magic;       // マジックパケット受信による起床
    uint32_t netPattern;  // 登録済みパターンデータ受信による起床
    uint32_t disconnect;  // 切断フレーム受信による起床
    uint32_t beaconLost;  // ビーコンロスとによる切断起床
    uint32_t gtkFail;     // GTK交換失敗による起床
    uint32_t tcpData;     // 登録済みTCPセッションからの有意なデータ受信による起床
    uint32_t other;       // その他の要因による起床
};

/* !
 * @brief WoWLスリープ中の無線チップの行動結果などの統計情報格納用(デバッグ用)@n
 */
struct WowlSleepStats {
    uint32_t cpuWakes; // 無線チップCPUが起床した回数
    uint32_t cpuActiveTime; // 無線チップが活動した時間（単位us）
    uint32_t totalMeasuredTime; // 計測時間
    uint32_t dtim; // DTIMを処理した回数
    uint32_t txPackets; // パケット送信回数
    uint32_t txBytes; // 累計送信データサイズ
    uint32_t txActiveTime; // 累計送信処理時間
    uint32_t rxPackets; // パケット受信回数
    uint32_t rxBytes; // 累計受信データサイズ
    uint32_t rxActiveTime; // 累計受信処理時間
    uint32_t idleTime; // 無線チップCPUの累計アイドル時間（単位us）
    uint32_t arpReply; // ARP応答回数
    uint32_t tcpKaAck; // TCP Keepalive 応答回数
    uint32_t gtkRenewal; // GTK更新応答回数
};

/* !
 * @brief チャンネルの混雑具合を示すパラメータ(デバッグ用)@n
 */
struct ChannelStats {
    uint16_t channel;         // 測定されたチャンネル
    uint8_t  txop;            // どれだけ送信機会があったかを示す値(0～255の間に正規化されている)
    int8_t   noiseLevel;      // WiFi以外の電波も考慮した電波ノイズレベル(dBm)
};


/* !
 * @brief ActionFrameのEtherType以降で使用している任天堂独自のフォーマット@n
 */
struct NintendoActionFrameHeader {
    uint8_t etherType;
    Bit8 oui[3];
    uint8_t subtype;
    uint8_t reserved;
};

/* !
 * @brief すれちがい通信にて使用するhash。@n
 */
struct DetectHash {
    uint8_t hash[8];
};

/* !
 * @brief すれちがい通信にて使用するプロトコルヘッダ。@n
 * すれちがい用ActionFrameでは @ref NintendoActionFrameHeaderの後にこのヘッダが続いている@n
 */
struct DetectHeader {
    uint8_t major;    // Major バージョン
    uint8_t minor;    // Minor バージョン
    uint8_t cmd;      // cmdフィールド
    uint8_t reserved;
    DetectHash hash;  // hash
};

/* !
 * @brief すれちがいActionFrameの送信、受信、Idle時間の設定（1周期分）
 */
struct DetectPeriodicAfCycle {
    uint32_t txInterval;    // 次の送信までの間隔(ms)
    uint32_t txCount;       // 1周期に送信を何回行うか
    uint32_t idleCount;     // 次の周期までに空ける間隔(txIntervalの倍数)
    uint32_t rxStartCount;  // 受信開始タイミング(txIntervalの倍数)
    uint32_t rxCount;       // 1周期に受信を何回行うか
};

/* !
 * @brief すれちがいスリープ時のフィルター用ハッシュリスト
 */
struct DetectHashList {
    uint64_t* pHashList;  // ハッシュリストへのポインタ
    uint32_t numOfHash;   // ハッシュの数
};

/*!
 * @brief すれちがいスリープから起床した際の起床要因です。
 */
enum DetectWakeReason
{
    DetectWakeReason_Nothing,       //!< 起床要因が記録されていません。ユーザー操作やシステムによる自発的起床が発生した場合にこの要因となります。
    DetectWakeReason_CacheFull      //!< ActionFrame受信バッファに空きがなくなったことによる起床
};

/* !
 * @brief すれちがいスリープからの要因別復帰回数（デバッグ用）@n
 */
struct DetectWakeCount {
    uint32_t system;      // ユーザー操作やシステム起因の起床
    uint32_t cacheFull;   // ActionFrame受信バッファに空きがなくなったことによる起床
    uint32_t other;       // その他の要因による起床
};

/* !
 * @brief @ref SetPeriodicActionFrameCycle で指定したパラメータの適用先@n
 */
enum DetectPeriodicAfCycleTarget
{
    DetectPeriodicAfCycleTarget_Hd,     // HostDrivenモード（起床時ActionFrame）に適用
    DetectPeriodicAfCycleTarget_Sa,     // StandAloneモード（スリープActionFrame）に適用）
    DetectPeriodicAfCycleTarget_Both,   // Hd, Sa 両方に適用
    DetectPeriodicAfCycleTarget_None    // どこにも適用しない
};

} // end of namespace wlan
} // end of namespace nn
