﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

namespace nn   {
namespace cbuf {

struct Frame
{
    uint32_t size;
    uint32_t reserved;
    uint8_t  data[1536];
};

class Cbuf
{
private:
    static const uint32_t ArraySize = 128;

    uint32_t      m_ReadIndex;
    uint32_t      m_WriteIndex;
    uint32_t      m_RxId;
    Frame         m_Array[ArraySize];

public:

    Cbuf()
    NN_NOEXCEPT :
        m_ReadIndex(0),
        m_WriteIndex(0),
        m_RxId(0)
    {

    }

    inline uint32_t GetRxId()
    NN_NOEXCEPT
    {
        return m_RxId;
    }

    inline void SetRxId(uint32_t rxId)
    NN_NOEXCEPT
    {
        m_RxId = rxId;
    }

    Frame* GetWritePointer()
    NN_NOEXCEPT
    {
        nn::os::FenceMemoryLoadLoad();
        uint32_t readIndex  = m_ReadIndex;
        uint32_t writeIndex = m_WriteIndex;
        uint32_t emptyCount = (readIndex + ArraySize - writeIndex - 1) & (ArraySize - 1);
        NN_ABORT_UNLESS(readIndex < ArraySize);
        NN_ABORT_UNLESS(writeIndex < ArraySize);
        if (emptyCount)
        {
            return &m_Array[writeIndex];
        }
        return nullptr;
    }

    void CommitWrite()
    NN_NOEXCEPT
    {
        uint32_t writeIndex = m_WriteIndex;
        NN_ABORT_UNLESS(writeIndex < ArraySize);
        m_WriteIndex = (writeIndex + 1) & (ArraySize - 1);
    }

    Frame* GetReadPointer()
    NN_NOEXCEPT
    {
        Frame* pEntryOut = nullptr;
        uint32_t readIndex  = m_ReadIndex;
        uint32_t writeIndex = m_WriteIndex;
        uint32_t totalCount = (writeIndex + ArraySize - readIndex) & (ArraySize - 1);
        NN_ABORT_UNLESS(readIndex < ArraySize);
        NN_ABORT_UNLESS(writeIndex < ArraySize);
        if (totalCount > 0)
        {
            pEntryOut = &m_Array[readIndex];
        }
        return pEntryOut;
    }

    void AcknowledgeRead()
    NN_NOEXCEPT
    {
        uint32_t readIndex = m_ReadIndex;
        NN_ABORT_UNLESS(readIndex < ArraySize);
        m_ReadIndex  = (readIndex + 1) & (ArraySize - 1);
    }

    nn::Result Initialize()
    NN_NOEXCEPT
    {
        // check array size is power of 2
        NN_ABORT_UNLESS((ArraySize & (ArraySize - 1)) == 0);

        m_ReadIndex  = 0;
        m_WriteIndex = 0;
        return ResultSuccess();
    }

    nn::Result Finalize()
    NN_NOEXCEPT
    {
        return ResultSuccess();
    }
};

}}
