﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstring>
#include <nn/wlan/wlan_Common.h>
#include <nn/wlan/wlan_Types.h>
#include <nn/wlan/wlan_Ssid.h>
#include <nn/wlan/wlan_MacAddress.h>

namespace nn {
namespace wlan {

enum ElementId {
    ElementId_Ssid               = 0,
    ElementId_SupportedRates     = 1,
    ElementId_FhParameterSet     = 2,
    ElementId_DsParameterSet     = 3,
    ElementId_CfParameterSet     = 4,
    ElementId_Tim                = 5,
    ElementId_IbssParameterSet   = 6,
    ElementId_Country            = 7,
    ElementId_HpParameters       = 8,
    ElementId_HpTable            = 9,
    ElementId_Request            = 10,
    ElementId_QbssLoad           = 11,
    ElementId_EdcaParameterSet   = 12,
    ElementId_Tspec              = 13,
    ElementId_TrafficClass       = 14,
    ElementId_Schedule           = 15,
    ElementId_ChallengeText      = 16,

    ElementId_PowerConstraint    = 32,
    ElementId_PowerCapability    = 33,
    ElementId_TpcRequest         = 34,
    ElementId_TpcReport          = 35,
    ElementId_SupportedChannels  = 36,
    ElementId_ChSwitchAnnounce   = 37,
    ElementId_MeasureRequest     = 38,
    ElementId_MeasureReport      = 39,
    ElementId_Quiet              = 40,
    ElementId_IbssDfs            = 41,
    ElementId_ErpInformation     = 42,
    ElementId_TsDelay            = 43,
    ElementId_TclassProcessing   = 44,
    ElementId_HtCapability       = 45,
    ElementId_QosCapability      = 46,
    ElementId_Rsn                = 48,
    ElementId_ExSupportedRates   = 50,
    ElementId_HtInformation      = 61,

    ElementId_Vendor             = 221,
    ElementId_Wpa                = 221
};


/*!
    @name     情報要素解析
    @{
 */

/*!
    :category   情報要素解析
    @brief 情報要素解析クラス

    BeaconDescriptionReader 等から情報要素を取得した際に、この型で返されます。
    本クラスを単独で宣言およびコピーすることはできません。

    また、本クラスおよびその派生クラスは、バイナリデータからのキャストにより、各情報を抽出します。
    従って、本クラスおよびその派生クラスでバイナリ列に含まれないパディング領域が入るようにしたり、
    仮想関数を定義するといった事は行わないでください。
 */
class InfoElementReader
{
private:
    uint8_t       m_ElementId;
    uint8_t       m_Length;
public:
    // 単独宣言禁止
    InfoElementReader() NN_NOEXCEPT
    {
        NN_SDK_ASSERT("This class can not be created as an object. You can use this class for type casting only.\n");
    }

    ~InfoElementReader() NN_NOEXCEPT
    {
        NN_SDK_ASSERT("This class can not be created as an object. You can use this class for type casting only.\n");
    }

    // コピー禁止
    explicit InfoElementReader( const InfoElementReader& ) NN_NOEXCEPT
    {
        NN_SDK_ASSERT("This class can not be copied. You can use this class for type casting only.\n");
    }

    InfoElementReader & operator = (const InfoElementReader &) NN_NOEXCEPT
    {
        NN_SDK_ASSERT("This class can not be assigned. You can use this class for type casting only.\n");
        return *this;
    }

    /*!
       @brief  情報要素の Element ID を取得します
       @return 情報要素の Element ID
     */
    inline uint8_t       GetElementId() const NN_NOEXCEPT
    {
        return m_ElementId;
    }

    /*!
       @brief  情報要素の長さを取得します
       @return 情報要素の長さ
     */
    inline uint8_t       GetLength() const NN_NOEXCEPT
    {
        return m_Length;
    }

    /*!
       @brief  情報要素のデータの先頭へのポインタを取得します
       @return 情報要素のデータの先頭へのポインタ
     */
    inline const Bit8*  GetData() const NN_NOEXCEPT
    {
        return reinterpret_cast<const Bit8*>(this) + sizeof(InfoElementReader);
    }
};

/*!
    :category   情報要素解析
    @brief ベンダ情報要素解析クラス
 */
class VendorInfoElementReader : public InfoElementReader
{
private:
    uint8_t            m_Oui[MacAddress::OuiSize];
public:

    /*!
       @brief  ベンダ情報要素の OUI を取得します
       @return ベンダ情報要素の OUI 配列
     */
    inline uint8_t* GetOui() NN_NOEXCEPT
    {
        return m_Oui;
    }

    /*!
       @brief  ベンダ情報要素の OUI を取得します
       @return ベンダ情報要素の OUI 配列
     */
    inline const uint8_t* GetOui() const NN_NOEXCEPT
    {
        return m_Oui;
    }

    /*!
       @brief  ベンダ情報要素の OUI を除いたデータの先頭へのポインタを取得します
       @return ベンダ情報要素の OUI を除いたデータの先頭へのポインタ
     */
    inline const Bit8* GetVendorData() const NN_NOEXCEPT
    {
        return reinterpret_cast<const Bit8*>(this) + sizeof(VendorInfoElementReader);
    }
};


/*!
    :category   情報要素解析
    @brief WiFi 情報要素解析クラス
 */
class WifiInfoElementReader : public VendorInfoElementReader
{
private:
    uint8_t           m_Type;
public:
    /*!
       @brief  対象の情報要素が、WiFi 情報要素かどうかを返します
       @return WiFi 情報要素であるか否か
     */
    inline bool       IsValid() const NN_NOEXCEPT
    {
        return !std::memcmp(GetOui(), "\x00\x50\xF2", MacAddress::OuiSize) ? true:false;
    }

    /*!
       @brief  WiFi 情報要素の Type フィールドの値を取得します
       @return WiFi 情報要素の Type フィールドの値
     */
    inline uint8_t    GetType() const NN_NOEXCEPT
    {
        return m_Type;
    }
};

/*!
    :category   情報要素解析
    @brief WPA 情報要素解析クラス。

    WPA/WPA2の 情報要素を解析します。それぞれが以下のような型で
    BeaconDescriptionReader のGetWpaIe()/GetWpa2Ie()から渡されます。

    WPA: WpaInfoElementReader<WifiInfoElementReader>

    WPA2: WpaInfoElementReader<InfoElementReader>
 */
template <typename IeReader>
class WpaInfoElementReader : public IeReader
{
private:
    inline uint16_t            ToLe16(uint8_t hi, uint8_t lo) const NN_NOEXCEPT
    {
        return static_cast<uint16_t>( (hi << 8UL) | (lo & 0x00FFUL) );
    }
public:
    /*!
       @brief  WPA 情報要素のデータの先頭を示すポインタを取得します
       @return WPA 情報要素のデータの先頭を示すポインタ
     */
    inline const Bit8*         GetWpaData() const NN_NOEXCEPT
    {
        return reinterpret_cast<const Bit8*>(this) + sizeof(WpaInfoElementReader);
    }

    /*!
       @brief  WPA の version 情報を取得します
       @return WPA の version 情報
     */
    inline uint16_t             GetVersion() const NN_NOEXCEPT
    {
        const uint8_t* pPtr = &GetWpaData()[0];
        return ToLe16(pPtr[1], pPtr[0]);
    }

    /*!
       @brief  AP がサポートするグループ暗号のタイプを取得します。
       @return AP がサポートするグループ暗号のタイプ。
     */
    inline WpaCipherSuiteType    GetGroupCipherSuite() const NN_NOEXCEPT
    {
        return static_cast<WpaCipherSuiteType>(GetWpaData()[2 + 3]);
    }

    /*!
       @brief  AP がサポートするペア暗号の数を取得します。
       @return AP がサポートするペア暗号の数。
     */
    inline uint16_t               GetPairwiseCipherSuiteCount() const NN_NOEXCEPT
    {
        const uint8_t* pPtr = &GetWpaData()[2 + 4];
        return ToLe16(pPtr[1], pPtr[0]);
    }

    /*!
       @brief     AP がサポートするペア暗号のタイプを取得します。
       @param[in] index 取得するペア暗号が何番目のものかを示すインデクス値(0 から開始)
       @return    AP がサポートするペア暗号のタイプ。
     */
    inline WpaCipherSuiteType     GetPairwiseCipherSuite(uint8_t index) const NN_NOEXCEPT
    {
        if ( index >= GetPairwiseCipherSuiteCount() )
        {
            return WPA_CIPHER_NONE;
        }
        return static_cast<WpaCipherSuiteType>(GetWpaData()[2 + 4 + 2 + 4 * index + 3]);
    }

    /*!
       @brief  AP がサポートする鍵管理方式の数を取得します。
       @return AP がサポートする鍵管理方式の数。
     */
    inline uint16_t               GetAuthKeyManagementSuiteCount() const NN_NOEXCEPT
    {
        const uint8_t* pPtr = &GetWpaData()[2 + 4 + 2 + 4 * GetPairwiseCipherSuiteCount()];
        return ToLe16(pPtr[1], pPtr[0]);
    }

    /*!
       @brief  AP がサポートする鍵管理方式のタイプを取得します。
       @return AP がサポートする鍵管理方式のタイプ。
     */
    inline KeyManagementType      GetAuthKeyManagementSuite(uint8_t index) const NN_NOEXCEPT
    {
        if ( index >= GetAuthKeyManagementSuiteCount() )
        {
            return AKM_NO_WPA;
        }
        return static_cast<KeyManagementType>(GetWpaData()[2 + 4 + 2 + 4 * GetPairwiseCipherSuiteCount() + 2 + 4 * index + 3]);
    }
};



/*!
    @}
 */

/*!
    @name       BSS情報解析
    @{
 */

/*!
    :category   BSS情報解析
    @brief BssDescription を表す型です。
 */
class BssDescription
{
    // 各クラスの関係が分かりやすくなるように仮の型を使う
};

/*!
    :category   BSS情報解析
    @class BssDescriptionReaderBase
    @brief BSS Description  解析基底クラス
 */
class BssDescriptionReaderBase
{
public:
    /*!
       @brief     BssDescriptionReaderBase クラスのコンストラクタです。
       @param[in] pBuffer   解析するスキャンバッファのポインタ。
     */
    explicit BssDescriptionReaderBase(const BssDescription* pBuffer) NN_NOEXCEPT
        : m_pBuffer(reinterpret_cast<const uint8_t*>(pBuffer))
    {}

    /*!
       @brief     BssDescriptionReaderBase クラスのデストラクタです。
     */
    virtual ~BssDescriptionReaderBase() NN_NOEXCEPT
    {}

    /*!
       @brief BssDescription バッファを指定した領域にコピーします。
       @param[out] pOutput BssDescription をコピーする出力先のポインタ
       @param[in]  length  pOutput のサイズ
     */
    inline void  CopyDescriptionBuffer( void* pOutput, size_t length ) const NN_NOEXCEPT
    {
        size_t descLen = GetLength();
        if ( length < descLen )
        {
            return;
        }
        std::memcpy(static_cast<uint8_t*>(pOutput), m_pBuffer, GetLength());
    }

    /*!
       @brief BssDescription バッファへのポインタを取得します。
              取得されたバッファポインタのアライメントは保証されていませんのでご注意ください。
       @return BssDescription バッファへのポインタ。
     */
    inline const uint8_t* GetDescriptionBuffer() const NN_NOEXCEPT
    {
        return m_pBuffer;
    }

    /*!
      @brief BSS Description のサイズを取得します。
      @return BSS Description のサイズ。
     */
    size_t   GetLength() const NN_NOEXCEPT;

    /*!
      @brief RSSI (受信信号強度) の値を取得します。(無線ハードウェア評価用)
      @return RSSI の値。
     */
    int16_t  GetRssi() const NN_NOEXCEPT;

    /*!
      @brief LinkLevel (通信品質) の値を取得します。
      @return LinkLevel の値。
     */
    LinkLevel  GetLinkLevel() const NN_NOEXCEPT;

    /*!
      @brief 無線チャンネル情報を取得します。
      @return 無線チャンネルの値です。
     */
    int16_t GetChannel() const NN_NOEXCEPT;

    /*!
      @brief BSSID を取得します。
      @return BSSID が格納された Mac クラス。
     */
    MacAddress GetBssid() const NN_NOEXCEPT;

    //! Beacon 情報要素の数を取得します。(純粋仮想関数として定義されます。)
    virtual uint32_t GetIeCount() const NN_NOEXCEPT = 0;

    /*!
      @brief SSIDを取得します。Ssidクラスのコピーが渡されます。(長さ 0 の SSID が返されます。派生クラスにてそれぞれ SSID の取得コードが定義されます。)
      @return SSID が格納された Ssid クラス。
     */
    inline virtual Ssid  GetSsid() const NN_NOEXCEPT
    {
        return Ssid();
    }

    //! Beacon のベンダ情報要素を取得します。(純粋仮想関数として定義されます。)
    virtual const VendorInfoElementReader* GetVendorIeWithOui(const Bit8 pOui[MacAddress::OuiSize]) const NN_NOEXCEPT = 0;

    //! Beacon のベンダ情報要素を取得します。OUIの次の1バイトもチェックします。(純粋仮想関数として定義されます。)
    virtual const VendorInfoElementReader* GetVendorIeWithOuiAndType(const Bit8 pOui[MacAddress::OuiSize], Bit8 type) const NN_NOEXCEPT = 0;

protected:
    const uint8_t* m_pBuffer;
};

/*!
    :category   BSS情報解析
    @class BeaconDescriptionReader
    @brief Beacon Description  解析クラス

    Beacon および ProbeResponseの情報解析を行います。

    BssDescriptionのバッファを与えることにより、コンストラクタによる初期化を行います。

    初期化後、各メソッドを呼び出すことで、Beaconの各情報を取得することができます。

 */
class BeaconDescriptionReader : public BssDescriptionReaderBase
{
private:
    static const Bit8 WpaOui[MacAddress::OuiSize];
public:
    /*!
       @brief     BeaconDescriptionReader クラスのコンストラクタです。
       @param[in] pBuffer   解析するスキャンバッファのポインタ。
     */
    explicit BeaconDescriptionReader(const BssDescription* pBuffer) NN_NOEXCEPT;

    /*!
       @brief     BeaconDescriptionReader クラスのデストラクタです。
     */
    virtual ~BeaconDescriptionReader() NN_NOEXCEPT
    {}

    /*!
      @brief  Beacon 情報要素の数を取得します。
      @return Beacon 情報要素の数。
     */
    virtual uint32_t GetIeCount() const NN_NOEXCEPT;

    /*!
      @brief SSIDを取得します。Ssidクラスのコピーが渡されます。
      @return Beacon の SSID が格納された Ssid クラス。
     */
    virtual Ssid GetSsid() const NN_NOEXCEPT;

    /*!
      @brief Beacon Intervalを取得します。取得値の単位はTU(=1024μ秒)です。
      @return Beacon Interval。TU(=1024μ秒)単位。
     */
    uint16_t GetInterval() const NN_NOEXCEPT;

    /*!
      @brief Beacon の Capability Informationを取得します。
      @return Beacon の Capability Information。
     */
    Bit16 GetCapabilityInformation() const NN_NOEXCEPT;

    /*!
     * BSS情報内の特定IEを取得します。
     * @param elementId 取得するIEのElementID
     * @return 指定したElementIDのIE領域先頭へのポインタ。
     */
    virtual const Bit8* GetIe(Bit8 elementId) const NN_NOEXCEPT;

    /*!
      @brief Beacon のベンダ情報要素を取得します。OUI のみをマッチングします。
      @param[in] pOui 取得対象とするベンダ情報要素の OUI。
      @return Beacon のベンダ情報要素。@ref VendorInfoElementReader のポインタで返されます。
     */
    virtual const VendorInfoElementReader* GetVendorIeWithOui(const Bit8 pOui[MacAddress::OuiSize]) const NN_NOEXCEPT;

    /*!
      @brief Beacon のベンダ情報要素を取得します。OUI と Type をマッチングします。
      @param[in] pOui 取得対象とするベンダ情報要素の OUI。
      @param[in] type 取得対象とするベンダ情報要素の Type フィールド。(OUI の次の1バイト)
      @return Beacon のベンダ情報要素。@ref VendorInfoElementReader のポインタで返されます。
     */
    virtual const VendorInfoElementReader* GetVendorIeWithOuiAndType(const Bit8 pOui[MacAddress::OuiSize], Bit8 type) const NN_NOEXCEPT;

    /*!
      @brief Beacon のベンダ情報要素を取得します。OUI と その先に続く2バイトの値 をマッチングします。@n
              ローカル通信専用です。検索対象となる2バイトはProtocol IDという名称で扱われます。
      @param[in] pOui 取得対象とするベンダ情報要素の OUI。
      @param[in] protocolId 取得対象とするベンダ情報要素の Protocol ID。(OUI の次の2バイト)
      @return Beacon のベンダ情報要素。@ref VendorInfoElementReader のポインタで返されます。
     */
    virtual const VendorInfoElementReader* GetVendorIeWithOuiAndProtocolId(const Bit8 pOui[MacAddress::OuiSize], const Bit8 (&protocolId)[2]) const NN_NOEXCEPT;

    /*!
      @brief Beacon の WPA 情報要素を取得します。
      @return Beacon の WPA 情報要素。WpaInfoElementReader<WifiInfoElementReader> のポインタで返されます。
     */
    const WpaInfoElementReader<WifiInfoElementReader>* GetWpaIe() const NN_NOEXCEPT;

    /*!
      @brief WPA をサポートしているかを返します。
      @return WPA をサポートしているか否か。
     */
    inline bool  IsWpaSupported() const NN_NOEXCEPT
    {
        return GetWpaIe() != NULL ? true : false;
    }

    /*!
      @brief Beacon の WPA2 情報要素を取得します。
      @return Beacon の WPA2 情報要素。WpaInfoElementReader<InfoElementReader> のポインタで返されます。
     */
    const WpaInfoElementReader<InfoElementReader>* GetWpa2Ie() const NN_NOEXCEPT;

    /*!
      @brief WPA2 をサポートしているかを返します。
      @return WPA2 をサポートしているか否か。
     */
    inline bool  IsWpa2Supported() const NN_NOEXCEPT
    {
        return GetWpa2Ie() != NULL ? true : false;
    }

    /*!
      @brief Beacon の WPS 情報要素を取得します。
      @return Beacon の WPS 情報要素。@ref InfoElementReader のポインタで返されます。
     */
    const InfoElementReader* GetWpsIe() const NN_NOEXCEPT;

    /*!
      @brief WPS をサポートしているかを返します。
      @return WPS をサポートしているか否か。
     */
    inline bool  IsWpsSupported() const NN_NOEXCEPT
    {
        return GetWpsIe() != NULL ? true : false;
    }

};

/*!
    @}
 */

} // end of namespace wlan
} // end of namespace nn
