﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief Cruiser起動引数が共通で利用する API (非公開)
 */

#pragma once

#include <nn/nn_SdkAssert.h>
#include <nn/nn_Result.h>
#include <nn/applet/applet.h>
#include <nn/applet/applet_Types.h>
#include <nn/la/la_Api.h>
#include <nn/web/common/web_CommonArgData.h>
#include <nn/web/common/web_CommonTypes.h>

namespace nn { namespace web { namespace common {

class CommonApi
{
public:
    //------------------------------------------------------------------------
    /**
     * @brief 入力チャンネルに指定したデータを持つストレージをプッシュします。
     *
     * @param[in]   libraryAppletHandle プッシュ対象のライブラリアプレットのハンドルを指定してください。
     * @param[in]   inParamBuf          ストレージに格納するデータに書き込むバッファを指定してください。
     * @param[in]   inParamSize         ストレージに格納するデータに書き込むバッファのサイズを指定してください。
     *
     */
    inline static void PushStorageToInChannel(const nn::applet::LibraryAppletHandle& libraryAppletHandle,
        const void* inParamBuf, size_t inParamSize) NN_NOEXCEPT;

    //------------------------------------------------------------------------
    /**
     * @brief 指定したストレージからデータを読み込みます。
     *
     * @param[out]  outParamBuf     データの格納先を指定してください。
     * @param[in]   outParamSize    データの格納先のサイズを指定してください。
     * @param[in]   storageHandle   対象となるストレージのハンドルを指定してください。
     *
     * @details 指定したストレージは破棄されません。
     */
    inline static void ReadFromStorage(void* outParamBuf, size_t outParamSize,
        const nn::applet::StorageHandle& storageHandle) NN_NOEXCEPT;

    //------------------------------------------------------------------------
    /**
     * @brief ライブラリアプレットの起動準備をします。
     *
     * @param[in]   appletId    起動予定のライブラリアプレットのIDを指定してください。
     *
     * @return 生々したライブラリアプレットハンドルを返します。
     */
    inline static nn::applet::LibraryAppletHandle PrepareLibraryApplet(nn::applet::AppletId appletId) NN_NOEXCEPT;

    //------------------------------------------------------------------------
    /**
     * @brief ライブラリアプレットを起動します。
     *
     * @param[in]   handle          ライブラリアプレットハンドルを指定してください。
     *
     * @details ライブラリアプレットが終了する迄待機しません。
     *          ライブラリアプレットを終了させるには JoinAndFinishLibraryApplet を呼ぶ必要があります。
     */
    inline static void StartLibraryApplet(nn::applet::LibraryAppletHandle& handle) NN_NOEXCEPT;

    //------------------------------------------------------------------------
    /**
     * @brief ライブラリアプレットを起動し終了迄待機します。
     *
     * @param[out]  outParamBuf     戻り値の格納先バッファを指定してください。
     * @param[in]   outParamSize    戻り値の格納先バッファのサイズを指定してください。
     * @param[in]   handle          ライブラリアプレットハンドルを指定してください。
     *
     * @return
     *
     * @details 正常終了時はOutChannelから outParamBufに値がPopされます。
     *          ライブラリアプレットが終了する迄処理をブロックします。
     */
    inline static nn::Result StartLibraryAppletAndWait(void* outParamBuf, size_t outParamSize, nn::applet::LibraryAppletHandle& handle) NN_NOEXCEPT;

    //------------------------------------------------------------------------
    /**
     * @brief ライブラリアプレットが終了する迄待機し結果を取得します。
     *
     * @param[out]  outParamBuf     戻り値の格納先バッファを指定してください。
     * @param[in]   outParamSize    戻り値の格納先バッファのサイズを指定してください。
     * @param[in]   handle          ライブラリアプレットハンドルを指定してください。
     *
     * @details 正常終了時はOutChannelから outParamBufに値がPopされます。
     *          終了する迄処理をブロックします。
     */
    inline static nn::Result JoinAndFinishLibraryApplet(void* outParamBuf, size_t outParamSize, nn::applet::LibraryAppletHandle& handle) NN_NOEXCEPT;

    //------------------------------------------------------------------------
    /**
     * @brief   対象とするフッターボタンが起動時に表示されるかどうかを設定します。
     *
     * @param[in]   buttonId     対象のフッターボタンの ID を指定します。
     * @param[in]   visible      対象のフッターボタンを起動時に表示するかどうかを指定します。
     *
     * @details デフォルト値は A ボタン、 B ボタンが true で、それ以外は false です。
     *          同じフッターボタンに対して二回呼び出した場合、そのフッターボタンに対する設定は上書きされます。
     */
    inline static void SetBootFooterButtonVisible(uint8_t* pDst, const FooterButtonId buttonId, const bool visible) NN_NOEXCEPT;

private:
    // LibraryAppletExitReason_Abnormal なら abort します。
    inline static void AbortIfAbnormal(nn::applet::LibraryAppletExitReason reason) NN_NOEXCEPT;
    // LibraryAppletExitReason_Unexpected なら abort します。
    inline static void AbortIfUnexpected(nn::applet::LibraryAppletExitReason reason) NN_NOEXCEPT;
    // LibarryAppletExitReason_Abnormal or LibraryAppletExitReason_Unexpected なら abort します。
    inline static void AbortIfAbnormalOrUnexpected(nn::applet::LibraryAppletExitReason reason) NN_NOEXCEPT;

};

//------------------------------------------------------------------------
void CommonApi::PushStorageToInChannel(const nn::applet::LibraryAppletHandle& libraryAppletHandle,
    const void* inParamBuf, size_t inParamSize) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
    NN_SDK_ASSERT(libraryAppletHandle != nn::applet::InvalidLibraryAppletHandle);
    nn::applet::StorageHandle storageHandle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::CreateStorage(&storageHandle, inParamSize));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::WriteToStorage(storageHandle, 0, inParamBuf, inParamSize));
    // push
    nn::applet::PushToInChannel(libraryAppletHandle, storageHandle);
#else
    NN_UNUSED(libraryAppletHandle);
    NN_UNUSED(inParamBuf);
    NN_UNUSED(inParamSize);
#endif
}

//------------------------------------------------------------------------
void CommonApi::ReadFromStorage(void* outParamBuf, size_t outParamSize,
const nn::applet::StorageHandle& storageHandle) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
    NN_SDK_ASSERT(storageHandle != nn::applet::InvalidStorageHandle);
    {
        auto storageSize = nn::applet::GetStorageSize(storageHandle);
        NN_ABORT_UNLESS(storageSize == outParamSize);

        // データ読み込み
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::applet::ReadFromStorage(storageHandle, 0, outParamBuf, outParamSize));
    }
#else
    NN_UNUSED(outParamBuf);
    NN_UNUSED(outParamSize);
    NN_UNUSED(storageHandle);
#endif
}

//------------------------------------------------------------------------
nn::applet::LibraryAppletHandle CommonApi::PrepareLibraryApplet(nn::applet::AppletId appletId) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
    nn::applet::LibraryAppletHandle handle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::CreateLibraryApplet(
        &handle,
        appletId,
        nn::applet::LibraryAppletMode_AllForeground));
    return handle;
#else
    NN_UNUSED(appletId);
    nn::applet::LibraryAppletHandle handle;
    return handle;
#endif
}

//------------------------------------------------------------------------
void CommonApi::StartLibraryApplet(nn::applet::LibraryAppletHandle& handle) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
    NN_SDK_ASSERT(handle != nn::applet::InvalidLibraryAppletHandle);

    // 非末端アプレット設定
    nn::la::LibraryAppletStartHookUserArg callbackArg;
    callbackArg.isExtremity = false;

    // ライブラリアプレットの起動
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::StartLibraryApplet(handle, &callbackArg));
#else
    NN_UNUSED(handle);
#endif
}

//------------------------------------------------------------------------
nn::Result CommonApi::StartLibraryAppletAndWait(void* outParamBuf, size_t outParamSize, nn::applet::LibraryAppletHandle& handle) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
    StartLibraryApplet(handle);
    return JoinAndFinishLibraryApplet(outParamBuf, outParamSize, handle);
#else
    NN_UNUSED(outParamBuf);
    NN_UNUSED(outParamSize);
    NN_UNUSED(handle);
    return nn::Result();
#endif
}

//------------------------------------------------------------------------
nn::Result CommonApi::JoinAndFinishLibraryApplet(void* outParamBuf, size_t outParamSize, nn::applet::LibraryAppletHandle& handle) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)

    nn::Result result = nn::ResultSuccess();

    // ライブラリアプレットが終了するまで待機し、終了結果を取得します。
    nn::applet::JoinLibraryApplet(handle);
    nn::applet::LibraryAppletExitReason exitReason = nn::applet::GetLibraryAppletExitReason(handle);

    // 異常終了チェック
    AbortIfAbnormalOrUnexpected(exitReason);

    // キャンセルされた
    if (exitReason == nn::applet::LibraryAppletExitReason_Canceled) {
        result = nn::la::ResultLibraryAppletCanceled();
    }

    // 呼び出しに成功してたらOutChannelから戻り値を取得します。
    if (result.IsSuccess()) {
        size_t readSize = 0;
        nn::la::PopFromOutChannel(handle, outParamBuf, outParamSize, &readSize);
        NN_ABORT_UNLESS(outParamSize == readSize);
    }

    // ハンドルを解放します。
    nn::applet::CloseLibraryApplet(handle);

    return result;
#else
    NN_UNUSED(outParamBuf);
    NN_UNUSED(outParamSize);
    NN_UNUSED(handle);
#endif

}

//------------------------------------------------------------------------
void CommonApi::SetBootFooterButtonVisible(uint8_t* pDst, const FooterButtonId buttonId, const bool visible) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
    // 無効なフッターボタンIDを指定していた場合、何もしない
    if (buttonId == FooterButtonId::None || buttonId == FooterButtonId::Max)
    {
        return;
    }

    // すでにデータがセットされている場合、データを取得
    auto preSettingList = reinterpret_cast<const BootFooterButtonVisibleSettingList*>(
        CommonArgData::GetData(pDst, CommonArgData::ArgKind::BootFooterButtonVisibleSettingList).pBuffer);

    BootFooterButtonVisibleSettingList newSettingList = {};

    // すでにデータがセットされていたら、指定のフッターボタンに対してすでに設定されているかどうかチェック
    if (preSettingList)
    {
        newSettingList = *preSettingList;

        // すでに対象のフッターボタンに対して設定されていた場合、上書きして設定
        for (int i = 0; i < BootFooterButtonVisibleSettingListSize; ++i)
        {
            auto preSetting = preSettingList->settings[i];
            if (preSetting.footerButtonId == buttonId)
            {
                preSetting.visible = visible;
                newSettingList.settings[i] = preSetting;
                CommonArgData::SetData(
                    pDst, CommonArgData::ArgKind::BootFooterButtonVisibleSettingList, newSettingList);
                return;
            }
        }

        // 対象のフッターボタンに対する設定がされていなかった場合、配列の空いている要素を探して設定
        for (int i = 0; i < BootFooterButtonVisibleSettingListSize; ++i)
        {
            auto preSetting = preSettingList->settings[i];
            if (preSetting.footerButtonId == FooterButtonId::None)
            {
                preSetting.footerButtonId = buttonId;
                preSetting.visible = visible;
                newSettingList.settings[i] = preSetting;
                CommonArgData::SetData(
                    pDst, CommonArgData::ArgKind::BootFooterButtonVisibleSettingList, newSettingList);
                return;
            }
        }
        NN_ABORT_UNLESS(false);
    }
    // データがセットされていなかったら、配列の先頭に保存
    else
    {
        newSettingList.settings[0] = { buttonId, visible };
        CommonArgData::SetData(
            pDst, CommonArgData::ArgKind::BootFooterButtonVisibleSettingList, newSettingList);
    }
#else
    NN_UNUSED(pDst);
    NN_UNUSED(buttonId);
    NN_UNUSED(visible);
#endif
}

//------------------------------------------------------------------------
void CommonApi::AbortIfAbnormal(nn::applet::LibraryAppletExitReason reason) NN_NOEXCEPT
{
    if (reason == nn::applet::LibraryAppletExitReason_Abnormal) {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::la::ResultLibraryAppletExitedAbnormally());
    }
}

//------------------------------------------------------------------------
void CommonApi::AbortIfUnexpected(nn::applet::LibraryAppletExitReason reason) NN_NOEXCEPT
{
    if (reason == nn::applet::LibraryAppletExitReason_Unexpected) {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::la::ResultLibraryAppletExitedUnexpectedly());
    }
}

//------------------------------------------------------------------------
void CommonApi::AbortIfAbnormalOrUnexpected(nn::applet::LibraryAppletExitReason reason) NN_NOEXCEPT
{
    // @note (nn::la::la_Api)
    //   SAがLAを強制終了させたとき LibraryAppletExitReason_Canceledになります。
    //   それは許容する必要があります。
    //   異常終了のときはABORTでいいはずです。
    //   ポリシー：http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=164702960
    AbortIfAbnormal(reason);

    // Unexpectedはデバッグで撲滅する必要があります。
    AbortIfUnexpected(reason);
}

}}} // namespace nn::web::common
