﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief Layer manipulation functions for the VI library.
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/vi/vi_Types.h>
#include <nn/vi/vi_PixelFormat.h>
#include <nn/vi/vi_ScalingMode.h>

namespace nn { namespace vi {

    /**
     * @brief Creates a new layer on a particular display.
     *
     * @param[out] pOutLayer Layer handle.
     * @param[in]  pDisplay  Display handle.
     *
     * @return                          Whether the layer was created.
     * @retval ResultSuccess            Layer created successfully.
     * @retval ResultOperationFailed    An error occurred.
     *
     * @pre  VI library is initialized. @n
     *       @a pOutLayer != nullptr @n
     *       @a pDisplay != nullptr and @a pDisplay is a valid handle.
     * @post A valid handle is written to @a pOutLayer.
     *
     * @details The layer is visible by default.  Until a frame is submitted by the process, the layer will
     *          be blank.
     *
     *          The textures associated with this layer have restrictions due to a limitation with downscaling: @n
     *          @a width  <= 1920 @n
     *          @a height <= 1080 @n
     *          This restriction may be lifted in the future.
     *
     *          A particular display may only support a limited number of layers system-wide.  If a process exceeds
     *          its quota then this function will return ResultOperationFailed.  At minimum, a process can use a
     *          single layer to display content.  To query how many layers can be created by the process, use ListDisplays().
     *
     *          If this API is used with a graphics API that does not have an explicit swap chain, like EGL,
     *          then the rendering dimensions will use default values of 1280x720.
     */
    Result CreateLayer(Layer** pOutLayer, Display* pDisplay) NN_NOEXCEPT;

    /**
     * @brief Creates a new layer on a particular display.
     *
     * @param[out] pOutLayer Layer handle.
     * @param[in]  pDisplay  Display handle.
     * @param[in]  width     The default width of the layer in pixels.
     * @param[in]  height    The default height of the layer in pixels.
     *
     * @return                          Whether the layer was created.
     * @retval ResultSuccess            Layer created successfully.
     * @retval ResultInvalidDimensions  The valid range is @a width > 0 and @a height > 0.
     * @retval ResultOperationFailed    An unknown error occurred.
     *
     * @pre  VI library is initialized. @n
     *       @a pOutLayer != nullptr @n
     *       @a pDisplay != nullptr and @a pDisplay is a valid handle.
     * @post A valid handle is written to @a pOutLayer.
     *
     * @details The layer is visible by default.  Until a frame is submitted by the process, the layer will
     *          be blank.
     *
     *          The textures associated with this layer have restrictions due to a limitation with downscaling: @n
     *          @a width  <= 1920 @n
     *          @a height <= 1080 @n
     *          This restriction may be lifted in the future.
     *
     *          A particular display may only support a limited number of layers system-wide.  If a process exceeds
     *          its quota then this function will return ResultOperationFailed.  At minimum, a process can use a
     *          single layer to display content.  To query how many layers can be created by the process, use ListDisplays().
     *
     *          The @a width and @a height arguments specify the default width and height of the layer in pixels.
     *          For graphics APIs that do not have an explicit swap chain, like EGL, these dimensions will be used
     *          as the rendering dimensions.  The arguments will not affect the active display area.
     */
    Result CreateLayer(Layer** pOutLayer, Display* pDisplay, int width, int height) NN_NOEXCEPT;

    /**
     * @brief Destroys the layer and disables content.
     *
     * @param[in] pLayer Layer handle.
     *
     * @pre  VI library is initialized. @n
     *       @a pLayer != nullptr and @a pLayer is a valid handle.
     * @post @a pLayer invalidated.
     *
     * @details No content from @a pLayer will be seen on the display after this call.
     *          The handle is invalidated and should not be used.
     */
    void DestroyLayer(Layer* pLayer) NN_NOEXCEPT;

    /**
     * @brief Changes how content is scaled when the framebuffer dimensions do not match
     *        the layer dimensions.
     *
     * @param[in] pLayer Layer handle.
     * @param[in] mode   The new behavior to use for scaling.
     *
     * @return                       Whether the scaling behavior was changed.
     * @retval ResultSuccess         Scaling behavior defined by @a mode has taken effect.
     * @retval ResultNotSupported    The display cannot support this scaling mode.
     * @retval ResultOperationFailed An unknown error occurred.
     *
     * @pre  VI library is initialized. @n
     *       @a pLayer != nullptr and @a pLayer is a valid handle.
     * @post The requested scaling mode is now in effect.
     */
    Result SetLayerScalingMode(Layer* pLayer, ScalingMode mode) NN_NOEXCEPT;

    /**
     * @brief Retrieves a handle to the platform's native window type.
     *
     * @param[out] pOutWindow Native window handle for use with nvn::WindowBuilder.
     * @param[in]  pLayer     Layer handle.
     *
     * @return                       Whether a handle was written to @a pOutWindow.
     * @retval ResultSuccess         @a pOutWindow contains the native window handle.
     * @retval ResultOperationFailed An unknown error occurred.
     *
     * @pre  VI library is initialized. @n
     *       @a pLayer != nullptr and @a pLayer is a valid handle. @n
     *       @a pOutWindow != nullptr
     * @post On nn::ResultSuccess, the window handle is written to @a pOutWindow.
     *
     * @details This handle may be safely used by other APIs.  Its exact type will depend
     *          on the platform.
     */
    Result GetNativeWindow(NativeWindowHandle* pOutWindow, Layer* pLayer) NN_NOEXCEPT;

}}
