﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/vi/vi_Types.h>

namespace nn{ namespace vi{

    typedef uint64_t IndirectProducerHandleType;
    class IndirectLayer;

    //! @brief VI ライブラリを IndirectProducer 向けに初期化します。
    nn::Result InitializeForIndirectProducer() NN_NOEXCEPT;

    //! @brief VI ライブラリを終了します。
    void FinalizeForIndirectProducer() NN_NOEXCEPT;

    //! @brief 指定したハンドルの IndirectLayer を開きます。
    nn::Result OpenIndirectLayer(IndirectLayer** pOutLayer, IndirectProducerHandleType handle) NN_NOEXCEPT;

    //! @brief IndirectLayer を閉じます。
    nn::Result CloaseIndirectLayer(IndirectLayer* pLayer) NN_NOEXCEPT;

    //! @brief IndirectLayer の NativeHandle を取得します。
    nn::Result GetIndirectLayerNativeWindow(NativeWindowHandle* pOutNativeWindow, IndirectLayer* pLayer) NN_NOEXCEPT;

    //! @brief IndirectLayer のフレームバッファをフリップします。
    //! @retval nn::vi::ResultDenied 指定された IndirectLayer がグラフィクス API によって使用されていません。
    //! @retval nn::vi::ResultBroken 指定された IndirectLayer の状態が不正です。
    //! @retval nn::vi::ResultNotReady フリップ対象のバッファがなかったか、バッファへの描画が完了していませんでした。
    //! @details
    //!   通常のディスプレイと異なり、 IndirectLayer は Vsync によるフレームのフリップが発生しません。
    //!   グラフィクス API からフレームバッファを提出した後、この関数を呼び出すことで新しいバッファが IndirectConsumer から取得されるようになります。
    //!   これにより以前のフレームバッファ押し出された場合、IndirectDisplay からアプレットに返却され、グラフィクス API からフレームバッファを取得できるようになります。
    //!
    //!   この関数の呼び出しによって IndirectConsumer から取得されるバッファが変化した場合に成功を返します。
    //!   この関数が呼び出された時点でフレームバッファが IndirectLayer に提出されていなかったり、
    //!   提出されたフレームバッファへの GPU による書き込みが完了していなかった場合、ResultNotReady を返してこの関数は失敗します。
    //!
    //!   通常、この関数はグラフィクスループとは独立したスレッドから繰り返し呼び出して使用します。
    nn::Result FlipIndirectLayer(IndirectLayer* pLayer) NN_NOEXCEPT;


}}
