﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/os.h>

namespace nn{ namespace vi{

    typedef uint64_t IndirectConsumerHandleType;

    //! @brief IndirectImage を取得するためのバッファに対するサイズとアライメントの制約を取得します。
    //! @param[out] pOutRequiredSize      必要なバッファの大きさ（バイト）を受け取る変数のポインタ
    //! @param[out] pOutRequiredAlignment 必要なバッファのアライメント（バイト）を受け取る変数のポインタ
    //! @param[in] width  取得する画像の幅（ピクセル）
    //! @param[in] height 取得する画像の高さ（ピクセル）
    //! @pre VI ライブラリが初期化済
    //! @pre pOutRequiredSize != nullptr
    //! @pre pOutRequiredAlignment != nullptr
    //! @pre width > 0
    //! @pre height > 0
    //! @details
    //!   取得されるバッファのサイズはハードウェアの制約により GetIndirectImage() で取得される画像のサイズよりも大きい場合があります。
    nn::Result GetIndirectImageMemoryRequirement(
        size_t* pOutRequiredSize,
        size_t* pOutRequiredAlignment,
        int width,
        int height
        ) NN_NOEXCEPT;

    //! @brief IndirectConsumer から画像を取得します。
    //! @param[out] pOutSize   取得した画像のサイズ（バイト）を受け取る変数のポインタ
    //! @param[out] pOutStride 取得した画像の 1 ラインのサイズ（バイト）を受け取る変数のポインタ
    //! @param[in] pBuffer     画像を取得するバッファのポインタ
    //! @param[in] bufferSize  画像を取得するバッファの大きさ（バイト）
    //! @param[in] handle      画像を取得する IndirectConsumer のハンドル
    //! @param[in] width       取得する画像の幅（ピクセル）
    //! @param[in] height      取得する画像の高さ（ピクセル）
    //! @pre VI ライブラリが初期化済
    //! @pre pOutSize != nullptr
    //! @pre pOutStride != nullptr
    //! @pre pBuffer != nullptr
    //! @pre bufferSize > 0
    //! @pre handle > 0
    //! @pre width > 0
    //! @pre height > 0
    //! @retval nn::vi::ResultNotReady 取得する画像が存在しませんでした。
    //! @details
    //!   IndirectConsumer から画像を取得します。
    //!   取得される画像データは左上を原点とした各チャンネル 8 ビットの sRGBA 画像です。
    //!
    //!   対応する IndirectProducer が生成した画像を指定した幅と高さにスケーリングして取得します。
    //!   IndirectProducer の生成する画像サイズを取得する API を nn::vi は提供していませんので、
    //!   IndirectProducer となる各アプレットの仕様としてユーザーが取得できるようにする必要があります。
    nn::Result GetIndirectImage(
        size_t* pOutSize,
        size_t* pOutStride,
        void* pBuffer,
        size_t bufferSize,
        IndirectConsumerHandleType handle,
        int width,
        int height
        ) NN_NOEXCEPT;

    //! @brief IndirectConsumer から画像を取得します。
    //! @param[out] pOutSize   取得した画像のサイズ（バイト）を受け取る変数のポインタ
    //! @param[out] pOutStride 取得した画像の 1 ラインのサイズ（バイト）を受け取る変数のポインタ
    //! @param[in] pBuffer     画像を取得するバッファのポインタ
    //! @param[in] bufferSize  画像を取得するバッファの大きさ（バイト）
    //! @param[in] handle      画像を取得する IndirectConsumer のハンドル
    //! @param[in] width       取得する画像の幅（ピクセル）
    //! @param[in] height      取得する画像の高さ（ピクセル）
    //! @pre VI ライブラリが初期化済
    //! @pre pOutSize != nullptr
    //! @pre pOutStride != nullptr
    //! @pre pBuffer != nullptr
    //! @pre bufferSize > 0
    //! @pre handle > 0
    //! @pre width > 0
    //! @pre height > 0
    //! @pre sourceRectX >= 0
    //! @pre sourceRectY >= 0
    //! @pre sourceRectWidth >= 0
    //! @pre sourceRectHeight >= 0
    //! @pre sourceRectX + sourceRectWidth <= 1
    //! @pre sourceRectY + sourceRectHeight <= 1
    //! @retval nn::vi::ResultNotReady 取得する画像が存在しませんでした。
    //! @details
    //!   IndirectConsumer から画像を取得します。
    //!   取得される画像データは左上を原点とした各チャンネル 8 ビットの sRGBA 画像です。
    //!
    //!   対応する IndirectProducer が生成した画像を指定した幅と高さにスケーリングして取得します。
    //!   IndirectProducer の生成する画像サイズを取得する API を nn::vi は提供していませんので、
    //!   IndirectProducer となる各アプレットの仕様としてユーザーが取得できるようにする必要があります。
    //!
    //!   sourceRectX, sourceRectY, sourceRectWidth, sourceRectHeight によりコピー元画像からコピーする範囲を指定することができます。
    //!   これらの値はコピー元画像の左上を原点として画像サイズを幅 1, 高さ 1 に正規化した座標系における矩形範囲として指定します。
    //!   例えばコピー元画像の下半分を取得したい場合、 (X=0, Y=0.5, Width=1, Height=0.5) を指定します。
    //!   指定したコピー元画像の範囲を OpenIndirectImage() で指定した width, height の範囲にスケーリングしてコピーします。
    nn::Result GetIndirectImage(
        size_t* pOutSize,
        size_t* pOutStride,
        void* pBuffer,
        size_t bufferSize,
        IndirectConsumerHandleType handle,
        int width,
        int height,
        float sourceRectX,
        float sourceRectY,
        float sourceRectWidth,
        float sourceRectHeight
        ) NN_NOEXCEPT;

}}
