﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   System events relating to displays.
 * @details The functions described in this file are for internal developers only
 *          and not to be released to third parties.
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/os/os_SystemEventTypes.h>
#include <nn/vi/vi_Types.h>
#include <nn/vi/vi_HotplugState.h>

namespace nn { namespace vi {

    /**
     * @brief Initializes an event that will signal on hotplug state changes.
     *
     * @param[out] pOutEvent Event to initialize.
     * @param[in]  pDisplay  Display that will trigger hotplug events.
     *
     * @return                       Whether the event was initialized successfully.
     * @retval ResultSuccess         The event was initialized successfully.
     * @retval ResultNotSupported    @a pDisplay cannot support these notifications.
     * @retval ResultOperationFailed An unknown error occurred.
     *
     * @pre
     *      - VI library is initialized.
     *      - @a pOutEvent != nullptr
     *      - @a pDisplay != nullptr and @a pDisplay is a valid handle.
     * @post @a pOutEvent will signal on @a pDisplay hotplug events.
     *
     * @details @a pOutEvent will autoclear.  Only one thread will wake up when the event is signalled.
     *
     *          Clients are expected to use GetDisplayHotplugState() to query the physical connection status.
     */
    nn::Result GetDisplayHotplugEvent(nn::os::SystemEventType* pOutEvent, Display* pDisplay) NN_NOEXCEPT;

    /**
    * @brief Retrieves the current hotplug state of the display.
    *
    * @param[out] pOutState The current hotplug state.
    * @param[in]  pDisplay  Display handle.
    *
    * @return                       Whether the current hotplug state of @a pDisplay was written
    *                               to @a pOutState.
    * @retval ResultSuccess         The hotplug state was written to @a pOutState.
    * @retval ResultNotSupported    The display does not have any hotpluggable connections.
    * @retval ResultOperationFailed An unknown error occurred.
    *
    * @pre  VI library is initialized. @n
    *       @a pOutState != nullptr @n
    *       @a pDisplay != nullptr and @a pDisplay is a valid handle.
    * @post The current hotplug state of @a pDisplay matches that of @a pOutState.
    *
    * @details This function only describes whether the cable is connected, not the current
    *          output path.
    */
    nn::Result GetDisplayHotplugState(HotplugState* pOutState, const Display* pDisplay) NN_NOEXCEPT;

    /**
     * @brief Initializes an event that will signal on display mode changes for a particular display.
     *
     * @param[out] pOutEvent Event to initialize.
     * @param[in]  pDisplay  Display that will trigger mode change events.
     *
     * @return                       Whether the event was initialized successfully.
     * @retval ResultSuccess         The event was initialized successfully.
     * @retval ResultNotSupoprted    @a pDisplay cannot support these notifications.
     * @retval ResultOperationFailed An unknown error occurred.
     *
     * @pre
     *      - VI library is initialized.
     *      - @a pOutEvent != nullptr
     *      - @a pDisplay != nullptr and @a pDisplay is a valid handle.
     * @post @a pOutEvent will signal on @a pDisplay mode change events.
     *
     * @details @a pOutEvent will autoclear.  Only one thread will wake up when the event is signalled.
     *
     *          Clients are expected to use GetDisplayMode() to query the current display mode.
     *
     *          This event will not signal on power or hotplug state changes.
     */
    nn::Result GetDisplayModeChangedEvent(nn::os::SystemEventType* pOutEvent, Display* pDisplay) NN_NOEXCEPT;

}}
