﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   CMU settings for the VI library.
 * @details These functions are for internal SDK developers only and not to be released in any package.
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>

namespace nn { namespace vi {

    class Display;

    /**
     * @brief Sets a luma coefficient to use in the CMU.
     *
     * @param[in] pDisplay Display handle.
     * @param[in] luma     New luma coefficient to use.
     *
     * @return                       Whether the luma coefficient used for @a pDisplay matches that of @a luma.
     * @retval ResultSuccess         The display's luma setting matches @a luma.
     * @retval ResultInvalidRange    The valid range is -1.f <= @a luma <= 1.f.
     * @retval ResultNotSupported    The display does not support this setting.
     * @retval ResultOperationFailed An unknown error occurred.
     *
     * @pre
     *      - VI library is initialized.
     *      - @a pDisplay != nullptr and @a pDisplay is a valid handle.
     * @post The current luma setting of @a pDisplay matches that of @a luma.
     *
     * @details This adjusts a coefficient for use in the CMU and is always applied regardless
     *          of CMU mode.  The only exception is if the CMU is disabled.
     *
     *          A value of 0.f for @a luma is the identity.  Values in the range of [-1.f, 0)
     *          will decrease the luminance of the display.  Similarly, the range (0, 1.f] will
     *          increase the luminance.
     */
    Result SetDisplayCmuLuma(Display* pDisplay, float luma) NN_NOEXCEPT;

    /**
     * @brief Retrieves the display's current CMU luma coefficient.
     *
     * @param[out] pOutLuma Display's luma setting.
     * @param[in]  pDisplay Display handle.
     *
     * @return                       Whether the luma settings used for @a pDisplay was retrieved successfully.
     * @retval ResultSuccess         Successfully retrieved the display's luma setting.
     * @retval ResultNotSupported    The display does not support this setting.
     * @retval ResultOperationFailed An unknown error occurred.
     *
     * @pre
     *      - VI library is initialized.
     *      - @a pOutLuma != nullptr
     *      - @a pDisplay != nullptr and @a pDisplay is a valid handle.
     * @post The current luma setting used by @a pDisplay is written to @a pOutLuma.
     *
     * @details A value of 0.f for @a luma is the identity.  Values in the range of [-1.f, 0)
     *          will decrease the luminance of the display.  Similarly, the range (0, 1.f] will
     *          increase the luminance.
     */
    Result GetDisplayCmuLuma(float* pOutLuma, const Display* pDisplay) NN_NOEXCEPT;

}}
