﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/vi/sf/vi_ServiceTypes.h>
#include <nn/vi/manager/vi_Manager.h>

namespace nn{ namespace vi{ namespace manager{

    struct ManagerFrameBufferData;

    // @brief ManagerLayer に出画を行うためのフレームバッファを管理するクラスです。
    // @details
    //   libandroid の API をラップしています。
    class ManagerFrameBuffer
    {
    public:
        // @brief デフォルトコンストラクタ
        // @post オブジェクトは未初期化
        // @details
        //   未初期化状態のオブジェクトを作ります。
        //   Initialize() を呼び出して初期化してください。
        ManagerFrameBuffer() NN_NOEXCEPT;

        // @brief デストラクタ
        // @pre オブジェクトは未初期化
        // @details
        //   オブジェクトを破棄します。
        //   先に Finalize() を呼び出して未初期化にしておく必要があります。
        ~ManagerFrameBuffer() NN_NOEXCEPT;

        // @brief フレームバッファを指定した幅、高さ、フォーマットで初期化します。
        // @param[in] width  幅(px)
        // @param[in] height 高さ(px)
        // @param[in] format フォーマット。 nn::vi::PixelFormat_Rgba8888 を指定してください。
        // @pre オブジェクトは未初期化
        // @pre format == nn::vi::PixelFormat_Rgba8888
        // @post オブジェクトは初期化済
        // @details
        //   フレームバッファに必要なメモリは nv::SetGraphicsAllocator() で設定されたアロケータから確保されます。
        nn::Result Initialize(DisplayManagerService* pService, int64_t width, int64_t height, nn::vi::PixelFormatType format) NN_NOEXCEPT;

        // @brief フレームバッファを破棄します。
        // @pre オブジェクトは初期化済
        // @post オブジェクトは未初期化
        void Finalize() NN_NOEXCEPT;

        // @brief フレームバッファにメモリの内容をコピーします。
        // @param[in] pBuffer    コピーするバッファ
        // @param[in] bufferSize pBuffer の大きさ（バイト）
        // @details
        //   指定されたメモリの内容をフレームバッファにコピーします。
        //   入力のバッファはパディングなしの PitchLinear のデータとして解釈されます。
        nn::Result CopyFromMemory(const void* pBuffer, size_t bufferSize) NN_NOEXCEPT;

        nn::Result CopyFromMemory(int x, int y, int width, int height, const void* pBuffer, size_t bufferSize) NN_NOEXCEPT;

        // @brief フレームバッファを塗り潰します。
        // @param[in] clearColor 塗り潰すカラー値
        // @details
        //   フレームバッファを指定された clearColor で塗り潰します。
        nn::Result Clear(Bit32 clearColor) NN_NOEXCEPT;

        ManagerFrameBufferData* GetData() NN_NOEXCEPT
        {
            return reinterpret_cast<ManagerFrameBufferData*>(&m_Storage[0]);
        }
    private:
        bool m_IsInitialized;
        NN_ALIGNAS(16) char m_Storage[256]; // 大きさは適当。
    };


}}}
