﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/vi/vi_Types.h>
#include <nn/vi/vi_PixelFormat.h>

namespace nn { namespace vi { namespace buffer{
    //For VI direct rendering

    static const int MaxBuffers = 3;

    typedef void* BufferQueueHandle;

    struct BufferResource;

    /**
     * @brief Information about a scan buffer to attach to layer.
     */
    struct BufferInfo
    {
        int width;             //!< width in pixels
        int height;            //!< height in pixels
        int bufferCount;       //!< buffer number to attach to layer
        PixelFormat format;    //!< pixel format
    };

    class BufferQueue
    {
        public:
            /**
             * @brief Default constructor.
             *
             * @post Object has not initialized yet.
             *
             * @details
             *    Create uninitialized object
             *    Call Initialize() to initialize the Object.
             */
            BufferQueue() NN_NOEXCEPT;

            /**
             * @brief Destructor.
             *
             * @pre object is not initialized
             *
             * @details
             *    tear down object
             *    Call Finalize() in advance to un-initialize the Object.
             */
            ~BufferQueue() NN_NOEXCEPT;

            /**
             * @brief Initialize object.
             *
             * @pre object is not initialized
             * @post object is initialized
             *
             * @param[in] pLayer  layer handle
             * @param[in] info    buffer information(width, height, pix format)
             *
             * @details
             *     Open Internal resources (gralloc, NvRM etc)
             */
            nn::Result Initialize(Layer* pLayer, const BufferInfo& info) NN_NOEXCEPT;

            /**
             * @brief Finalize object.
             *
             * @pre object is initialized
             * @post object is not initialized
             *
             * @details
             *     Close Internal resources (gralloc, NvRM etc)
             */
            void Finalize() NN_NOEXCEPT;

            /**
             * @brief Return required memory size for specified buffer format.
             *
             * @param[in] info buffer information(width, height, pix format)
             *
             * @pre object is initialized
             *
             * @return size of memory to be allocated
             */
            size_t GetRequiredMemorySize(const BufferInfo& info) NN_NOEXCEPT;

            /**
             * @brief Return alignment for specified buffer format.
             *
             * @param[in] info buffer information(width, height, pix format)
             *
             * @pre object is initialized
             *
             * @return size of alignment
             */
            size_t GetRequiredMemoryAlignment(const BufferInfo& info) NN_NOEXCEPT;

            /**
             * @brief Return stride for specified buffer format.
             *
             * @param[in] info buffer information(width, height, pix format)
             *
             * @pre object is initialized
             *
             * @return size of stride
             */
            size_t GetStride(const BufferInfo& info) NN_NOEXCEPT;

            /**
             * @brief Register client buffer as scan buffer.
             *
             * @param[in] pLayer   layer handle
             * @param[in] pBuffer  buffer data
             *
             * @pre object is initialized
             *
             * @return whether the buffer is registered successfully
             *
             * @details
             *    currently only nn::vi::PixelFormat_Rgba888 is supported as pixel format
             *    buffer is expected to be pitch-linear
             *
             *    System will will use pBuffer memory for not only scan buffer but also internal hose keeping.
             *    Do not assume that start address of the passed buffer with this API matches actual scan buffer start address.
             *    Please call GetScanBufferAddress to obtain the start address when rendering contents in the buffer.
             */
            nn::Result SetScanBuffer(int bufferIndex, void* pBuffer, size_t size)  NN_NOEXCEPT;

            /**
             * @brief obtain the handle to the empty buffer.
             *
             * @param[out] pOutHandle  buffer handle
             *
             * @pre object is initialized
             *
             * @return whether the buffer is obtained successfully
             */
            nn::Result DequeueBuffer(BufferQueueHandle** pOutHandle) NN_NOEXCEPT;

            /**
             * @brief submit the buffer to be scanned on the next frame.
             *
             * @param[in] pHandle  buffer handle
             *
             * @pre object is initialized
             *
             * @return whether the buffer is submitted successfully
             */
            nn::Result QueueBuffer(BufferQueueHandle* pHandle) NN_NOEXCEPT;

            /**
             * @brief Get scan buffer address from buffer handle.
             *
             * @param[out] pOutAddr  address of the buffer
             * @param[in]  pHandle   buffer handle
             *
             * @pre object is initialized
             *
             * @return whether the buffer address is obtained successfully
             */
            nn::Result GetScanBufferAddress(void** pOutAddr, BufferQueueHandle* pBuffer) NN_NOEXCEPT;

            BufferResource* GetData() NN_NOEXCEPT
            {
                return reinterpret_cast<BufferResource*>(&m_Storage[0]);
            }

        private:
            bool m_IsInitialized;
            NN_ALIGNAS(16) char m_Storage[256];
    };

}}}
