﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file  usb_Limits.h
 * @brief USB stack implementation limits
 *
 * @details
 */

#pragma once

#include <nn/dd/dd_DeviceAddressSpaceCommon.h>

namespace nn {
namespace usb {

//----------------------------------------------------------------------------
// Limitations imposed by hardware
//----------------------------------------------------------------------------

/**
 * @brief DMA buffer alignment requirement
 */
const int HwLimitDmaBufferAlignmentSize =
    nn::dd::DeviceAddressSpaceMemoryRegionAlignment;

/**
 * @brief Data cache line size
 */
const int HwLimitDataCacheLineSize = 64;

/**
 * @brief Maximum number of USB ports
 */
const int HwLimitMaxPortCount = 4;

//----------------------------------------------------------------------------
// Limitations imposed by USB Standard
//----------------------------------------------------------------------------

/**
 * @brief Maximum number of endpoints which can exist
 */
const int UsbLimitMaxEndpointsCount = 32;

/**
 * @brief Maximum number of endpoint pairs which can exist
 */
const int UsbLimitMaxEndpointPairCount = 16;

//----------------------------------------------------------------------------
// Limitations imposed by Host Stack implementation
//----------------------------------------------------------------------------

/**
 * @brief Maximum hierarchical enumerated hub depth
 */
const int HsLimitMaxHubHierarchyCount = 8;

/**
 * @brief Maximum number of ports on a hub
 */
const int HsLimitMaxHubPortsCount = 10;

/**
 * @brief Upper sanity bound on maximum number of configurations
 *        a device would be allowed to present.
 */
const int HsLimitMaxConfigurationsPerDeviceCount = 8;

/**
 * @brief Upper sanity bound on maximum number of interfaces a
 *        device would be allowed to present.
 */
const int HsLimitMaxInterfacesPerConfigurationCount = 8;

/**
 * @brief Upper sanity bound on maximum number of alternate
 *        settings a device interface would be allowed to have.
 */
const int HsLimitMaxAltInterfaceSettingsCount = 64;

/**
 * @brief Maximum size of name displayed for debug/validation purposes
 */
const int HsLimitMaxDebugNameSize = 64;

/**
 * @brief Maximum number of devices supported by host stack
 */
const int HsLimitMaxDevicesCount = 20;

/**
 * @brief Maximum number of interfaces supported by host stack
 */
const int HsLimitMaxInterfacesCount =
    HsLimitMaxInterfacesPerConfigurationCount * HsLimitMaxDevicesCount;

/**
 * @brief Maximum number of clients which could use host stack service
 *        at the same time.
 */
const int HsLimitMaxClientCount = HsLimitMaxDevicesCount;

/**
 * @brief Upper sanity bound on maximum number of interfaces
 *        a client would be allowed to initialize (own).
 */
const int HsLimitMaxInterfacesPerClientCount = 8;

/**
 * @brief Upper sanity bound on number of requests which could
 *        be posted to a single endpoint (streaming).
 */
const int HsLimitMaxUrbPerEndpointCount = 16;

/**
 * @brief Upper sanity bound on number of batched requests carried
 *        in a single URB.
 */
const int HsLimitMaxXferPerUrbCount = 8;

/**
 * @brief Upper sanity bound on transfer size for each URB.
 */
const int HsLimitMaxUrbTransferSize = 0x10000 * (4096 / 16 - 1);

/**
 * @brief The only xHCI PAGESIZE supported by this implementation
 */
const int HsLimitXhciPageSize = 0x1000;

/**
 * @brief Maximum number of xHCI scratchpad buffer
 */
const int HsLimitXhciMaxScratchpadBufferCount = 8;

/**
 * @brief Maximum number of interface availability events per client
 */
const int HsLimitMaxClientInterfaceAvailabilityEventCount = 3;

/**
 * @brief Default control transfer timeout (in ms)
 */
const int HsLimitControlXferTimeout = 7500;

/**
 * @brief Maximum number of server side session
 */
const int HsLimitMaxSessionCount = HsLimitMaxClientCount;

/**
 * @brief Maximum number of subdomain which can be created
 */
const int HsLimitMaxSubdomainCount = HsLimitMaxSessionCount;

/**
 * @brief Maximum number of shared objects which can be craeted in
 *        one domain
 */
const int HsLimitMaxObjectPerSubdomain = 256;

/**
 * @brief Number of threads used to provide host stack service
 */
const int HsLimitSfThreadCount = 2;

/**
 * @brief Stack size of host stack service thread
 */
const int HsLimitSfThreadStackSize  = 1024 * 16;

//----------------------------------------------------------------------------
// Limitations imposed by Device Stack implementation
//----------------------------------------------------------------------------

/**
 * @brief Upper sanity bound on maximum number of configurations
 *        a device would be allowed to present.
 */
const int DsLimitMaxConfigurationsPerDeviceCount = 1;

/**
 * @brief Upper sanity bound on maximum number of interfaces a
 *        device would be allowed to present.
 */
const int DsLimitMaxInterfacesPerConfigurationCount = 4;

/**
 * @brief Maximum length of ASCII string which can be put into a string
 *        descriptor
 */
const int DsLimitMaxNameSize = 64;

/**
 * @brief Size of the transaction ring for each endpoint
 */
const int DsLimitRingSize = 8;

/**
 * @brief Maximum number of clients which could use device stack service
 *        at the same time.
 */
const int DsLimitMaxClientCount = 4;
/**
 * @brief Maximum number of server side session
 */
const int DsLimitMaxSessionCount = DsLimitMaxClientCount;

/**
 * @brief Maximum number of subdomain which can be created
 */
const int DsLimitMaxSubdomainCount = DsLimitMaxSessionCount;

/**
 * @brief Maximum number of shared objects which can be craeted in
 *        one domain
 */
const int DsLimitMaxObjectPerSubdomain = 256;

/**
 * @brief Number of threads used to provide device stack service
 */
const int DsLimitSfThreadCount = DsLimitMaxSessionCount;

/**
 * @brief Stack size of device stack service thread
 */
const int DsLimitSfThreadStackSize = 1024 * 16;

//----------------------------------------------------------------------------
// Limitations imposed by Port Manager implementation
//----------------------------------------------------------------------------

/**
 * @brief Maximum number of server side session
 */
const int PmLimitMaxSessionCount = 5;

//----------------------------------------------------------------------------
// Misc
//----------------------------------------------------------------------------

/**
 * @brief Stack size of ISR thread
 */
const int UsbLimitIsrThreadStackSize = 1024 * 16;

/**
 * @brief Stack size of input notifier thread
 */
const int UsbLimitInputNotifierThreadStackSize = 1024 * 4;

} // end of namespace usb
} // end of namespace nn


