﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file    usb_Host.h
 * @brief   USB Host Stack Library
 */
#include <atomic>

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/sf/sf_ExpHeapAllocator.h>
#include <nn/sf/sf_HipcSimpleClientSessionManager.h>

#include <nn/usb/usb_Limits.h>
#include <nn/usb/usb_Result.h>
#include <nn/usb/usb_HostTypes.h>
#include <nn/usb/hs/sfdl/usb_HsClientEpSession.sfdl.h>
#include <nn/usb/hs/sfdl/usb_HsClientIfSession.sfdl.h>
#include <nn/usb/hs/sfdl/usb_HsClientRootSession.sfdl.h>

#include <nn/usb/detail/usb_Ring.h>

namespace nn {
namespace usb {


//--------------------------------------------------------------------------
//  Host class
//--------------------------------------------------------------------------
class Host
{
public:
    friend class HostInterface;
    friend class HostEndpoint;

    Host() NN_NOEXCEPT
        : m_HeapHandle(nullptr)
        , m_RootHandle(nullptr)
        , m_IsInitialized(false)
        , m_RefCount(0)
    {
    }
    ~Host() NN_NOEXCEPT
    {
        m_RootHandle = nullptr;
    }

    /**
     * @brief Initialize usb host library
     *
     * @retresult
     *   @handleresult{nn::ResultSuccess, Initialize succeeded}
     *   @handleresult{nn::usb::ResultAlreadyInitialized, Object is already in the initialized state}
     *   @handleresult{nn::usb::ResultMemAllocFailure, Insufficient internal resources available to complete this operation}
     * @endretresult
     *
     * @pre
     *   Initialize() has not already been called, or if it was then Finalize()
     *   was called.
     *
     * @post
     *   Usb host library is ready for operation.
     *
     * @details
     *   This method is not reentrant against other Initialize() or Finalize()
     *   calls to the same Host object.
     */
    Result Initialize() NN_NOEXCEPT;

    /**
     * @brief Finalize usb host library
     *
     * @retresult
     *   @handleresult{nn::ResultSuccess, Finalize succeeded}
     *   @handleresult{nn::usb::ResultNotInitialized, This object is not initialized}
     * @endretresult
     *
     * @pre
     *   Initialize() has already been called.
     *   Any HostInterface or HostEndpoint objects initialized against this
     *   object have since been finalized.
     *
     * @post
     *   Usb host library is can no longer be used by this process.
     *
     * @details
     *   This method is not reentrant against other Initialize() or Finalize()
     *   calls to the same Host object.
     */
    Result Finalize() NN_NOEXCEPT;

    /**
     * @brief Check if this host session has been initialized
     *
     * @retval Boolean
     *   - True: The host session has been initialized
     *   - False: The host session hss not been initialized
     *
     * @details
     *   Determines if this host session has been initialized. Has Initialize()
     *   been called?
     */
    bool IsInitialized() NN_NOEXCEPT;

    /**
     * @brief Query all interfaces
     *
     * @param[out] pOutIfCount
     *   Pointer to returned interface count.
     *
     * @param[out] pOutBuffer
     *   Pointer to buffer, into which query output will be returned.
     *
     * @param[in] queryBufferSize
     *   Size of provided buffer referred to by pOutBuffer.
     *
     * @param[in] pDevFilter
     *   Pointer to filter specification, potentially limiting scope of
     *   returned interfaces.
     *
     * @retresult
     *   @handleresult{nn::ResultSuccess, Query succeeded}
     *   @handleresult{nn::usb::ResultNotInitialized, This object is not initialized}
     * @endretresult
     *
     * @pre
     *   Initialize() must be called once first.
     *
     * @post
     *   Internal state of USB service or device is not affected.
     *
     * @details
     *   All interfaces which are enumerated and matching criteria of specified
     *   filter are returned, regardless whether they are acquired (via call
     *   to HostInterface::Initialize()) or not.
     *
     *   You are not able to acquire the interface with its InterfaceHandle
     *   returned by this call because the InterfaceHandle returned is always
     *   invalid. Use QueryAvailableInterfaces() instead if you need to acquire
     *   any interface.
     */
    Result QueryAllInterfaces(int32_t*              pOutIfCount,
                              InterfaceQueryOutput* pOutBuffer,
                              size_t                queryBufferSize,
                              DeviceFilter*         pDevFilter) NN_NOEXCEPT;

    /**
     * @brief Query available interfaces
     *
     * @param[out] pOutIfCount
     *   Pointer to returned interface count.
     *
     * @param[out] pOutBuffer
     *   Pointer to buffer, into which query output will be returned.
     *
     * @param[in] queryBufferSize
     *   Size of provided buffer referred to by pOutBuffer.
     *
     * @param[in] pDevFilter
     *   Pointer to filter specification, potentially limiting scope of
     *   returned interfaces.
     *
     * @retresult
     *   @handleresult{nn::ResultSuccess, Query succeeded}
     *   @handleresult{nn::usb::ResultNotInitialized, This object is not initialized}
     * @endretresult
     *
     * @pre
     *   Initialize() must be called once first.
     *
     * @post
     *   Internal state of USB service or device is not affected.
     *
     * @details
     *   Only interfaces which are enumerated, available and matching criteria
     *   of specified filter are returned.
     */
    Result QueryAvailableInterfaces(int32_t*              pOutIfCount,
                                    InterfaceQueryOutput* pOutBuffer,
                                    size_t                queryBufferSize,
                                    DeviceFilter*         pDevFilter) NN_NOEXCEPT;

    /**
     * @brief Query acquired interfaces
     *
     * @param[out] pOutIfCount
     *   Pointer to returned interface count.
     *
     * @param[out] pOutBuffer
     *   Pointer to buffer, into which query output will be returned.
     *
     * @param[in]  queryBufferSize
     *   Size of provided buffer referred to by pOutBuffer
     *
     * @retresult
     *   @handleresult{nn::ResultSuccess, Query succeeded}
     *   @handleresult{nn::usb::ResultNotInitialized, This object is not initialized}
     * @endretresult
     *
     * @pre
     *   Initialize() must be called once first.
     *
     * @post
     *   Internal state of USB service or device is not affected.
     *
     * @details
     *   Only interfaces which have been previously acquired, via call to
     *   HostInterface::Initialize(), are returned.
     */
    Result QueryAcquiredInterfaces(int32_t*              pOutIfCount,
                                   InterfaceQueryOutput* pOutBuffer,
                                   size_t                queryBufferSize) NN_NOEXCEPT;

    /**
     * @brief Create interface availability event
     *
     * @param[out] pOutEvent
     *   Supplied system event is initialized.
     *
     * @param[in] eventClearMode
     *   Mode of operation assumed by initialized system event, related to how
     *   set event is cleared.
     *
     * @param[in] filterIndex
     *   Zero based index referring to which filter specification the caller
     *   wants to assign to the specified filter. Valid range of this value
     *   is 0 to nn::usb::HsLimitMaxClientInterfaceAvailabilityEventCount-1.
     *
     * @param[in] pDevFilter
     *   Pointer to specified filter parameters, potentially limiting
     *   scope of devices for which availability indication will be provided.
     *
     * @retresult
     *   @handleresult{nn::ResultSuccess, Creation succeeded}
     *   @handleresult{nn::usb::ResultNotInitialized, This object is not initialized}
     * @endretresult
     *
     * @pre
     *   Initialize() must be called once first.
     *
     * @post
     *   The slot associated with the specified filter index is now utilized.
     *   Specified system event is now initialized.
     *
     * @details
     *   When an interface with characteristics matching those described by the
     *   specified filter becomes available, this system event will be signalled.
     */
    Result CreateInterfaceAvailableEvent(nn::os::SystemEventType* pOutEvent,
                                         nn::os::EventClearMode   eventClearMode,
                                         uint8_t                  filterIndex,
                                         DeviceFilter*            pDevFilter) NN_NOEXCEPT;

    /**
     * @brief Destroy interface availability event
     *
     * @param[in] pInEvent
     *   Supplied system event is destroyed.
     *
     * @param[in] filterIndex
     *   Zero based index referring to which filter specification the caller
     *   wants to assign to the specified filter. Valid range of this value
     *   is 0 to nn::usb::HsLimitMaxClientInterfaceAvailabilityEventCount-1.
     *
     * @retresult
     *   @handleresult{nn::ResultSuccess, Destroy succeeded}
     *   @handleresult{nn::usb::ResultNotInitialized, This object is not initialized}
     * @endretresult
     *
     * @pre
     *   Initialize() must be called once first. Specified system event should
     *   have been successfully created via prior call to
     *   CreateInterfaceAvailableEvent().
     *
     * @post
     *   Specified system event is now finalized.
     *
     * @details
     *   The system event is no longer receving signals, and should not be used.
     */
    Result DestroyInterfaceAvailableEvent(nn::os::SystemEventType* pInEvent,
                                          uint8_t                  filterIndex) NN_NOEXCEPT;

    /**
     * @brief Get the system event for Interface State Change Event
     *
     * @return A pointer to the system event, or nullptr if object was not initialized.
     *
     * @pre
     *   Initialize() must be called once first.
     *
     * @details
     *   The returned pointer points to a system event object managed by this
     *   object. When any interface owned by this class driver experiences a
     *   change in state, this system event will be signalled. It is effectively
     *   a global event for any acquired interface. The potential advantage of
     *   using this global event is to avoid complex multiwait holder logic when
     *   class driver needs to manage multiple acquired devices.
     *
     *   The returned system event should be cleared manually.
     *
     *   Note that the pointer returned will become invalid if this Host object
     *   is finalized or destroied.
     */
    nn::os::SystemEventType* GetInterfaceStateChangeEvent() NN_NOEXCEPT;

    /**
     * @brief Enable test mode for a specific port
     *
     * @param[in] port
     *   The physical port number, starting from 0
     *
     * @param[in] mode
     *   One of the test mode (J / K / SE0_NAK / PACKET)
     *
     * @param[in] offset
     *   Drive strength offset to apply
     *
     * @param[out] pOutStrength
     *   The net value after applying the offset
     *
     * @retresult
     *   @handleresult{nn::usb::ResultNotImplemented}
     *   @handleresult{nn::usb::ResultInvalidParameter}
     * @endretresult
     *
     * @pre
     *   Initialize() must be called once first.
     *
     * @details
     *   Put the port to specified test mode. After this call, the only way
     *   to return to normal operation is reboot.
     */
    Result SetTestMode(uint32_t port, TestMode mode, int32_t offset, uint32_t *pOutStrength) NN_NOEXCEPT;

    /**
     * @brief Print interface query
     *
     * @param[in]  pInQuery
     *   Pointer interface query information to be printed.
     *
     * @pre
     *   Initialize() must be called once first.
     *
     * @post
     *   No state is changed.
     *
     * @details
     *   Prints all fields of InterfaceQueryOutput structure, for use during
     *   debug and validation.
     */
    static void PrintInterfaceQuery(InterfaceQueryOutput* pInQuery) NN_NOEXCEPT;

    /**
     * @brief Print interface profile
     *
     * @param[in]  pInInterfaceProfile
     *   Pointer interface profile information to be printed.
     *
     * @pre
     *   Initialize() must be called once first.
     *
     * @post
     *   No state is changed.
     *
     * @details
     *   Prints all fields of InterfaceProfile structure, for use during debug
     *   and validation.
     */
    static void PrintInterfaceProfile(InterfaceProfile *pInInterfaceProfile) NN_NOEXCEPT;

private:
    typedef nn::sf::ExpHeapAllocator MyAllocator;

    MyAllocator                                   m_Allocator;
    std::aligned_storage<4 * 1024>::type         m_HeapBuffer;
    nn::lmem::HeapHandle                          m_HeapHandle;

    nn::sf::HipcSimpleClientSessionManager        m_Domain;
    nn::sf::SharedPointer<hs::IClientRootSession> m_RootHandle;
    bool                                          m_IsInitialized;
    ::std::atomic<int>                            m_RefCount;

    nn::os::SystemEventType                       m_IfStateChangeEvent;
};


//--------------------------------------------------------------------------
//  HostInterface class
//--------------------------------------------------------------------------
class HostInterface
{
public:
    friend class HostEndpoint;

    HostInterface() NN_NOEXCEPT
        : m_pHost(nullptr)
        , m_pHandle(nullptr)
        , m_IsInitialized(false)
        , m_RefCount(0)
        , m_Mutex(false)
        , m_Quirks(0)
    {
        memset(&m_DeviceProfile,    0, sizeof(m_DeviceProfile));
        memset(&m_InterfaceProfile, 0, sizeof(m_InterfaceProfile));
    }

    ~HostInterface() NN_NOEXCEPT
    {
        m_pHandle = nullptr;
    }


    /**
     * @brief Initializes session for specified device interface
     *
     * @param[in]  pHost
     *   Pointer to this host class driver.
     *
     * @param[in]  ifHandle
     *   Handle of interface which is to be acquired.
     *
     * @retresult
     *   @handleresult{nn::ResultSuccess, Initialize succeeded}
     *   @handleresult{nn::usb::ResultAlreadyInitialized, Object is already in the initialized state}
     *   @handleresult{nn::usb::ResultMemAllocFailure, Insufficient internal resources available to complete this operation}
     *   @handleresult{nn::usb::ResultInterfaceInvalid, Device in which this interface is contained has disconnected}
     *   @handleresult{nn::usb::ResultInterfaceLimitExceeded, An excessive number of interface objects have been initialized}
     * @endretresult
     *
     * @pre
     *   Initialize() has not already been called, or if it was then Finalize() was called.
     *
     * @details
     *   By doing this operation, this host class driver now "owns" this device
     *   interface.
     *
     *   This method is not reentrant against other Initialize() or Finalize()
     *   calls to the same HostInterface object.
     */
    Result Initialize(Host* pHost, InterfaceHandle ifHandle) NN_NOEXCEPT;

    /**
     * @brief Terminates and destroys previously established device interface session.
     *
     * @retresult
     *   @handleresult{nn::ResultSuccess, Finalize succeeded}
     *   @handleresult{nn::usb::ResultNotInitialized, This object is not initialized}
     * @endretresult
     *
     * @pre
     *   Initialize() has already been called.
     *   Any HostEndpoint objects initialized against this object have since been finalized.
     *
     * @details
     *   Terminates previously established device interface session. Ownership
     *   of the interface is released.
     *
     *   This method is not reentrant against other Initialize() or Finalize()
     *   calls to the same HostInterface object.
     */
    Result Finalize() NN_NOEXCEPT;

    /**
     * @brief Check if this device interface session has been initialized
     *
     * @retval Boolean
     *   - True: The device interface session has been initialized
     *   - False: The device interface session hss not been initialized
     *
     * @details
     *   Determines if this device interface session has been initialized. Has
     *   Initialize() been called?
     */
    bool IsInitialized() NN_NOEXCEPT;

    /**
     * @brief Get the system event for Interface State Change Event
     *
     * @return A pointer to the system event, or nullptr if object was not initialized.
     *
     * @pre
     *   Initialize() must be called once first.
     *
     * @details
     *   The returned pointer points to a system event object managed by this
     *   object. When this interface experiences a change in state, this system
     *   event will be signalled.
     *
     *   The returned system event should be cleared manually.
     *
     *   Note that the pointer returned will become invalid if this
     *   HostInterface object is finalized.
     */
    nn::os::SystemEventType* GetStateChangeEvent() NN_NOEXCEPT;

    /**
     * @brief Performs standard set-interface command
     *
     * @param[in]  bAlternateSetting
     *   Alternate setting to be applied
     *
     * @retresult
     *   @handleresult{nn::ResultSuccess, Operation succeeded}
     *   @handleresult{nn::usb::ResultNotInitialized, This object is not initialized}
     *   @handleresult{nn::usb::ResultInvalidAlternateSetting, Specified bAlternateSetting is not valid for this interface}
     *   @handleresult{nn::usb::ResultInterfaceInvalid, Device in which this interface is contained has disconnected}
     * @endretresult
     *
     * @pre
     *   Initialize() must be called once first.
     *
     * @details
     *   Performs standard set-interface command.
     */
    Result SetInterface(uint8_t bAlternateSetting) NN_NOEXCEPT;

    /**
     * @brief Get the interface profile
     *
     * @param[out]  pOutInterfaceProfile
     *   Returned interface profile
     *
     * @retresult
     *   @handleresult{nn::ResultSuccess, Operation succeeded}
     *   @handleresult{nn::usb::ResultNotInitialized, This object is not initialized}
     *   @handleresult{nn::usb::ResultInterfaceInvalid, Device in which this interface is contained has disconnected}
     * @endretresult
     *
     * @pre
     *   Initialize() must be called once first.
     *
     * @details
     *   Retrieves profile associated with currenly active alternate setting.
     */
    Result GetInterface(InterfaceProfile* pOutInterfaceProfile) NN_NOEXCEPT;

    /**
     * @brief Get the interface profile for the specified alternate setting

     * @param[out]  pOutInterfaceProfile
     *   Returned interface profile
     *
     * @param[in]  bAlternateSetting
     *   Alternate setting for which profile is to be retrieved
     *
     * @retresult
     *   @handleresult{nn::ResultSuccess, Operation succeeded}
     *   @handleresult{nn::usb::ResultNotInitialized, This object is not initialized}
     *   @handleresult{nn::usb::ResultInterfaceInvalid, Device in which this interface is contained has disconnected}
     *   @handleresult{nn::usb::ResultInvalidAlternateSetting}
     * @endretresult
     *
     * @pre
     *   Initialize() must be called once first.
     *
     * @details
     *   Retrieve interface profile for the specified alternate setting, without
     *   actually activating the setting.
     */
    Result GetAlternateInterface(InterfaceProfile* pOutInterfaceProfile,
                                 uint8_t           bAlternateSetting) NN_NOEXCEPT;

    /**
     * @brief Retrieve the current USB host controller frame count
     *
     * @param[out]  pFrame
     *   Returned frame number
     *
     * @retresult
     *   @handleresult{nn::ResultSuccess, Operation succeeded}
     * @endretresult
     *
     * @pre
     *   Initialize() must be called once first.
     *
     * @details
     *   Retrieve the current USB host controller frame count. This is usually
     *   relevant only for isochronous scheduling.
     */
    Result GetCurrentFrame(FrameNumber *pFrame) NN_NOEXCEPT;

    /**
     * @brief Submit control request
     *
     * @param[out]  pOutTransferredSize
     *   Actually data bytes send or received
     *
     * @param[in,out]  pData
     *   Depends on bmRequestType, this is a pointer to data which is retrieved
     *   from device and written to your memory, or a pointer to data which is
     *   read from your memory and written to the device.
     *
     * @param[in]  bmRequestType
     *   Characteristics of request. See USB spec for details.
     *
     * @param[in]  bRequest
     *   Specific request. See USB spec for details.
     *
     * @param[in]  wValue
     *   Word-sized field that varies according to request. See USB spec for
     *   details.
     *
     * @param[in]  wIndex
     *   Word-sized field that varies according to request. See USB spec for
     *   details.
     *
     * @param[in]  wLength
     *   Number of bytes to transfer if there is a data stage. See USB spec for
     *   details.
     *
     * @retresult
     *   @handleresult{nn::ResultSuccess, Operation succeeded}
     *   @handleresult{nn::usb::ResultNotInitialized, This object is not initialized}
     *   @handleresult{nn::usb::ResultAlignmentError, Specified pData was not properly aligned}
     *   @handleresult{nn::usb::ResultTransferDescriptorStall, Control transfers complete with stall if device is logically unable to complete the request}
     *   @handleresult{nn::usb::ResultInterfaceInvalid, Device in which this interface is contained has disconnected}
     * @endretresult
     *
     * @pre
     *   Initialize() must be called once first.
     *
     * @post
     *   Data was read from device.
     *
     * @details
     *   Submit a USB control request. All the control requests to the same
     *   device are implicitly serialized.
     */
    Result ControlRequest(size_t    *pOutTransferredSize,
                          void      *pData,
                          uint8_t    bmRequestType,
                          uint8_t    bRequest,
                          uint16_t   wValue,
                          uint16_t   wIndex,
                          uint16_t   wLength) NN_NOEXCEPT;

    /**
     * @brief Reset the device
     *
     * @retresult
     *   @handleresult{nn::ResultSuccess, Creation succeeded}
     *   @handleresult{nn::usb::ResultNotInitialized, This object is not initialized}
     *   @handleresult{nn::usb::ResultInterfaceInvalid, Device in which this interface is contained has disconnected}
     * @endretresult
     *
     * @pre
     *   The interface has been initialized via call to Initialize().
     *
     * @post
     *   The device to which this interface belongs begins the process of reset
     *   and re-enumeration.
     *
     * @details
     *   This method has asynchronous behavior. The actual process of resetting
     *   the device could happen after this call has returned.
     */
    Result ResetDevice() NN_NOEXCEPT;

    /**
     * @brief Retrieves reference to the current interface descriptor
     *
     * @retval UsbInterfaceDescriptor
     *
     * @pre
     *   The interface has been initialized via call to Initialize().
     */
    UsbInterfaceDescriptor& GetDescriptor() NN_NOEXCEPT;

private:
    Host*                                           m_pHost;
    nn::sf::SharedPointer<hs::IClientIfSession>     m_pHandle;
    DeviceProfile                                   m_DeviceProfile;
    InterfaceProfile                                m_InterfaceProfile;
    bool                                            m_IsInitialized;
    ::std::atomic<int>                              m_RefCount;
    nn::os::Mutex                                   m_Mutex;
    nn::os::SystemEventType                         m_StateChangeEvent;
    nn::os::SystemEventType                         m_CtrlXferCompletionEvent;
    uint32_t                                        m_Quirks;
};


//--------------------------------------------------------------------------
//  HostEndpoint class
//--------------------------------------------------------------------------
class HostEndpoint
{
public:
    friend class HostInterface;
    HostEndpoint() NN_NOEXCEPT
        : m_pIf(nullptr)
        , m_pHandle(nullptr)
        , m_EpType(UsbEndpointType_Invalid)
        , m_EndpointNumber(0)
        , m_EpDirection(UsbEndpointDirection_Invalid)
        , m_IsInitialized(false)
        , m_RefCount(0)
        , m_ReportRing(m_pHandle)
        , m_MaxUrbCount(0)
    {
        memset(&m_EndpointDescriptor, 0, sizeof(m_EndpointDescriptor));
    }

    ~HostEndpoint() NN_NOEXCEPT
    {
        m_pHandle = nullptr;
    }

    /**
     * @brief Initializes an device endpoint session.
     *
     * @param[in]  pIf
     *   Pointer to client interface to which this endpoint belongs.
     *
     * @param[in]  pEpDescriptor
     *   Pointer to descriptor values of endpoint to be opened
     *
     * @param[in]  maxUrb
     *   Maximum number of simultaneously pending (pipelined) requests which
     *   could be submitted by you on this endpoint.
     *
     * @param[in]  maxXferSize
     *   Deprecated and ignored.
     *
     * @retresult
     *   @handleresult{nn::ResultSuccess, Initialize succeeded}
     *   @handleresult{nn::usb::ResultAlreadyInitialized, Object is already in the initialized state}
     *   @handleresult{nn::usb::ResultMemAllocFailure, Insufficient internal resources available to complete this operation}
     *   @handleresult{nn::usb::ResultEndpointLimitExceeded, An excessive number of endpoint objects have been initialized}
     *   @handleresult{nn::usb::ResultUrbLimitExceeded, Specified maxUrb parameter is invalid}
     *   @handleresult{nn::usb::ResultSizeError, Specified maxXferSize parameter is invalid}
     *   @handleresult{nn::usb::ResultInterfaceInvalid, Device in which this interface is contained has disconnected}
     * @endretresult
     *
     * @pre
     *   Initialize() has not already been called, or if it was then Finalize()
     *   was called.
     *
     * @details
     *   This method performs same operation as other Initialize() call, but
     *   simply has fewer parameters, instead accepting the endpoint descriptor
     *   structure.
     *
     *   This method is not reentrant against other Initialize() or Finalize()
     *   calls to the same HostEndpoint object.
     */
    Result Initialize(HostInterface*         pIf,
                      UsbEndpointDescriptor* pEpDescriptor,
                      uint16_t               maxUrb,
                      uint32_t               maxXferSize) NN_NOEXCEPT;

    /**
     * @brief Terminates and destroys previously established device endpoint session.
     *
     * @retresult
     *   @handleresult{nn::ResultSuccess, Finalize succeeded}
     *   @handleresult{nn::usb::ResultNotInitialized, This object is not initialized}
     * @endretresult
     *
     * @pre
     *   Initialize() has already been called.
     *
     * @details
     *   If called while endpoint transactions are posted they will be
     *   cancelled, during which time this API will block.
     *
     *   This method is not reentrant against other Initialize() or Finalize()
     *   calls to the same HostEndpoint object.
     */
    Result Finalize() NN_NOEXCEPT;

    /**
     * @brief Check if this device endpoint session has been initialized
     *
     * @retval Boolean
     *   - True: The device endpoint session has been initialized
     *   - False: The device endpoint session hss not been initialized
     *
     * @details
     *   Determines if this device endpoint session has been initialized. Has
     *   Initialize() been called?
     */
    bool IsInitialized() NN_NOEXCEPT;

    /**
     * @brief Statically sMMU map the specified buffer
     *
     * @param[in] buffer
     *   The data buffer pointer. Must be page aligned.
     *
     * @param[in] size
     *   Size of that buffer. Must be page aligned.
     *
     * @retresult
     *   @handleresult{nn::usb::ResultAlignmentError, The buffer is not properly aligned}
     *   @handleresult{nn::usb::ResultOperationDenied, A static map is already established}
     *   @handleresult{nn::usb::ResultNotInitialized, This object is not initialized}
     * @endretresult
     *
     * @pre
     *   Initialize() has already been called.
     *
     * @details
     *   USB server statically maps the specified buffer. If a later transfer
     *   request references a data source / destination residing in this
     *   buffer, the USB server just uses already established map. Otherwise,
     *   map and unmap is performed before and after the transfer.
     *
     *   Only one static map can be established, and once it's established,
     *   there is no way to explicitly modify or unmap it. It will be unmapped
     *   automatically when this object is Finalize() or destructed.
     */
    Result CreateSmmuSpace(void *buffer, uint32_t size) NN_NOEXCEPT;

    /**
     * @brief Ask USB server to put transfer report data into shared buffer
     *
     * @param[in] buffer
     *   The data buffer pointer. Must be page aligned. The buffer cannot be on stack.
     *
     * @param[in] size
     *   Size of that buffer. Must be page aligned.
     *
     * @retresult
     *   @handleresult{nn::usb::ResultAlignmentError, The buffer is not properly aligned}
     *   @handleresult{nn::usb::ResultInvalidParameter, The buffer size is too small}
     *   @handleresult{nn::usb::ResultOperationDenied, The transfer report is already shared}
     *   @handleresult{nn::usb::ResultNotInitialized, This object is not initialized}
     * @endretresult
     *
     * @pre
     *   Initialize() has already been called.
     *
     * @details
     *   Usually GetXferReport() would retrive transfer report data from USB
     *   server by SF calls. This call setup a shared memory between class
     *   driver and USB server. USB server will put all transfer report into
     *   this shared buffer. GetXferReport() now retrives the report directly
     *   from the buffer, thus eliminates the need of SF calls.
     *
     *   This should be only called once after Initialize() if the class driver
     *   want to minimize the CPU load. Once the buffer is shared, there is no
     *   way to explicitly unshare it. It will be unshared automatically when
     *   this object is Finizhe() or destructed.
     */
    Result ShareReportRing(void *buffer, uint32_t size) NN_NOEXCEPT;

    /**
     * @brief Get the system event for data transfer completion
     *
     * @return A pointer to the system event, or nullptr if object was not initialized.
     *
     * @pre
     *   Initialize() must be called once first.
     *
     * @details
     *   The returned pointer points to a system event object managed by this
     *   object. When a data transfer posted before is completed, this system
     *   event will be signalled.
     *
     *   The returned system event should be cleared manually.
     *
     *   Note that the pointer returned will become invalid if this HostEndpoint
     *   object is finalized. It's the class driver's responsibility to make sure
     *   that the event is not being used after the HostEndpoint finalization.
     *
     *   Usually the class driver only needs this pointer to work with MultiWait.
     *   Otherwise TimedWaitForCompletion() shall be used.
     */
    nn::os::SystemEventType* GetCompletionEvent() NN_NOEXCEPT;

    /**
     * @brief Wait for the data transfer completion
     *
     * @param[in] timeout
     *   The timeout until the wait returns. 0 means wait forever.
     *
     * @return
     *   True if transfer is completed. False if times out.
     *
     * @pre
     *   Initialize() must be called once first.
     *
     * @details
     *   This API blocks until a transaction is completed, or until specified
     *   time is elapsed.
     */
    bool TimedWaitForCompletion(nn::TimeSpan timeout) NN_NOEXCEPT;

    /**
     * @brief Transfer data with this endpoint
     *
     * @param[out] pOutTransferredSize
     *   The actual size of data transferred from/into user buffer
     *
     * @param[in,out] buffer
     *   The user buffer to provide data or store received data
     *
     * @param[in] bytes
     *   The size of user buffer in bytes
     *
     * @retresult
     *   @handleresult{nn::ResultSuccess, Operation succeeded}
     *   @handleresult{nn::usb::ResultNotInitialized, This object is not initialized}
     *   @handleresult{nn::usb::ResultEndpointStateInvalid, Endpoint not in a state where this operation is possible}
     *   @handleresult{nn::usb::ResultInterfaceInvalid, Device in which this interface is contained has disconnected}
     *   @handleresult{nn::usb::ResultUrbLimitExceeded, Pending request limit previously specified as maxUrb has been exceeded}
     *   @handleresult{nn::usb::ResultAlignmentError, Specified buffer pointer was not properly aligned}
     *   @handleresult{nn::usb::ResultTransferDescriptorStall, Transactions complete with stall if device is logically unable to perform the operation}
     * @endretresult
     *
     * @pre
     *   Initialize() must be called once first.
     *   The buffer must be aligned to nn::usb::HwLimitDmaBufferAlignmentSize.
     *
     * @details
     *   Depends on the direction of this endpoint, this call will transfer data
     *   from or receive data to the user provided buffer.
     *
     *   The buffer must be aligned to nn::usb::HwLimitDmaBufferAlignmentSize.
     *
     *   You cannot transfer more than nn::usb::HsLimitMaxUrbTransferSize bytes.
     *
     *   You are not supposed to interleave this call with PostBufferAsync().
     */
    Result PostBuffer(size_t    *pOutTransferredSize,
                      void      *buffer,
                      uint32_t   bytes) NN_NOEXCEPT;

    /**
     * @brief Transfer data with this endpoint
     *
     * @param[out] pOutXferId
     *   The unique ID of submitted transfer
     *
     * @param[in,out] buffer
     *   The user buffer to provide data or store received data
     *
     * @param[in] bytes
     *   The size of user buffer in bytes
     *
     * @param[in] context
     *   Arbitrary data which will be passed back unchanged in Xfer Report.
     *
     * @retresult
     *   @handleresult{nn::ResultSuccess, Transaction has been successfully initiated}
     *   @handleresult{nn::usb::ResultNotInitialized, This object is not initialized}
     *   @handleresult{nn::usb::ResultEndpointStateInvalid, Endpoint not in a state where this operation is possible}
     *   @handleresult{nn::usb::ResultInterfaceInvalid, Device in which this interface is contained has disconnected}
     *   @handleresult{nn::usb::ResultUrbLimitExceeded, Pending request limit previously specified as maxUrb has been exceeded}
     *   @handleresult{nn::usb::ResultAlignmentError, Specified buffer pointer was not properly aligned}
     * @endretresult
     *
     * @pre
     *   Initialize() must be called once first.
     *   The buffer must be aligned to nn::usb::HwLimitDmaBufferAlignmentSize.
     *
     * @details
     *   Depends on the direction of this endpoint, this call submits a request
     *   to transfer data from or receive data to the user provided buffer. The
     *   call returns immediately and doesn't wait for data transfer completion.
     *
     *   The maximum number of pending requests is maxUrb, as specified upon
     *   prior call to nn::usb::HostEndpoint::Inititialize().
     *   nn::usb::ResultUrbLimitExceeded is returned if this transaction limit
     *   is reached, at which point you must wait for prior transactions to
     *   complete before trying again. Hopefully you class driver was written
     *   carefully enough to deterministically know and self limit pipelined
     *   transaction counts.
     *
     *   The size of all pending transfers cannot exceed
     *   nn::usb::HsLimitMaxUrbTransferSize. This implies that the size of a
     *   single transfer also shouldn't be bigger than that value.
     *
     *   The transaction status of all the requests can be polled by
     *   GetXferReport(). Look into the report with the Xfer ID returned by this
     *   call as the key if you need to know the status of a specific request.
     *
     *   The buffer must be aligned to nn::usb::HwLimitDmaBufferAlignmentSize.
     *   The ownership of the buffer is effectively transferred to DMA
     *   Controller, thus you shouldn't access the buffer in any means before
     *   the request is completed.
     *
     *   You are not supposed to interleave this call with PostBuffer().
     */
    Result PostBufferAsync(uint32_t   *pOutXferId,
                           void       *buffer,
                           uint32_t    bytes,
                           uint64_t    context) NN_NOEXCEPT;

    /**
     * @brief Issue a batch of data transfers with this endpoint
     *
     * @param[out] pOutXferId
     *   The unique ID of submitted transfer
     *
     * @param[in,out] buffer
     *   The user buffer to provide data or store received data
     *
     * @param[in] pSizeArray
     *   An array holding the size of each transfer in this batch
     *
     * @param[in] xferCount
     *   Number of transfers in this batch
     *
     * @param[in] context
     *   Arbitrary data which will be passed back unchanged in Xfer Report.
     *
     * @param[in] policy
     *   Isoch transfer schedule policy
     *
     * @param[in] frameId
     *   The frame id of the first isoch transfer in the batch
     *
     * @retresult
     *   @handleresult{nn::ResultSuccess, Transaction has been successfully initiated}
     *   @handleresult{nn::usb::ResultNotInitialized, This object is not initialized}
     *   @handleresult{nn::usb::ResultEndpointStateInvalid, Endpoint not in a state where this operation is possible}
     *   @handleresult{nn::usb::ResultInterfaceInvalid, Device in which this interface is contained has disconnected}
     *   @handleresult{nn::usb::ResultUrbLimitExceeded, Pending request limit previously specified as maxUrb has been exceeded}
     *   @handleresult{nn::usb::ResultAlignmentError, Specified buffer pointer was not properly aligned}
     * @endretresult
     *
     * @pre
     *   Initialize() must be called once first.
     *   The buffer must be aligned to nn::usb::HwLimitDmaBufferAlignmentSize.
     *
     * @details
     *   Depends on the direction of this endpoint, this call submits a batch of
     *   requests to transfer data from or receive data to the user provided
     *   buffer. The data for those requests are placed sequentially in user
     *   provided buffer. The call returns immediately and doesn't wait for data
     *   transfer completion.
     *
     *   The maximum number of pending requests is maxUrb, as specified upon
     *   prior call to nn::usb::HostEndpoint::Inititialize().
     *   nn::usb::ResultUrbLimitExceeded is returned if this transaction limit
     *   is reached, at which point you must wait for prior transactions to
     *   complete before trying again. Hopefully you class driver was written
     *   carefully enough to deterministically know and self limit pipelined
     *   transaction counts.
     *
     *   The size of all pending transfers cannot exceed
     *   nn::usb::HsLimitMaxUrbTransferSize. This implies that the size of a
     *   single transfer also shouldn't be bigger than that value.
     *
     *   The transaction status of all the requests can be polled by
     *   GetXferReport(), which are available only after the completion of the
     *   last transaction in the batch. All the transactions in the batch share
     *   the same Xfer ID which is returned by this call.
     *
     *   "policy" and "frameId" are ignored by non-isoch endpoints.
     *
     *   The buffer must be aligned to nn::usb::HwLimitDmaBufferAlignmentSize.
     *   The ownership of the buffer is effectively transferred to DMA
     *   Controller, thus you shouldn't access the buffer in any means before
     *   the request is completed.
     */
    Result BatchBufferAsync(uint32_t       *pOutXferId,
                            void           *buffer,
                            const uint32_t *pSizeArray,
                            uint32_t        xferCount,
                            uint64_t        context,
                            SchedulePolicy  policy,
                            uint32_t        frameId) NN_NOEXCEPT;

    /**
     * @brief Poll the status of submitted transfers
     *
     * @param[out] pOutCount
     *   Number of reports returned
     *
     * @param[out] pOutReport
     *   Pointer to the buffer where the reports will be stored
     *
     * @param[out] count
     *   Number of reports requested
     *
     * @retresult
     *   @handleresult{nn::ResultSuccess, Operation succeeded}
     *   @handleresult{nn::usb::ResultNotInitialized, This object is not initialized}
     * @endretresult
     *
     * @details
     *   Get the status report of completed transfers.
     */
    Result GetXferReport(uint32_t     *pOutCount,
                         XferReport   *pOutReport,
                         uint32_t      count) NN_NOEXCEPT;

    /**
     * @brief Clears an endpoint halt condition
     *
     * @retresult
     *   @handleresult{nn::ResultSuccess, Operation succeeded}
     *   @handleresult{nn::usb::ResultNotInitialized, This object is not initialized}
     *   @handleresult{nn::usb::ResultEndpointStateInvalid, Endpoint not in a state where this operation is possible}
     *   @handleresult{nn::usb::ResultInterfaceInvalid, Device in which this interface is contained has disconnected}
     *   @handleresult{nn::usb::ResultMemAllocFailure, Insufficient internal resources available to complete this operation}
     * @endretresult
     *
     * @pre
     *   Initialize() must be called once first.
     *
     * @details
     *  Clears an endpoint halt condition for recovery from a stall, as
     *  described by section 9.4.5 of the USB 2.0 standard. As part of
     *  the procedure, the endpoint has prior pending transactions
     *  cancelled, and all internal state is reset.
     */
    Result ClearEndpointHalt() NN_NOEXCEPT;

    /**
     * @brief Retrieves reference to the current endpoint descriptor
     *
     * @retval UsbEndpointDescriptor
     *
     * @pre
     *   Initialize() must be called once first.
     */
    UsbEndpointDescriptor& GetDescriptor() NN_NOEXCEPT;

    /**
     * @brief Terminate the endpoint
     *
     * @retresult
     *   @handleresult{nn::ResultSuccess, Operation succeeded}
     *   @handleresult{nn::usb::ResultNotInitialized, This object is not initialized}
     *   @handleresult{nn::usb::ResultEndpointStateInvalid, Endpoint not in a state where this operation is possible}
     *   @handleresult{nn::usb::ResultInterfaceInvalid, Device in which this interface is contained has disconnected}
     * @endretresult
     *
     * @pre
     *   Initialize() must be called once first.
     *
     * @details
     *   Terminates the endpoint. Any previously or subsequently posted
     *   transactions will complete with status ResultEndpointClosed().
     */
    Result Terminate() NN_NOEXCEPT;

private:
    class ReportRing : public detail::RingBase<XferReport>
    {
    public:
        explicit ReportRing(nn::sf::SharedPointer<hs::IClientEpSession>& epSession) NN_NOEXCEPT;
        virtual ~ReportRing() NN_NOEXCEPT;

        Result Attach(void *buffer, uint32_t size, uint32_t itemCount) NN_NOEXCEPT;
        void   Detach() NN_NOEXCEPT;

        Result GetXferReport(uint32_t     *pOutCount,
                             XferReport   *pOutReport,
                             uint32_t      count) NN_NOEXCEPT;

    private:
        nn::os::Mutex                                   m_Mutex;
        nn::os::NativeHandle                            m_Handle;
        nn::sf::SharedPointer<hs::IClientEpSession>&    m_EpSession;
    };

    HostInterface*                                 m_pIf;
    nn::sf::SharedPointer<hs::IClientEpSession>    m_pHandle;
    UsbEndpointDescriptor                          m_EndpointDescriptor;
    UsbEndpointType                                m_EpType;
    EndpointNumber                                 m_EndpointNumber;
    UsbEndpointDirection                           m_EpDirection;
    bool                                           m_IsInitialized;
    ::std::atomic<int>                             m_RefCount;
    nn::os::SystemEventType                        m_CompletionEvent;

    ReportRing                                     m_ReportRing;
    uint32_t                                       m_MaxUrbCount;
};


} // end of namespace usb
} // end of namespace nn
