﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   UART ドライバライブラリの開発用途限定のポートアクセス機能の API 宣言。
 */

#pragma once

#include <nn/nn_Common.h>

#include "uart_PortTypes.h"

namespace nn {
namespace uart {

//! @name ポート情報の取得
//! @{

/**
 * @brief   指定したポートが、ターゲット環境上で使用可能かを取得します。
 *
 * @param[in]   portIndex   確認対象のポート番号
 *
 * @return  ポートが使用可能かを返します。
 *
 * @details
 *  本 API は開発用途限定の API となります。
 *  製品で使用する場合は @ref nn::uart::HasPort() を使用してください。@n
 *  ポートの指定が識別子でなくポート番号である以外の動作は @ref nn::uart::HasPort() と同様です。
 *
 * @see
 *  nn::uart::HasPort()
 */
bool HasPortForDev(int portIndex) NN_NOEXCEPT;

/**
 * @brief   指定したポートが、指定したボーレートをサポートしているかを取得します。
 *
 * @param[in]   portIndex   ポート番号
 * @param[in]   baudRate    ボーレート
 *
 * @return  サポート有無を返します。
 *
 * @details
 *  本 API は開発用途限定の API となります。
 *  製品で使用する場合は @ref nn::uart::IsSupportedBaudRate() を使用してください。@n
 *  ポートの指定が識別子でなくポート番号である以外の動作は @ref nn::uart::IsSupportedBaudRate() と同様です。
 *
 * @see
 *  nn::uart::IsSupportedBaudRate()
 */
bool IsSupportedBaudRateForDev(int portIndex, BaudRate baudRate) NN_NOEXCEPT;

/**
 * @brief   指定したポートが、指定したフロー制御モードをサポートしているかを取得します。
 *
 * @param[in]   portIndex       ポート番号
 * @param[in]   flowControlMode フロー制御モード
 *
 * @return  サポート有無を返します。
 *
 * @details
 *  本 API は開発用途限定の API となります。
 *  製品で使用する場合は @ref nn::uart::IsSupportedFlowControlMode() を使用してください。@n
 *  ポートの指定が識別子でなくポート番号である以外の動作は @ref nn::uart::IsSupportedFlowControlMode() と同様です。
 *
 * @see
 *  nn::uart::IsSupportedFlowControlMode()
 */
bool IsSupportedFlowControlModeForDev(int portIndex, FlowControlMode flowControlMode) NN_NOEXCEPT;

/**
 * @brief   指定したポートが、指定したポートイベントをサポートしているかを取得します。
 *
 * @param[in]   portIndex       ポート番号
 * @param[in]   portEvent       ポートイベント
 *
 * @return  サポート有無を返します。
 *
 * @details
 *  本 API は開発用途限定の API となります。
 *  製品で使用する場合は @ref nn::uart::IsSupportedPortEvent() を使用してください。@n
 *  ポートの指定が識別子でなくポート番号である以外の動作は @ref nn::uart::IsSupportedPortEvent() と同様です。
 *
 * @see
 *  nn::uart::IsSupportedPortEvent()
 */
bool IsSupportedPortEventForDev(int portIndex, PortEventType portEvent) NN_NOEXCEPT;

//! @}

//! @name ポートアクセス機能
//! @{

/**
 * @brief   開発目的で使用するための特定のポートを直接指定して、ポートをオープンします。
 *
 * @param[out]  pOutSession オープンされたポートセッション
 * @param[in]   portIndex   ポート番号
 * @param[in]   portConfig  ポート設定
 *
 * @return      処理の結果を返します。
 *
 * @details
 *  本 API は開発用途限定の API となります。
 *  製品で使用する場合は @ref nn::uart::OpenPort() を使用してください。@n
 *  ポートの指定が識別子でなくポート番号である以外の動作は @ref nn::uart::OpenPort() と同様です。
 *
 * @see
 *  nn::uart::OpenPort()
 */
bool OpenPortForDev(PortSession* pOutSession, int portIndex, const PortConfigType& portConfig) NN_NOEXCEPT;

//! @}

} // uart
} // nn


