﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   UART ライブラリのポートアクセス機能の API 宣言。
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/os/os_SystemEvent.h>

#include "uart_PortTypes.h"
#include "uart_PortName.h"
#include "uart_Result.h"

namespace nn {
namespace uart {

struct PortSession
{
    void* _handle;
    nn::os::SystemEventType* _sendBufferEmptyEvent;
    nn::os::SystemEventType* _sendBufferReadyEvent;
    nn::os::SystemEventType* _receiveBufferReadyEvent;
    nn::os::SystemEventType* _receiveEndEvent;
};


//! @name ポート情報の取得
//! @{

/**
 * @brief   指定したポート識別子が、ターゲット環境上で使用可能かを取得します。
 *
 * @param[in]   name        確認対象のポート識別子
 *
 * @return  ポートが使用可能かを返します。
 * @retval  true    指定したポートは使用可能です。
 * @retval  false   指定したポートは存在しないか、または使用できません。
 *
 * @pre
 *  ライブラリは初期化済の状態である必要があります。
 *
 * @details
 *  指定したポート識別子が、ターゲット環境上で使用可能かを取得します。@n
 *  本関数はターゲット環境において対応するバスインタフェースが実装されているかを取得するもので、
 *  ポートの対向にデバイスが接続されているかどうかは検知しません。
 */
bool HasPort(PortName name) NN_NOEXCEPT;

/**
 * @brief   指定したポートが、指定したボーレートをサポートしているかを取得します。
 *
 * @param[in]   name        ポート識別子
 * @param[in]   baudRate    ボーレート
 *
 * @return  サポート有無を返します。
 * @retval  true    指定したポートは、指定したボーレートでの動作をサポートしています。
 * @retval  false   指定したポートは、指定したボーレートでの動作をサポートしていません。
 *
 * @pre
 *  ライブラリは初期化済の状態である必要があります。@n
 *  指定したポートは存在している必要があります。
 */
bool IsSupportedBaudRate(PortName name, BaudRate baudRate) NN_NOEXCEPT;

/**
 * @brief   指定したポートが、指定したフロー制御モードをサポートしているかを取得します。
 *
 * @param[in]   name            ポート識別子
 * @param[in]   flowControlMode フロー制御モード
 *
 * @return  サポート有無を返します。
 * @retval  true    指定したポートは、指定したフロー制御モードでの動作をサポートしています。
 * @retval  false   指定したポートは、指定したフロー制御モードでの動作をサポートしていません。
 *
 * @pre
 *  ライブラリは初期化済の状態である必要があります。@n
 *  指定したポートは存在している必要があります。
 */
bool IsSupportedFlowControlMode(PortName name, FlowControlMode flowControlMode) NN_NOEXCEPT;

/**
 * @brief   指定したポートが、指定したポートイベントをサポートしているかを取得します。
 *
 * @param[in]   name            ポート識別子
 * @param[in]   portEvent       ポートイベント
 *
 * @return  サポート有無を返します。
 * @retval  true    指定したポートは、指定したポートイベントをサポートしています。
 * @retval  false   指定したポートは、指定したポートイベントをサポートしていません。
 *
 * @pre
 *  ライブラリは初期化済の状態である必要があります。@n
 *  指定したポートは存在している必要があります。
 */
bool IsSupportedPortEvent(PortName name, PortEventType portEvent) NN_NOEXCEPT;

//! @}

//! @name ポートの動作設定
//! @{

/**
 * @brief   ポート設定オブジェクトを初期化します。
 *
 * @param[out]  pOutPortConfig      ポート設定オブジェクト
 * @param[in]   baudRate            ボーレート値
 * @param[in]   sendBuffer          送信バッファへのポインタ
 * @param[in]   sendBufferLength    送信バッファサイズ [byte]
 * @param[in]   receiveBuffer       受信バッファへのポインタ
 * @param[in]   receiveBufferLength 受信バッファサイズ [byte]
 *
 * @pre
 *  送受信バッファのアドレスおよびサイズはともに @ref nn::os::MemoryPageSize の整数倍である必要があります。
 *
 * @details
 *  ポート設定オブジェクトを初期化します。@n
 *  @n
 *  ポート設定オブジェクトが持つ設定パラメータには、明示的な指定が必須なものと任意であるものがありますが、
 *  この初期化関数は、引数にて明示的な指定が必須なすべてのパラメータの指定を要求するため、
 *  初期化が完了した時点で @ref nn::uart::OpenPort() に渡すことができます。@n
 *  @n
 *  指定が任意であるパラメータは、本関数内ではデフォルト値で初期化され、必要な場合は別途対応する関数で設定します。
 *  そのようなパラメータとデフォルト値、および設定用の関数は以下の通りです。
 *
 *  @li フロー制御モード : デフォルト値 @ref FlowControlMode_None,
 *                         設定用関数 @ref nn::uart::SetPortConfigFlowControlMode()
 *
 *  送信バッファは、@ref nn::uart::Send() で送信予約されたデータを一時的にバッファリングする領域です。@n
 *  受信バッファは、@ref nn::uart::Receive() で取り出されるまで受信済データを一時的にバッファリングする領域です。@n
 *  @n
 *  送受信バッファのアドレスおよびサイズはともに @ref nn::os::MemoryPageSize の整数倍である必要があります。@n
 *  送受信バッファのサイズは、想定スループットや関数の呼び出し頻度を考慮して十分な大きさを指定してください。@n
 *  @n
 *  少なくとも一方のバッファには上記の条件を満たすアドレスとサイズを設定してください。@n
 *  使用しない通信方向のバッファのサイズには 0 を指定できます。その場合、対応するアドレスに有効なポインタを渡す
 *  必要はありません。
 *
 * @see
 *  nn::uart::PortConfig, nn::uart::PortConfigType, nn::uart::SetPortConfigFlowControlMode()
 */
void InitializePortConfig(PortConfigType* pOutPortConfig, BaudRate baudRate, char* sendBuffer, size_t sendBufferLength, char* receiveBuffer, size_t receiveBufferLength) NN_NOEXCEPT;

/**
 * @brief   ポート設定オブジェクトに対しフロー制御モードを設定します。
 *
 * @param[in]   pPortConfig         ポート設定オブジェクト
 * @param[in]   flowControlMode     フロー制御モード
 *
 * @details
 *  ポート設定オブジェクトに対しフロー制御モードを設定します。
 */
void SetPortConfigFlowControlMode(PortConfigType* pPortConfig, FlowControlMode flowControlMode) NN_NOEXCEPT;

/**
* @brief   ポート設定オブジェクトに対し各ピンの Invert 状態を設定します。
*
* @param[in]   pPortConfig         ポート設定オブジェクト
* @param[in]   isInvertTx          TX の Invert 設定
* @param[in]   isInvertRx          RX の Invert 設定
* @param[in]   isInvertRts         RTS の Invert 設定
* @param[in]   isInvertCts         CTS の Invert 設定
*
* @details
*  ポート設定オブジェクトに対し各ピンの Invert 状態を設定します。
*/
void SetPortConfigInvertPins(PortConfigType* pPortConfig, bool isInvertTx, bool isInvertRx, bool isInvertRts, bool isInvertCts) NN_NOEXCEPT;

//! @}

//! @name ポートアクセス機能
//! @{

/**
 * @brief   指定したポートをオープンします。
 *
 * @param[out]  pOutSession オープンされたポートセッション
 * @param[in]   name        ポート識別子
 * @param[in]   portConfig  ポート設定
 *
 * @return      処理の結果を返します。
 * @retval      true        指定したポートはオープンされ、セッションの取得に成功しました。
 * @retval      false       指定したポートは既にオープンされており、セッションを取得できませんでした。
 *
 * @pre
 *  ライブラリは初期化済の状態である必要があります。@n
 *  指定したポートは存在している必要があります。@n
 *  @a pOutSession はアクセス可能なアドレスを指している必要があります。@n
 *  @a portConfig には有効な値が設定されている必要があります。
 *  有効値の条件については @ref nn::uart::InitializePortConfig() を参照してください。
 *
 * @post
 *  true を返した場合、指定したポートがアクセス可能な状態になります。@n
 *  また、@a pOutSession に有効なセッションが格納されます。
 *
 * @details
 *  指定したポートをオープンします。すなわちポートを初期化し、アクセス可能な状態にします。
 *  オープン済のポートに対して実行した場合、false が返ります。
 *  @a pOutSession には無効なセッション値が格納されます。
 *
 * @see
 *  nn::uart::InitializePortConfig()
 */
bool OpenPort(PortSession* pOutSession, PortName name, const PortConfigType& portConfig) NN_NOEXCEPT;

/**
 * @brief   送信バッファに追記可能なデータのサイズを返します。
 *
 * @param[in]   pSession    ポートセッション
 *
 * @return      送信バッファに追記可能なデータのサイズ
 *
 * @pre
 *  ライブラリは初期化済の状態である必要があります。@n
 *  @a pSession は有効なセッションである必要があります。
 *
 * @details
 *  送信バッファに追記可能なデータのサイズを返します。
 */
size_t GetWritableLength(const PortSession* pSession) NN_NOEXCEPT;

/**
 * @brief   指定したデータ列を送信バッファに書き込みます。
 *
 * @param[out]  pOutDoneBytes   バッファに格納された送信予約済データ列の長さ [bytes]
 * @param[in]   pSession        ポートセッション
 * @param[in]   data            送信するデータ列
 * @param[in]   dataBytes       送信するデータ列の長さ [bytes]
 *
 * @pre
 *  ライブラリは初期化済の状態である必要があります。@n
 *  @a pSession は有効なセッションである必要があります。
 *  @ref nn::uart::OpenPort() 呼び出し時に渡した @ref nn::uart::PortConfigType オブジェクトで
 *  送信バッファが指定されている必要があります。
 *
 * @post
 *  @a pOutDoneBytes に返した分だけのデータ列が送信バッファに追記された状態になります。@n
 *  @ref nn::uart::GetWritableLength() が返す追記可能データサイズが、@a pOutDoneBytes に返した分だけ減少します。
 *
 * @details
 *  指定したバッファ上のデータを、送信バッファに、最大で @a dataBytes で指定した長さだけコピーします。@n
 *  @a pOutDoneBytes に、コピーされた送信予約済データ列のサイズを格納します。@n
 *  送信予約済データ列は送信バッファに格納されるため、コピーされるサイズは送信バッファのサイズを超えません。@n
 *  この関数はポートのデータ送信完了を待つことはなく、送信バッファにコピー可能な分をコピーするとすぐに返ります。@n
 *  送信バッファがいっぱいの場合には、@a pOutDoneBytes には 0 がセットされて返ります。
 */
void Send(size_t* pOutDoneBytes, PortSession* pSession, const void* data, size_t dataBytes) NN_NOEXCEPT;

/**
 * @brief   受信バッファに溜まっている受信済データ列のサイズを返します。
 *
 * @param[in]   pSession    ポートセッション
 *
 * @return      受信バッファに溜まっている受信済データ列のサイズ
 *
 * @pre
 *  ライブラリは初期化済の状態である必要があります。@n
 *  @a pSession は有効なセッションである必要があります。
 *
 * @details
 *  受信バッファに溜まっている受信済データ列のサイズを返します。
 */
size_t GetReadableLength(const PortSession* pSession) NN_NOEXCEPT;

/**
 * @brief   受信済データを読み出します。
 *
 * @param[out]  pOutDoneBytes バッファに格納された受信済データ列の長さ [bytes]
 * @param[out]  outData       受信済データを格納するバッファ。
 *                            dataBytes で指定した以上のサイズで確保されている必要があります。
 * @param[in]   dataBytes     読み出す受信済データ列の最大長 [bytes]
 * @param[in]   pSession      ポートセッション
 *
 * @return      処理の結果を返します。
 * @retval      ResultSuccess           受信済データ列に問題は検出されず、指定したバッファに書き込まれました。
 * @retval      ResultHardwareOverrun   最後の受信 API 呼び出し以降、
 *                                      ハードウェア FIFO 上で受信データのオーバーランが検出されました。
 * @retval      ResultOutOfBuffer       最後の受信 API 呼び出し以降、受信バッファの溢れが発生しました。
 * @retval      ResultFrameError        フレームエラーによる受信データの化けが検出されました。
 *
 * @pre
 *  ライブラリは初期化済の状態である必要があります。@n
 *  @a pSession は有効なセッションである必要があります。@n
 *  @ref nn::uart::OpenPort() 呼び出し時に渡した @ref nn::uart::PortConfigType オブジェクトで
 *  受信バッファが指定されている必要があります。
 *
 * @post
 *  @a pOutDoneBytes に返した分だけの受信済データ列が指定したバッファに読み出された状態になります。@n
 *  @ref nn::uart::GetReadableLength() が返す受信済データサイズが @a pOutDoneBytes に返した分だけ減少します。
 *
 * @details
 *  受信済データ列を、指定したバッファに、最大で @a dataBytes で指定した長さだけコピーします。@n
 *  @a pOutDoneBytes に、コピーされた受信済データ列のサイズを格納します。@n
 *  受信済データは受信バッファに格納されているため、コピーされるサイズは受信バッファのサイズを超えません。@n
 *  この関数はポートのデータ受信を待つことはなく、受信バッファからコピー可能な分をコピーするとすぐに返ります。@n
 *  受信バッファが空の場合には、 @a pOutDoneBytes には 0 がセットされ、関数は成功します。@n
 *  エラー発生時には @a pOutDoneBytes には有効な値は入らず、受信バッファの中身は破棄されます。
 */
nn::Result Receive(size_t* pOutDoneBytes, void* outData, size_t dataBytes, PortSession* pSession) NN_NOEXCEPT;

/**
 * @brief   特定の状態変化に対してシステムイベントを関連づけます。
 *
 * @param[in]   pEvent      状態変化が発生した際にシグナル化するシステムイベント
 * @param[in]   pSession    ポートセッション
 * @param[in]   eventType   関連づける状態変化の種類
 * @param[in]   threshold   シグナル化を起こす閾値
 *
 * @return      処理の結果を返します。
 * @retval      true        関連づけに成功しました。
 * @retval      false       指定したセッションの指定した状態変化には、すでにシステムイベントが関連づけられています。
 *
 * @pre
 *  ライブラリは初期化済の状態である必要があります。@n
 *  @a pSession は有効なセッションである必要があります。@n
 *  @a pEvent が未初期化状態である必要があります。@n
 *  送信に関する状態変化に関連づける場合、@ref nn::uart::OpenPort() 呼び出し時に渡した @ref nn::uart::PortConfigType オブジェクトで送信バッファが指定されている必要があります。@n
 *  受信に関する状態変化に関連づける場合、@ref nn::uart::OpenPort() 呼び出し時に渡した @ref nn::uart::PortConfigType オブジェクトで受信バッファが指定されている必要があります。@n
 *  @a eventType に @ref PortEventType_SendBufferReady を指定した場合、@a threshold が 1 以上送信バッファサイズ以下である必要があります。@n
 *  @a eventType に @ref PortEventType_ReceiveBufferReady を指定した場合、@a threshold が 1 以上受信バッファサイズ以下である必要があります。
 *
 * @post
 *  @a pEvent が初期化状態になります。@n
 *  @a pSession で指定したポートにおける @a eventType で指定した状態変化が、@a pEvent に関連づけられた状態になります。
 *
 * @details
 *  @a pSession で指定したセッションに関する特定の状態変化に対してシステムイベントを関連づけます。@n
 *  関連づける状態変化の種類は @a eventType で指定します。@n
 *  各ポートの各状態変化の種類ごとに、システムイベントは 1 つまで指定できます。@n
 *
 *  | @a eventType の指定                   | @a threshold の意味                                                    |
 *  | ------------------------------------- | ---------------------------------------------------------------------- |
 *  | @ref PortEventType_SendBufferEmpty    | なし。値は無視されます。                                               |
 *  | @ref PortEventType_SendBufferReady    | 送信バッファの空きが @a threshold バイト以上になるとシグナル化します。 |
 *  | @ref PortEventType_ReceiveBufferReady | 受信済データが @a threshold バイト以上になるとシグナル化します。       |
 *  | @ref PortEventType_ReceiveEnd         | プラットフォームによって異なります。
 *
 * @platformbegin{NX}
 *  @ref PortEventType_ReceiveEnd を指定した場合の閾値は固定であり、@a threshold は無視されます。
 *  詳細は @ref PortEventType を参照してください。
 *
 * @platformend
 *
 * @see
 *  nn::uart::UnbindPortEvent()
 */
bool BindPortEvent(nn::os::SystemEventType* pEvent, PortSession* pSession, PortEventType eventType, size_t threshold) NN_NOEXCEPT;

/**
 * @brief   指定したポートの状態変化とシステムイベントとの関連づけを解除します。
 *
 * @param[in]   pEvent      システムイベント
 * @param[in]   pSession    ポートセッション
 *
 * @return      処理の結果を返します。
 * @retval      true        1 つ以上の状態変化とシステムイベントの関連づけを解除しました。
 * @retval      false       指定したポートに、指定したシステムイベントが関連づけられた状態変化はありませんでした。
 *
 * @pre
 *  ライブラリは初期化済の状態である必要があります。@n
 *
 * @post
 *  @a pSession で指定したポートのすべての状態変化について @a pEvent との関連づけがされていない状態になります。
 *
 * @details
 *  指定したポートの状態変化のうち、指定したシステムイベントに関連づけられたものをすべて解除します。
 *
 * @see
 *  nn::uart::BindPortEvent()
 */
bool UnbindPortEvent(nn::os::SystemEventType* pEvent, PortSession* pSession) NN_NOEXCEPT;

/**
 * @brief   指定したポートをクローズします。
 *
 * @param[in]   pSession    ポートセッション
 *
 * @pre
 *  ライブラリは初期化済の状態である必要があります。@n
 *  @a pSession は有効なセッションである必要があります。
 *
 * @post
 *  指定したポートがクローズ状態になります。
 *
 * @details
 *  指定したポートをクローズします。@n
 *  送信予約されているがまだ送信されていないデータについては失われる可能性があります。
 */
void ClosePort(PortSession* pSession) NN_NOEXCEPT;

//! @}

} // uart
} // nn


