﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   UART ライブラリのポートアクセス機能の公開ヘッダ。
 */

#pragma once

#include "uart_PortTypes.h"
#include "uart_PortName.h"
#include "uart_PortApi.h"

namespace nn {
namespace uart {

/**
 * @brief ポート設定オブジェクトを保持するクラスです。@n
 *
 * @details
 *  ポート設定オブジェクト @ref nn::uart::PortConfigType をラップし、パラメータ設定機能を付加したクラスです。@n
 *
 *  このクラスは、コンストラクタの引数において明示的な指定が必須なすべてのパラメータの指定を要求するため、
 *  生成された時点で必須要件を満たすことが保証されています。@n
 *
 *  指定がオプションであるパラメータについては、生成後に対応するセッター関数を呼び出すことで設定します。
 *
 * @see
 *  nn::uart::PortConfigType()
 */
class PortConfig
{
public:
    /**
     * @brief   ポート設定オブジェクトを構築し初期化します。
     *
     * @param[in]   baudRate            ボーレート値
     * @param[in]   sendBuffer          送信バッファへのポインタ
     * @param[in]   sendBufferLength    送信バッファサイズ [byte]
     * @param[in]   receiveBuffer       受信バッファへのポインタ
     * @param[in]   receiveBufferLength 受信バッファサイズ [byte]
     *
     * @details
     *  ポート設定オブジェクトを構築し初期化します。@n
     *  引数で指定されないポート設定パラメータについてはデフォルト値で初期化します。
     *
     * @see
     *  nn::uart::InitializePortConfig()
     */
    PortConfig(
        BaudRate baudRate,
        char* sendBuffer, size_t sendBufferLength,
        char* receiveBuffer, size_t receiveBufferLength) NN_NOEXCEPT
    {
        InitializePortConfig(&m_Body, baudRate, sendBuffer, sendBufferLength,
                             receiveBuffer, receiveBufferLength);
    }

    /**
     * @brief   フロー制御モードを設定します。
     *
     * @param[in]   flowControlMode    フロー制御モード
     */
    void SetFlowControlMode(FlowControlMode flowControlMode) NN_NOEXCEPT
    {
        SetPortConfigFlowControlMode(&m_Body, flowControlMode);
    }

    /**
     * @brief   PortConfigType オブジェクトへの参照を返します。
     *
     * @return  PortConfigType オブジェクトへの参照
     *
     * @details
     *  自インスタンスが持つ nn::uart::PortConfigType オブジェクトへの参照を返します。
     *  この変換演算子があるため、nn::uart::PortConfigType& を引数にとる関数に対して、
     *  PortConfig& を指定することが可能です。
     *
     * @see
     *  nn::uart::PortConfigType
     */
    NN_IMPLICIT operator PortConfigType&() NN_NOEXCEPT
    {
        return m_Body;
    }

    /**
     * @brief   PortConfigType オブジェクトへの const 参照を返します。
     *
     * @return  PortConfigType オブジェクトへの const 参照
     *
     * @details
     *  自インスタンスが持つ PortConfigType オブジェクトへの const 参照を返します。
     *  この変換演算子があるため、const PortConfigType& を引数にとる関数に対して、
     *  PortConfig& を指定することが可能です。
     *
     * @see
     *  nn::uart::PortConfigType
     */
    NN_IMPLICIT operator const PortConfigType&() const NN_NOEXCEPT
    {
        return m_Body;
    }

private:
    PortConfigType  m_Body;
};

} // uart
} // nn
