﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

/**
 * @file
 * @brief   温度センサーライブラリの温度検知機能に関する API の宣言
 */

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/os/os_SystemEvent.h>

#include <nn/ts/ts_Types.h>
#include <nn/ts/ts_Location.h>

namespace nn {
namespace ts {

//! @name 温度検知機能の設定
//! @{

/**
* @brief    セッション用パラメータです。
*
* @details  温度検知に関するセッション用パラメータです。
*           内部の変数に直接アクセスしないでください。
*/
struct DetectionSession
{
    void* _sessionHandle;
};


/**
 * @brief       指定する計測位置の温度検知に関するセッションをオープンします。
 *
 * @param[out]  pOutSession     セッション用パラメータ
 * @param[in]   location        対象となる計測位置
 *
 * @return      セッションが正常にオープンできたかどうかを返します。
 * @retval      true        指定した計測位置に対するセッションのオープン成功しました。
 * @retval      false       指定した計測位置に対するセッションは既にオープンされており、セッションを取得できませんでした。
 *
 * @pre
 *      初期化済み状態である必要があります。@n
 *      @a pOutSession はアクセス可能なアドレスを指している必要があります。@n
 *      @a location に対するセッションは他のプロセスからオープンされていない必要があります。
 *
 * @post
 *      true を返した場合、@a location に対する温度検知が可能な状態になります。
 *      また、@a pOutSession に必要な情報が格納されます。
 *
 * @details
 *      指定した計測位置に対する温度検知のセッションをオープンし、温度検知に関する制御を可能にします。
 */
bool OpenSession(DetectionSession* pOutSession, Location location) NN_NOEXCEPT;


/**
 * @brief       指定する計測位置の温度検知に関するセッションをクローズします。
 *
 * @param[in]   pSession    セッション用パラメータ
 *
 * @pre
 *      初期化済み状態である必要があります。@n
 *      紐付いている SystemEvent がある場合は待機しているスレッドがいない状態で実行してください。
 *
 * @post
 *      指定したセッションはクローズされ、温度検知に関する制御ができなくなります。@n
 *      紐付けられていた SystemEvent は破棄され、紐付けが解除されます。
 *
 * @details
 *      @a pSession で指定された計測位置に対する温度検知の制御を終了します。
 */
void CloseSession(DetectionSession* pSession) NN_NOEXCEPT;


/**
 * @brief       検知条件に登録できる LimitTemperature のサイズを取得します。
 *
 * @param[in]   pSession           セッション用パラメータ
 *
 * @return      検知条件に登録できる LimitTemperature のサイズを返します。
 *
 * @pre
 *      初期化済み状態である必要があります。
 *
 * @details
 *      @a pSession で指定するセッションの検知条件に登録できる LimitTemperature のサイズを返します。
 */
int GetMaxDetectionSettingsSize(DetectionSession* pSession) NN_NOEXCEPT;


/**
 * @brief       温度検知を行う検知条件を取得します。
 *
 * @param[out]  pOutSettings    検知設定の配列
 * @param[in]   pSession        セッション用パラメータ
 * @param[in]   size            @a pOutSettings に収録する LimitTemperature の数
 *
 * @pre
 *      初期化済み状態である必要があります。
 *
 * @details
 *      温度検知を行う検知条件を取得します。@n
 *      取得可能な数の上限は @ref nn::ts::GetMaxDetectionSettingsSize() で取得できます。
 */
void GetDetectionSettings(LimitTemperature* pOutSettings, DetectionSession* pSession, int size) NN_NOEXCEPT;


/**
 * @brief       温度検知を行う検知条件を設定します
 *
 * @param[in]   pSession    セッション用パラメータ
 * @param[in]   limitList   設定する温度値の配列
 * @param[in]   size        @a limitList にセットされた LimitTemperature の数
 *
 * @return      登録に成功した検知対象となる LimitTemperature の数を返します。
 *
 * @pre
 *      初期化済み状態である必要があります。
 *
 * @details
 *      温度検知を行う検知条件を設定します。@n
 *      @a limitSettings の先頭から @a size の数を温度検知の条件として設定します。@n
 *      設定可能な数の上限は @ref nn::ts::GetMaxDetectionSettingsSize() で取得できます。
 *      @a size で設定可能な数以上の値を受け取った場合は、@a limitSettings の先頭から
 *      設定可能な上限の数だけを登録します。@n
 *      過去に設定されていた値はすべて削除されます。
 */
int SetDetectionSettings(DetectionSession* pSession, const LimitTemperature& limitSettings, int size) NN_NOEXCEPT;


/**
 * @brief       直近に本関数を実行した時に計測位置に対する温度検知が発生したかどうかを取得します。
 *
 * @param[in]   pSession    セッション用パラメータ
 *
 * @return      温度検知が発生したかどうかを返します。
 *
 * @pre
 *      初期化済み状態である必要があります。
 *
 * @details
 *      本関数が直前に実行されてから再度実行されるまでの間に温度検知が発生したかどうかを返します。
 *      本関数を実行した時点での検知状態ではありません。@n
 *      厳密に状態を検知するためには @ref nn::ts::BindDetectionEvent() でイベントオブジェクトを
 *      紐付けておくことを推奨します。@n
 *      @ref nn::ts::SetDetectionSettings() により温度検知の設定が行われていない場合は、
 *      必ず false を返します。
 */
ExceededStatus GetDetectedStatus(DetectionSession* pSession) NN_NOEXCEPT;


/**
 * @brief       温度検知による割り込みシグナルを受け取るためのイベントオブジェクトを紐付けます。
 *
 * @param[in]   pSession    セッション用パラメータ
 * @param[in]   pLowEvent   限度値に到達した際にシグナルするイベントオブジェクト
 *
 * @return       紐付け処理に対する結果を返します。
 * @retval       ResultSuccess        成功しました。
 * @retval       ResultAlreadyBound   pSession で指定した計測位置に対して既に SystemEvent が紐付けされています。
 *
 * @pre
 *      初期化済み状態である必要があります。
 *
 * @post
 *      @a pOutEvent で温度検知のシグナルを受け取ることが可能な状態になります。@n
 *      @ref nn::ts::SetDetectionSettings() で検知条件を設定している場合は、
 *      @ref nn::ts::SetDetectionEventEnabled() でイベントに対する検知の有効状態を設定可能です。
 *
 * @details
 *      設定した限度値を検知した場合に @a pOutEvent に紐付けた SystemEvent にシグナル状態にします。
 *      一つのセッションに対して紐付けられるイベントは一つまでです。
 */
nn::Result BindDetectionEvent(DetectionSession* pSession, nn::os::SystemEventType* pOutEvent) NN_NOEXCEPT;


/**
 * @brief       温度検知による割り込みシグナルを受け取るためのイベントオブジェクトの紐付けを解除します。
 *
 * @param[in]   pSession    セッション用パラメータ
 *
 * @pre
 *      初期化済み状態である必要があります。
 *
 * @post
 *      @a pSession に紐付けられたイベントオブジェクトが解除されます。
 */
void UnbindDetectionEvent(DetectionSession* pSession) NN_NOEXCEPT;


/**
 * @brief       温度検知による割り込みシグナル発行の有効状態を取得します。
 *
 * @param[in]   pSession    セッション用パラメータ
 *
 * @return      温度検知によるシグナルが発行されるかどうかを返します。
 * @retval      true    シグナルの発行が有効な状態
 * @retval      false   シグナルの発行が無効な状態
 *
 * @pre
 *      @ref nn::ts::BindDetectionEvent() でSystemEventの紐付けが行われている必要があります。
 *
 * @details
 *      温度検知イベントに対して、@ref nn::ts::BindDetectionEvent() で紐付けした
 *      SystemEventへのシグナル発行の有効状態を取得します。
 *      @ref nn::ts::SetDetectionSettings() で検知条件が設定されていない場合は。@n
 *      @ref nn::ts::BindDetectionEvent() で SystemEvent の紐付けが行われていない場合は、
 *      必ず false を返します。
 */
bool GetDetectionEventEnabled(DetectionSession* pSession) NN_NOEXCEPT;


/**
 * @brief       温度検知による割り込みシグナル発行の有効状態を設定します。
 *
 * @param[in]   pSession    セッション用パラメータ
 * @param[in]   enable      計測位置での検知に対する通知の有効状態
 *                          true    : 有効
 *                          false   : 無効
 *
 * @pre
 *      シグナル発行の有効状態が @a enable で与えた状態になります。@n
 *      @ref nn::ts::SetDetectionSettings() で検知条件が設定されている必要があります。@n
 *      @ref nn::ts::BindDetectionEvent() で SystemEvent の紐付けが行われている必要があります。
 *
 * @details
 *      温度検知イベントに対して、@ref nn::ts::BindDetectionEvent() で紐付けした
 *      SystemEvent へのシグナル発行の有効状態を設定します。
 *
 */
void SetDetectionEventEnabled(DetectionSession* pSession, bool enable) NN_NOEXCEPT;

//! @}

} // ts
} // nn

