﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

/**
 * @file
 * @brief   温度センサーライブラリ実装部の温度値の取得に関する API の宣言
 */

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/os/os_SystemEvent.h>

#include <nn/ts/ts_Types.h>
#include <nn/ts/ts_Location.h>

namespace nn {
namespace ts {
namespace impl {

//! @name 温度値の取得
//! @{

/**
 * @brief       指定する計測位置で計測および検知が可能な温度値の限度値を取得します。
 *
 * @param[in]   location    対象となる計測位置
 *
 * @return      計測および検知が可能な温度値の限度値を返します。
 *
 * @pre
 *      初期化済み状態である必要があります。
 *
 * @details
 *      @a location で指定する計測位置で計測可能な温度値の上限値と下限値を取得します。
 */
LimitTemperature GetTemperatureRange(Location location) NN_NOEXCEPT;


/**
 * @brief       指定する地点の温度を取得します。
 *
 * @param[out]  pOutTemperatute 温度値 [℃]
 * @param[in]   location        対象となる計測位置
 *
 * @return      処理の結果を返します。
 * @retval      nn::ResultSuccess
 *                  成功しました。
 *
 * @pre
 *      初期化済み状態である必要があります。
 *
 * @details
 *      @a location で指定する計測位置の温度を読み取ります。
 */
nn::Result GetTemperature(int* pOutTemperature, Location location) NN_NOEXCEPT;


/**
 * @brief       指定する地点の温度をミリ度単位で取得します。
 *
 * @param[out]  pOutTemperatuteMilliC 単位ミリ度の温度値
 * @param[in]   location              対象となる計測位置
 *
 * @return      処理の結果を返します。
 * @retval      nn::ResultSuccess
 *                  成功しました。
 * @pre
 *      初期化済み状態である必要があります。
 *
 * @details
 *      @a location で指定する計測位置の温度を読み取ります。
 */
nn::Result GetTemperatureMilliC(int* pOutTemperatureMilliC, Location location) NN_NOEXCEPT;

//! @}

/**
 * @brief       温度センサーに計測モードの変更を通知します。
 *
 * @param[in]   location        対象となる計測位置
 * @param[in]   measurementMode 計測モード
 *
 * @details     指定した計測位置の温度センサーに計測モードの変更を通知します。
 */
void SetMeasurementMode(Location location, MeasurementMode measurementMode) NN_NOEXCEPT;

} // impl
} // ts
} // nn



