﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/timesrv/detail/tz/timesrv_DaylightSavingTimeType.h>
#include <nn/timesrv/timesrv_TimeZoneRuleInner.h>

#include <nn/time/time_CalendarTime.h>
#include <nn/time/time_CalendarAdditionalInfo.h>
#include <nn/time/time_PosixTime.h>

#include <nn/timesrv/detail/service/timesrv_ITimeZoneService.sfdl.h> // nn::time::sf::CalendarAdditionalInfo

namespace nn { namespace timesrv { namespace detail { namespace tz {

nn::Result ParseTimeZoneBinary(
    TimeZoneRuleInner* pOut,
    const char *pBinary,
    size_t size) NN_NOEXCEPT;

nn::Result ToCalendarTime(
    nn::time::CalendarTime* pOutCalendar,
    nn::time::sf::CalendarAdditionalInfo* pOutCalendarAdditionalInfo,
    const nn::time::PosixTime& posixTime,
    const TimeZoneRuleInner& rule) NN_NOEXCEPT;

/**
 * @brief   現地時刻から絶対時刻へ変換します。
 * @param[out]  pOutCount               pOutPosixTimeList に格納された要素数
 * @param[out]  pOutPosixTimeList      絶対時刻
 * @param[in]   count                   pOutPosixTimeList の要素数
 * @param[in]   calendarTime            現地時刻
 * @param[in]   rule                    タイムゾーンルール
 *
 * @return  処理結果
 * @retval  ResultSuccess           成功
 * @retval  ResultOverflowed        内部計算でオーバーフローまたはアンダーフローが発生
 * @retval  ResultInvalidArgument   タイムゾーンルールが構築済でない
 *
 * @pre
 *  - pOutCount != nullptr
 *  - pOutPosixTimeList != nullptr
 *  - count > 0
 *
 * @post
 *  - 0 <= *pOutCount <= 2
 *  - *pOutCount == 2 の場合、 pOutPosixTimeList[0] < pOutPosixTimeList[1]
 *
 * @details
 *  入力する現地時刻によって、本 API は次の通り 0 個から 2 個の絶対時刻を返します。
 *  | 入力の現地時刻のケース | 動作説明 |
 *  |:---------------------- |:-------- |
 *  | 1度だけ訪れる日時を指定した場合 | *pOutCount == 1 になります。 |
 *  | 時差の切り替わりによって時刻が巻き戻り、重複して2度訪れる日時を指定した場合 | *pOutCount == 2 になります。 @n pOutPosixTimeList[0] が時差の切り替わり前、 pOutPosixTimeList[1] が時差の切り替わり後の絶対時刻です。 |
 *  | 時差の切り替わりによって時刻が進み、スキップされて存在しない日時を指定した場合 | *pOutCount == 0 になります。 |
 *
 *  入力する現地時刻の夏時間の有効性が分からない場合には、@ref DaylightSavingTimeType_Unknown を指定することで、
 *  入力された日時から夏時間の有効性が適切に判定され、絶対時刻が計算されます。
 *  @n
 *  以下に例を示します。
 *  | 入力の現地時刻の例 | 日時から求まる本来の夏時間の有効性 | 絶対時刻の計算に使われる補正された現在時刻 |
 *  |:------------------ |:---------------------------------- |:------------------------------------------ |
 *  | 2016-01-15 18:00:00 , @ref DaylightSavingTimeType_Unknown | 無効 | 2016-01-15 18:00:00 , @ref DaylightSavingTimeType_NoEffect |
 *  | 2016-07-15 18:00:00 , @ref DaylightSavingTimeType_Unknown | 有効 | 2016-07-15 18:00:00 , @ref DaylightSavingTimeType_InEffect |
 *  | 2016-11-01 01:30:00 , @ref DaylightSavingTimeType_Unknown | 無効、有効のどちらにも存在する日時の場合 | 2016-11-01 01:30:00 , @ref DaylightSavingTimeType_NoEffect と@n 2016-11-01 01:30:00 , @ref DaylightSavingTimeType_InEffect の2つから、絶対時刻が2つ計算されます。 |
 *
 *  入力の calendarTime.DaylightSavingTimeType と、日時から求まる本来の夏時間の有効性が異なる場合には、
 *  時差を考慮した上で適切な夏時間の有効性に補正された日時を使って絶対時刻が計算されます。
 *  @n
 *  以下は、夏時間が有効な期間に時刻が1時間進むタイムゾーンでの例です。
 *  | 入力の現地時刻の例 | 日時から求まる本来の夏時間の有効性 | 絶対時刻の計算に使われる補正された現在時刻 |
 *  |:------------------ |:---------------------------------- |:------------------------------------------ |
 *  | 2016-01-15 18:00:00 , @ref DaylightSavingTimeType_InEffect | 無効 ( 入力と異なる ) | 2016-01-15 17:00:00 , @ref DaylightSavingTimeType_NoEffect @n (入力の日時が1時間進んでいると判断し、1時間さかのぼる) |
 *  | 2016-07-15 18:00:00 , @ref DaylightSavingTimeType_NoEffect | 有効 ( 入力と異なる ) | 2016-07-15 19:00:00 , @ref DaylightSavingTimeType_InEffect @n (入力の日時が1時間遅れていると判断し、1時間進む) |
 *
 */
nn::Result ToPosixTime(
    int* pOutCount,
    nn::time::PosixTime* pOutPosixtTimeList,
    size_t count,
    const nn::time::CalendarTime& calendarTime,
    const TimeZoneRuleInner& rule,
    DaylightSavingTimeType daylightSavingTimeType) NN_NOEXCEPT;

nn::Result AdjustCalendarTime(
    int* pOutCount,
    nn::time::CalendarTime* pOutCalendarTimeList,
    int count,
    const nn::time::CalendarTime& calendarTime,
    const TimeZoneRuleInner& rule) NN_NOEXCEPT;

}}}} // nn::timesrv::detail::tz
