﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/time/detail/time_CommonDetail.h>

#include <nn/timesrv/detail/core/timesrv_SystemClockCore.h>

namespace nn
{
namespace timesrv
{
namespace detail
{
namespace core
{

class SteadyClockCore;

/**
 * @brief   ユーザーシステム時計のコア実装
 * @details
 *  本クラスはスレッドアンセーフです。
 *
 *  自動補正有効かつ、ネットワーク時計が補正済である場合にはネットワーク時計を扱い、
 *  そうでない場合はローカル時計を扱います。
 *
 *  本クラスは設定永続化の処理をいっさい行わず、
 *  インスタンス生成時に、クラス内の設定値は無効な値を保持します。@n
 *  例えば IsAutomaticCorrectionEnabled() が有効な値を返すためには、
 *  事前に SetAutomaticCorrectionEnabled() で有効な値をセットしておく必要があります。
 */
class StandardUserSystemClockCore : public SystemClockCore
{
private:
    bool m_IsAutomaticCorrectionEnabled;
    mutable SystemClockCore* m_pStandardLocalSystemClockCore;
    SystemClockCore* m_pStandardNetworkSystemClockCore;
    nn::time::SteadyClockTimePoint m_AutomaticCorrectionUpdatedTime;

private:
    virtual Result GetSystemClockContextImpl(nn::time::SystemClockContext* pSystemClockContext) const NN_NOEXCEPT NN_OVERRIDE;
    virtual Result SetSystemClockContextImpl(const nn::time::SystemClockContext& systemClockContext) NN_NOEXCEPT NN_OVERRIDE;

    // ネットワーク時計との同期処理
    Result SyncNetworkClockCurrentTime() const NN_NOEXCEPT;

public:
    /**
     * @brief   コンストラクタ
     * @param[in]   pLocalClock     ローカル時計を指すポインタ
     * @param[in]   pNetworkClock   ネットワーク時計を指すポインタ
     *
     * @details
     *  IsAutomaticCorrectionEnabled() == true && pNetworkClock が補正済、
     *  である場合には pNetworkClock のコンテキストを扱い、
     *  そうでない場合は pLocalClock のコンテキストを扱います。
     */
    StandardUserSystemClockCore(
        SystemClockCore* pLocalClock, SystemClockCore* pNetworkClock) NN_NOEXCEPT;
    virtual ~StandardUserSystemClockCore() NN_NOEXCEPT;

    bool IsAutomaticCorrectionEnabled() const NN_NOEXCEPT; // 自動補正フラグの取得
    Result SetAutomaticCorrectionEnabled(bool value) NN_NOEXCEPT; // 自動補正フラグの設定

    nn::time::SteadyClockTimePoint GetAutomaticCorrectionUpdatedTime() const NN_NOEXCEPT; // 自動補正フラグ更新時間の変更
    void SetAutomaticCorrectionUpdatedTime(const nn::time::SteadyClockTimePoint& value) NN_NOEXCEPT; // 自動補正フラグ変更時間の取得
};

}
}
}
}
