﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
/**
*   @file
*   @brief  デバイス電源断で揮発する短命なネットワーク時計を表すクロックのクラス宣言
*/

#pragma once

#include <nn/time/time_Common.h>

#include <nn/time/time_Types.h>
#include <nn/time/time_SystemClockTraits.h>

#include <chrono>
#include <ctime>


namespace nn
{
namespace time
{

/**
 * @brief   デバイス電源断で揮発する短命なネットワーク時計を表すクロックです。
 *
 * @details
 *  このクロックは現在時刻を指し示し、人間の理解可能な日時表現に変換できる値を取得することができます。@n
 *  デバイスの電源断をまたぐと必ず未補正になり、デバイスの起動のたび、
 *  ネットワーク経由で時刻補正が行われることで初めて値を返せるようになります。
 *
 *  このクロックはシステムによる自動補正の影響を受け、
 *  指し示す日時が実際の時間経過以上に進んだり過去へ巻き戻ったりすることがあります。@n
 *  デバイスの電源が入っている限りは、このクロックが値を返せる状態から返せない状態へ遷移することはありません。
 *  逆に、値を返せない状態から返せる状態へは、アプリケーションが起動中であっても自動的に遷移することがあります。
 *
 *  このクラスは C++ 標準の TrivialClock の要件を満たしており、 std::chrono::system_clock に準ずる機能を提供します。
 *  ただし C++ 標準のインターフェースのみを利用している場合、時計が無効なあいだに取得した時刻は常に固定の値となります。
 *
 *  ### 時刻精度について
 *  このクロックの時間の進みには、システムチックと同等の誤差が生じます。
 *  ただし、この誤差は自動補正によって定期的に打ち消されるので、
 *  長期間ネットワークに接続できない場合を除き、時刻の誤差は一定範囲内に留まります。
 */
class EphemeralNetworkSystemClock
{
public:
    /**
     * @brief   クロックの指し示す現在時刻を取得します。
     *
     * @param[out]  pOutPosixTime   時刻を代入するバッファへのポインタを指定します。
     *
     * @return  処理の結果が返ります。事前条件を満たしていない場合、以下に列挙されていない失敗が返ることがあります。
     * @retval  ResultSuccess                       成功しました。
     * @retval  ResultClockInvalid                  時計が無効です。
     *
     * @pre
     *  - TIME ライブラリが初期化されている
     *  - pOutPosixTime != nullptr
     *
     * @details
     * @platformbegin{Windows}
     *  Windows 環境では本関数は必ず成功し、 PC 時計と同等の時刻が返されます。
     * @platformend
     */
    static Result GetCurrentTime(PosixTime* pOutPosixTime) NN_NOEXCEPT;

    /**
     * @brief   このクロックから取得・計算される時刻・時間の内部表現となる算術型を表します。
     *
     * @details
     */
    typedef SystemClockTraits::rep rep;

    /**
     * @brief   このクロックの持つ分解能を std::ratio 型で表したものです。
     *
     * @details
     *  1単位当たりの秒数で表されています。
     */
    typedef SystemClockTraits::period period;

    /**
     * @brief   経過時間を表す型です。
     *
     * @details
     */
    typedef SystemClockTraits::duration duration;

    /**
     * @brief   このクロックから取得される、時間の流れの中の一点を指す型です。
     *
     * @details
     */
    typedef SystemClockTraits::time_point time_point;

    /**
     * @brief   このクロックの単調増加性を表す値です。
     *
     * @details
     *  false です。
     *  システムによる自動補正により、このクロックが指す時刻は逆行する可能性があります。
     */
    static const bool is_steady = false;

    /**
     * @brief   このクロックの指し示す現在時刻を取得します。
     *
     * @return  このクロックが指し示す現在時刻を nn::time::EphemeralNetworkSystemClock::time_point 型で返します。
     *
     * @pre
     *  - TIME ライブラリが初期化されている
     *
     * @details
     *  時計が無効な場合、1970年1月1日0時0分0秒を表す nn::time::EphemeralNetworkSystemClock::time_point 型の値が返ります。
     *
     * @platformbegin{Windows}
     *  Windows 環境では PC 時計と同等の時刻が返されます。
     * @platformend
     */
    static time_point now() NN_NOEXCEPT;

    /**
     * @brief   nn::time::EphemeralNetworkSystemClock::time_point 型の時刻を std::time_t 型に変換します。
     *
     * @param[in]   t   nn::time::StandardUserSystemClock::time_point 型での時刻の値を指定します。
     *
     * @return  入力と同じ時刻を指す std::time_t 型の値が返ります。
     */
    static std::time_t to_time_t(const time_point& t) NN_NOEXCEPT;

    /**
     * @brief   std::time_t 型の時刻を nn::time::EphemeralNetworkSystemClock::time_point 型に変換します。
     *
     * @param[in]   t   std::time_t 型での時刻の値を指定します。
     *
     * @return  入力と同じ時刻を指す nn::time::EphemeralNetworkSystemClock::time_point 型の値が返ります。
     *
     * @details
     *  nn::time::EphemeralNetworkSystemClock::time_point 型の時刻の分解能が std::time_t 型のそれより低い場合、 nn::time::EphemeralNetworkSystemClock::time_point 型の分解能に丸められます。
     */
    static time_point from_time_t(std::time_t t) NN_NOEXCEPT;
};

}
}
