﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
*   @file
*   @brief  修理向け API の宣言
*/

#pragma once

#include <nn/time.h>
#include <nn/nn_TimeSpan.h>

namespace nn
{
namespace time
{

//! @name 修理向け API
//! @{

/**
 * @brief   修理向けの TIME ライブラリの初期化を行います。
 *
 * @return  処理の結果が返ります。
 * @retval  ResultSuccess                       成功しました。
 *
 * @details
 *  本関数を使って TIME ライブラリを初期化した場合にも、破棄には nn::time::Finalize を利用してください。
 *  他の権限向けの初期化を行う @ref Initialize, @ref InitializeForMenu 関数と重ねて呼ぶことはできません。
 *
 *  ライブラリがすでに初期化された状態でも重ねて呼ぶことができますが、その場合、ライブラリの利用を完全に終了するにはこの関数を呼んだのと同じ回数だけ
 *  @ref nn::time::Finalize 関数を呼ぶ必要があります。
 *
 *  この関数を成功させるには、特権が必要です。
 */
Result InitializeForRepair() NN_NOEXCEPT;

/**
 * @brief   内部オフセットを使ってシステムクロックを補正します。
 *
 * @return  処理の結果が返ります。
 * @retval  ResultSuccess                       成功しました。
 * @retval  ResultClockInvalid                  nn::time::StandardNetworkSystemClock が利用できません
 * @retval  ResultNetworkRequestNotAccepted     ネットワーク利用要求が受理されていません。
 * @retval  上記以外                            失敗しました
 *
 * @pre
 *  - nn::time::InitializeForRepair() でライブラリが初期化されている
 *
 * @details
 *  - 本関数はサーバーから時刻をダウンロードするため、処理完了までブロックします。
 *  - 事前に NIFM ライブラリでネットワーク利用要求が受理された状態で利用してください。
 *  - 各システムクロックのコンテキスト( nn::time::SystemClockContext ) を変更することなく時刻を操作します。
 */
Result CalibrateSystemClockWithInternalOffset() NN_NOEXCEPT;

/**
 * @brief   バックグラウンドで行われる時刻の自動補正処理を抑止します。
 * @details
 *  自身のプロセスに限らず、一度でも本関数が呼ばれると自動補正処理は停止されます。
 *  本体システムのシャットダウンによって設定は無効化されます。
 *
 *  本関数を呼んだのと同じ回数 ResumeAutonomicTimeCorrection() を呼ぶことで、自動補正処理が再開されます。
 *
 *  Windows 実行環境では、事前に NIFM ライブラリが初期化済である必要があります。
 */
void SuspendAutonomicTimeCorrection() NN_NOEXCEPT;

/**
 * @brief   バックグラウンドで行われる時刻の自動補正処理を再開します。
 * @pre
 *  - 自動補正処理が停止状態である
 * details
 *  SuspendAutonomicTimeCorrection() を呼んだ回数と同じ回数分本関数を呼ぶことで、自動補正処理が再開されます。
 */
void ResumeAutonomicTimeCorrection() NN_NOEXCEPT;

/**
 * @brief   StandardSteadyClock の内部オフセットを設定します。
 * @param[in]   internalOffset  内部オフセット
 * @pre
 *  - nn::time::InitializeForRepair でライブラリが初期化されている
 * @details
 *  nn::time::StandardSteadyClock への内部オフセットの反映は次回のデバイス起動時です。
 */
void SetStandardSteadyClockInternalOffset(const nn::TimeSpan internalOffset) NN_NOEXCEPT;

//! @}
}
}

#include <nn/time/time_SteadyClockPrivateApi.h>

