﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
*   @file
*   @brief  ネットワーク時計の有効性確立 API の宣言
*/

#pragma once

#include <nn/time/time_Common.h>
#include <nn/time/time_ResultCorrection.h>
#include <nn/util/util_TypedStorage.h>
#include <nn/os/os_SystemEvent.h>

namespace nn
{
namespace time
{

//! @name ネットワーク時計の有効性確立 API
//! @{

/**
 * @brief   ネットワーク時計の有効性の確立を行い、処理完了までブロックします。
 *
 * @return  処理の結果が返ります。
 * @retval  ResultSuccess                       成功しました。
 * @retval  ResultNetworkRequestNotAccepted     ネットワーク利用要求が受理されていません。
 * @retval  ResultCanceled                      処理がキャンセルされました。
 * @retval  上記以外                            失敗しました。
 *
 * @pre
 *  - ネットワーク接続管理ライブラリ (NIFM) でネットワーク利用要求が受理されている
 *  - TIME ライブラリが初期化されている
 *
 * @details
 *  時刻に一定の信頼がおけるネットワーク時計の有効性の確立を行い、
 *  処理の完了までブロックします。@n
 *
 *  @link network_clock_autonomic_correction ネットワーク時計の自動補正について @endlink も併せて参照してください。
 *
 *  本関数実行時、既にネットワーク時計が利用可能であるならば、
 *  サーバーにアクセスすることなくごく短時間で成功を返します。@n
 *  そうでない場合はネットワーク経由で時刻の取得を試行し、
 *  ネットワーク時計の同期処理が完了するまでブロックします。
 *
 *  本関数が成功を返したとき、 @ref StandardNetworkSystemClock から
 *  一定の信頼がおける時刻の取得が可能であることが保証されます。@n
 *  ただし、デバイスの電源断をまたいでは保証されないことに注意してください。
 *
 *  ネットワーク時計の有効性確立の失敗時に、自動的なリトライ処理はしてはいけません。@n
 *  詳細についてはガイドラインを参照してください。
 *
 * @platformbegin{Windows}
 *  Windows 環境においては、 PC 時計と同等の時刻にネットワーク時計が補正されます。
 * @platformend
 *
 * @see     CancelEnsuringNetworkClockAvailability() , AsyncEnsureNetworkClockAvailabilityTask クラス
 */
Result EnsureNetworkClockAvailability() NN_NOEXCEPT;

/**
 * @brief   ネットワーク時計の有効性の確立処理をキャンセルします。
 *
 * @details
 *  @ref EnsureNetworkClockAvailability 関数を実行したスレッドとは別のスレッドから本関数を利用することで、
 *  @ref EnsureNetworkClockAvailability 関数のブロックをキャンセルすることができます。
 */
void CancelEnsuringNetworkClockAvailability() NN_NOEXCEPT;

//! @}

/**
 * @brief   ネットワーク時計の有効性確立を非同期に行うためのクラスです。
 *
 * @details
 *  @link network_clock_autonomic_correction ネットワーク時計の自動補正について @endlink も併せて参照してください。
 *
 *  ネットワーク時計の有効性確立が成功したとき、 @ref StandardNetworkSystemClock から
 *  一定の信頼がおける時刻の取得が可能であることが保証されます。@n
 *  ただし、デバイスの電源断をまたいでは保証されないことに注意してください。
 *
 *  ネットワーク時計の有効性確立の失敗時に、自動的なリトライ処理はしてはいけません。@n
 *  詳細についてはガイドラインを参照してください。
 *
 * @platformbegin{Windows}
 *  Windows 環境においては、 PC 時計と同等の時刻にネットワーク時計が補正されます。
 * @platformend
 */
class AsyncEnsureNetworkClockAvailabilityTask
{
    NN_DISALLOW_COPY(AsyncEnsureNetworkClockAvailabilityTask);
    NN_DISALLOW_MOVE(AsyncEnsureNetworkClockAvailabilityTask);

public:
    /**
     * @brief   コンストラクタです。
     * @details
     */
    AsyncEnsureNetworkClockAvailabilityTask() NN_NOEXCEPT;

    /**
     * @brief   デストラクタです。
     * @details
     *  本クラスのオブジェクトは非同期処理の実行状態にかかわらず、
     *  いつでも破棄することができます。
     *
     *  非同期処理実行中のオブジェクトを破棄しても、
     *  システム側で非同期処理は続行されます。
     */
    ~AsyncEnsureNetworkClockAvailabilityTask() NN_NOEXCEPT;

    /**
     * @brief       オブジェクトを初期化します。
     * @param[in]   eventClearMode  @ref GetFinishEvent 関数で得られるシステムイベントのクリアモードを指定します。
     * @pre
     *  - ネットワーク接続ライブラリ( NIFM )でネットワーク利用要求が受理されている
     *  - TIME ライブラリが初期化されている
     *
     * @details
     *  他のどのメンバ関数よりも先にこの関数を実行する必要があります。
     *  複数回実行しても副作用はありません。
     */
    void Initialize(nn::os::EventClearMode eventClearMode) NN_NOEXCEPT;

    /**
     * @brief   ネットワーク時計の有効性確立の非同期処理を開始します。
     * @return  処理の結果が返ります。
     * @retval  ResultSuccess                       成功しました。
     * @retval  ResultNetworkRequestNotAccepted     ネットワーク利用要求が受理されていません。
     * @pre
     *  - @ref Initialize 関数が実行済である
     *
     * @details
     *  StartTask 関数の実行後に @ref GetFinishEvent 関数で得られる
     *  システムイベントのシグナルを検知することで、非同期処理の完了を知ることができます。
     *  既にネットワーク時計が利用可能な場合には、非同期処理はサーバーにアクセスすることなくごく短時間で完了します。
     *
     *  処理の実行中に StartTask 関数を再度呼び出しても、処理は一度しか行われません。
     *  その際に本関数は ResultSuccess を返します。
     *
     *  @ref GetFinishEvent 関数で得られるシステムイベントがシグナル後であれば、
     *  再度 @ref StartTask 関数で処理を再実行することができます。
     */
    nn::Result StartTask() NN_NOEXCEPT;

    /**
     * @brief   非同期処理の結果を取得します。
     * @return  処理の結果が返ります。
     * @retval  ResultSuccess       成功しました。
     * @retval  ResultNotStarted    処理が開始されていません。
     * @retval  ResultProcessing    処理の実行中です。
     * @retval  ResultCanceled      処理がキャンセルされました。
     * @retval  上記以外            失敗しました。
     * @pre
     *  - @ref Initialize 関数が実行済である
     *
     * @details
     *  @ref GetFinishEvent 関数で得られるシステムイベントがシグナル後に、
     *  非同期処理の結果が得られます。
     *
     *  @ref StartTask 関数実行前や、システムイベントのシグナル前に呼び出しても副作用はありませんが、
     *  非同期処理の完了をポーリングする場合、ハンドリングの容易さや CPU 負荷の観点から、
     *  @ref GetFinishEvent 関数で得られるイベントに対し TryWait 関数を呼び出すことを推奨します。
     */
    nn::Result GetResult() NN_NOEXCEPT;

    /**
     * @brief   非同期処理の完了時にシグナルされるシステムイベントを取得します。
     * @pre
     *  - @ref Initialize 関数が実行済である
     *
     * @details
     *  成功・失敗を問わず、非同期処理が完了したときにシグナル状態に遷移するイベントを取得します。
     */
    nn::os::SystemEvent& GetFinishEvent() NN_NOEXCEPT;

    /**
     * @brief   実行中の非同期処理をキャンセルします。
     * @pre
     *  - @ref Initialize 関数が実行済である
     *
     * @details
     *  @ref StartTask 関数実行前に呼び出しても副作用はありません。
     *
     *  キャンセルが正常に受け付けられた場合、
     *  @ref GetFinishEvent 関数で得られるシステムイベントがシグナル状態に遷移し、
     *  @ref GetResult 関数はキャンセルされたことを示す値を返すようになります。
     */
    void Cancel() NN_NOEXCEPT;

private:
    typedef nn::util::TypedStorage<char, 128u, 8u> StorageType;
    StorageType m_Storage;
    bool m_IsInitialized;
};

}
}
