﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
/**
*   @file
*   @brief  単調増加する時刻取得を提供するシステムクロッククラスの宣言(非公開)
*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/time/time_PosixTime.h>
#include <nn/time/time_SystemClockContext.h>
#include <nn/os/os_SdkMutex.h>

namespace nn
{
namespace time
{
namespace detail
{

/**
 * @brief   単調増加する時刻取得を提供するシステムクロッククラスです。
 * @details
 *  得られる時刻が巻き戻ることがなく、アプリケーションでの扱いが容易な時刻取得を提供するクラスです。
 *
 *  nn::time::StandardUserSystemClock や nn::time::StandardNetworkSystemClock
 *  の時刻を利用するアプリケーションの起動中、もしくは起動中のある特定範囲において、
 *  単調増加する時刻を扱うことができます。
 *
 *  ただし、システム側の時刻操作や時刻補正による時刻の変化には自動的に追従せず、
 *  システムの最新時刻とは異なる可能性があることに注意してください。@n
 *  システム側の最新時刻に追従させるには、最新のコンテキストを SetSystemClockContext() で指定する必要があります。
 *
 *  メンバ関数はスレッドセーフです。
 */
class MonotonicSystemClock
{
public:
    /**
     * @brief       コンストラクタです。
     * @details
     */
    MonotonicSystemClock() NN_NOEXCEPT;

    /**
     * @brief   クロックが利用しているコンテキストの値を取得します。
     * @details
     */
    nn::time::SystemClockContext GetSystemClockContext() const NN_NOEXCEPT;

    /**
     * @brief   クロックのコンテキストの値を設定します。
     * @param[in]   systemClockContext  コンテキスト
     * @pre
     *  - 時刻ライブラリが初期化されている
     *
     * @details
     */
    void SetSystemClockContext(const nn::time::SystemClockContext& systemClockContext) NN_NOEXCEPT;

    /**
     * @brief   現在時刻を取得します。
     * @param[out]  pOutPosixTime   現在時刻
     * @pre
     *  - pOutPosixTime != nullptr
     *
     * @return  処理の結果が返ります。事前条件を満たしていない場合、以下に列挙されていない失敗が返ることがあります。
     * @retval  ResultSuccess                       成功しました。
     * @retval  ResultClockInvalid                  時計が無効です。
     *
     * @details
     *  コンテキストをもとに現在時刻を高速に計算します。
     *  得られる時刻は単調増加することが保証されます。
     */
    Result GetCurrentTime(nn::time::PosixTime* pOutPosixTime) const NN_NOEXCEPT;

    /**
     * @brief   時計が時刻を返せるかどうかを取得します。
     */
    bool IsAdjusted() const NN_NOEXCEPT;

private:
    mutable nn::os::SdkMutex m_Lock;
    nn::time::SystemClockContext m_CachedContext;
};

}
}
}
