﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief TCAP サーバライブラリの API 宣言。
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_TimeSpan.h>

#include <nn/os/os_SystemEvent.h>
#include <nn/tcap/tcap_OperationModeTypes.h>

namespace nn { namespace tcap { namespace server {

//! @name TCAP サーバライブラリのサーバのイベントハンドラ向け API
//! @{

/**
 * @brief       TCAP サーバライブラリに経過時間を通知します。
 *
 * @param[in]   elapsedTime 経過時間を表す @ref nn::TimeSpan 型の値です。
 *
 * @details     TCAP サーバライブラリに経過時間を通知します。
 */
void Update(nn::TimeSpan elapsedTime) NN_NOEXCEPT;

/**
 * @brief       TCAP サーバライブラリの保持する経過時間をクリアします。
 *
 * @details     TCAP サーバライブラリの保持する経過時間をクリアします。
 */
void ClearElapsedTime() NN_NOEXCEPT;

//! @}

//! @name TCAP サーバライブラリの Manager セッション向け API
//! @{

/**
 * @brief       表面温度が長時間高温であるときにシグナルされるシステムイベントを取得します。
 *
 * @details     表面温度が長時間高温であるときにシグナルされるシステムイベントを取得します。@n
 *              厳密には製品相当環境で以下の条件を連続 8 時間満たし続けたときにシグナルされます。@n
 *
 *              - 表面温度が推定 48C 以上である
 *              - 動作モードが携帯モードである
 *              - モーションセンサー、タッチパネルなどの一部入力を除いたコントローラ入力の変化がない
 *              - スリープおよびシャットダウン状態ではない
 */
nn::os::SystemEventType* GetContinuousHighSkinTemperatureEvent() NN_NOEXCEPT;

/**
 * @brief       TCAP サーバライブラリに現在の動作モードを設定します。
 *
 * @param[in]   OperationMode 動作モードを表す @ref nn::tc::OperationMode 型の列挙値です。
 *
 * @details     TCAP サーバライブラリに現在の動作モードを設定します。
 */
void SetOperationMode(OperationMode operationMode) NN_NOEXCEPT;

/**
 * @brief       settings から設定値を再読み込みします。
 *
 * @post        settings の設定が反映されます。@n 反映に再起動は不要です。
 *
 * @details     settings から設定値を再読み込みします。
 */
void LoadAndApplySettings() NN_NOEXCEPT;

//! @}

}}} // namepsace nn::tcap::server
