﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/swkbd/swkbd_Types.h>
#include <nn/swkbd/swkbd_InlineTypes.h>

#include <nvn/nvn.h>
#include <nn/util/util_BitFlagSet.h>

/**
 * @file
 * @brief   ソフトウェアキーボードで使用される型や定数の宣言(非公開版)
 */

namespace nn { namespace swkbd {

#if defined (NN_BUILD_CONFIG_COMPILER_CLANG)
#pragma clang diagnostic push
#pragma clang diagnostic warning "-Wpadded"
#endif


/**
* @brief ソフトウェアキーボードを初期化する際に使用する構造体です。
*/
class InitializeArg
{
public:
    InitializeArg()
        : windowOriginMode( NVN_WINDOW_ORIGIN_MODE_LOWER_LEFT )
        , isOverlayLayer( false )
        , frameStep( 1 )
    {
        std::memset(_padding, 0, sizeof(_padding));
    }

    // NVNwindowOriginMode です。@n
    // この値は変更しないでください。
    NVNwindowOriginMode windowOriginMode;
    // ソフトウェアキーボードが描画した結果を
    // ソフトウェアキーボード自身に描画させるかを決めるフラグです。@n
    // デフォルトは false です。@n
    // この値は変更しないでください。
    bool isOverlayLayer;
    uint8_t frameStep;
    uint8_t _padding[2];
};


/**
 * @brief FinishedInitializeCallback で渡される構造体です。
 */
struct FinishedInitializeArg
{
    //! @brief 初期化に成功したかどうかが格納されます。
    bool isSuccess;
};


/**
 * @brief
 */
enum TabType : uint32_t
{
    TabType_Prev,
    TabType_Next,
    TabType_Max
};

/**
 * @brief
 */
enum ReplyType : uint32_t
{
    ReplyType_FinishedInitialize,   //!< @brief 初期化終了
    ReplyType_ChangedState,         //!< @brief 状態変更
    ReplyType_ChangedString,        //!< @brief 文字列変更
    ReplyType_MovedCursor,          //!< @brief カーソル移動
    ReplyType_MovedTab,             //!< @brief タブ移動 ( 3.0.0 予定 )
    ReplyType_DecidedEnter,         //!< @brief 決定抜け
    ReplyType_DecidedCancel,        //!< @brief キャンセル抜け
    ReplyType_ChangedStringUtf8,    //!< @brief 文字列変更 ( UTF-8版 )
    ReplyType_MovedCursorUtf8,      //!< @brief カーソル移動 ( UTF-8版 )
    ReplyType_DecidedEnterUtf8,     //!< @brief 決定抜け ( UTF-8版 )
    ReplyType_UnsetCustomizeDic,    //!< @brief カスタマイズ辞書の unmap
    ReplyType_ReleasedUserWordInfo, //!< @brief ストレージの解放
    ReplyType_Max
};

enum UpdateFlags
{
    UpdateFlags_InitializeArg    ,   //!< initializeArg の更新フラグのインデックス
    UpdateFlags_Volume           ,   //!< volume の更新フラグのインデックス
    UpdateFlags_AppearArg        ,   //!< appearArg の更新フラグのインデックス
    UpdateFlags_StringInfo       ,   //!< stringInfo の更新フラグのインデックス
    UpdateFlags_CursorPos        ,   //!< cursorPos の更新フラグのインデックス
    UpdateFlags_Utf8Mode         ,   //!< utf8Mode の更新フラグのインデックス
    UpdateFlags_UnsetCustomizeDic,   //!< カスタマイズ辞書のアンセットフラグのインデックス
    UpdateFlags_RequestDisappear ,   //!< 非表示リクエストのインデックス
    UpdateFlags_BgAlpha          ,   //!< アルファ値のインデックス
    UpdateFlags_ScaleTrans       ,   //!< スケールトランス値のインデックス
    UpdateFlags_UnsetUserWordInfo,   //!< ユーザー辞書のアンセットフラグのインデックス
    UpdateFlags_EnableHWKBD      ,   //!< HWKBD の有効無効フラグのインデックス
    UpdateFlags_InvalidButton    ,   //!< InvalidButton の更新フラグのインデックス
    UpdateFlags_EnableSe         ,   //!< targetSE のSEを有効化させるための更新フラグのインデックス
    UpdateFlags_DisableSe        ,   //!< targetSE のSEを無効化させるための更新フラグのインデックス
    UpdateFlags_EnableBackspace  ,   //!< バックスペースボタンを有効/無効フラグのインデックス
};

struct ProtocolVersion
{
    uint16_t major;
    uint16_t minor;

    /**
     * @brief 引数で与えられたバージョンが、最終リリース時に互換を保証するかどうか
     */
    bool IsCompatibleVersionForRelease( const ProtocolVersion& input ) const
    {
        return (input.major < this->major) || (input.major == this->major && input.minor == this->minor);
    }
};

inline bool operator == (const ProtocolVersion& lhs, const ProtocolVersion& rhs ) NN_NOEXCEPT
{
    return lhs.major == rhs.major && lhs.minor == rhs.minor;
}

inline bool operator != (const ProtocolVersion& lhs, const ProtocolVersion& rhs ) NN_NOEXCEPT
{
    return !(lhs == rhs);
}

inline bool operator > (const ProtocolVersion& lhs, const ProtocolVersion& rhs ) NN_NOEXCEPT
{
    return (lhs.major > rhs.major) || (lhs.major == rhs.major && lhs.minor > rhs.minor);
}

inline bool operator < (const ProtocolVersion& lhs, const ProtocolVersion& rhs ) NN_NOEXCEPT
{
    return (lhs.major < rhs.major) || (lhs.major == rhs.major && lhs.minor < rhs.minor);
}

inline bool operator >= (const ProtocolVersion& lhs, const ProtocolVersion& rhs ) NN_NOEXCEPT
{
    return !(lhs < rhs);
}

inline bool operator <= (const ProtocolVersion& lhs, const ProtocolVersion& rhs ) NN_NOEXCEPT
{
    return !(lhs < rhs);
}



class KeyboardStorageInfo
{
public:
    static const uint16_t cMajorVersion = 0;
    static const uint16_t cMinorVersion = 3;

    static ProtocolVersion Version() NN_NOEXCEPT
    {
        ProtocolVersion version = { cMajorVersion, cMinorVersion };
        return version;
    }

    KeyboardStorageInfo() NN_NOEXCEPT
    {
        SetDefault();
    }

    void SetDefault() NN_NOEXCEPT
    {
        version = Version();
        commandSize = static_cast<uint16_t>( sizeof(KeyboardStorageInfo) );
        updateFlags.Reset();

        isVisibleKeytop = true;
        isVisibleFooter = true;
        InitializeArg initialize_arg;
        initializeArg = initialize_arg;
        volume = 1.0f; // 音量のデフォルト値は 1
        cursorPos = 0;
        AppearArg appear_arg;
        appearArg = appear_arg;
        InputText input_text;
        inputText = input_text;
        isUseUtf8 = false; // デフォルトは UTF-16
        isSeparateMode = false;
        isValidBackspace = true;
        isPredictVisible = true;
        isShadowEnable   = false;
        isPreviewVisible = false;
        isFloatingKeytop = false;
        isScaleWithFooter = true;
        enableAlphaInInputMode = true;
        inputModeFadeType = 0;
        disableTouch = false;
        disableHwkbd = false;
        keyLayoutFilterFlag = 0;
        keytopScaleX = 1.0f;
        keytopScaleY = 1.0f;
        keytopTranslateX = 0.0f;
        keytopTranslateY = 0.0f;

        keytopBgAlpha  = 1.0f;
        footerBgAlpha  = 1.0f;

        baloonScale = 1.0f;
        previewScale = 1.0f;
        previewTranslateX = 0.0f;
        previewTranslateY = 0.0f;
        targetSe = 0;
        std::memset(previewFontColor, 0, sizeof(previewFontColor));

        std::memset(_padding2, 0, sizeof(_padding2));
    }
    // 変更要望が 1 つでもあったかどうか
    bool IsChanged() const NN_NOEXCEPT
    {
        return updateFlags.IsAnyOn();
    }

    ProtocolVersion  version;      //!< ストレージ構造体のバージョン
    uint16_t commandSize;  //!< @brief コマンドサイズ
    bool isVisibleKeytop;       //!< @brief キートップを表示するかフラグ (reserve)
    bool isVisibleFooter;       //!< @brief フッターを表示するかのフラグ (reserve)

    nn::util::BitFlagSet<64> updateFlags;   //!< 変更パラメータのフラグ

    InitializeArg initializeArg;
    float volume;
    int32_t cursorPos;
    AppearArg appearArg; // 8byte アライン
    InputText inputText;
    bool isUseUtf8;
    bool isSeparateMode;    //!< @brief セパレートモード (reserve)
    bool isValidBackspace;  //!< @brief BS ボタンを有効にするかどうか (reserve)
    bool isPredictVisible;  //!< @brief 予測変換欄を表示するかどうかのフラグ (reserve)
    bool isShadowEnable;    //!< @brief 文字の影の有効フラグ (reserve)
    bool isPreviewVisible;  //!< @brief プレビュー欄の表示とタッチハンドリングを LA 側でやるかどうか (reserve)
    bool isFloatingKeytop;  //!< @brief キートップをフローティングさせるかどうかのフラグ。下地の下部にマージンが付く。
    bool isScaleWithFooter; //!< @brief フッターも一緒にスケールするかどうかのフラグ
    bool enableAlphaInInputMode;  //!< @brief インプットモードシーンでもアルファを有効にするフラグ
    uint8_t inputModeFadeType; //!< @brief InputMode の時の裏のレイアウトのフェードモード
    bool    disableTouch;   //!< @brief タッチを無効化する
    bool    disableHwkbd; //!< @brief USB キーボードの利用を無効化する

    uint64_t keyLayoutFilterFlag; //!< @brief 使用できるキーレイアウトのマスク (reserve)
    float  keytopScaleX; //!< @brief キートップ描画の X スケール
    float  keytopScaleY; //!< @brief キートップ描画の Y スケール
    float  keytopTranslateX;  //!< @brief キートップ描画の X 座標オフセット
    float  keytopTranslateY;  //!< @brief キートップ描画の Y 座標オフセット
    float  keytopBgAlpha;     //!< @brief キートップの背景のアルファ値
    float  footerBgAlpha;     //!< @brief フッターの背景のアルファ値
    float  baloonScale;       //!< @brief バルーンのスケール値

    float  previewScale;      //!< @brief プレビュー欄のスケール (reserve)
    float  previewTranslateX; //!< @brief プレビュー欄の X 座標オフセット (reserve)
    float  previewTranslateY; //!< @brief プレビュー欄の Y 座標オフセット (reserve)
    uint8_t previewFontColor[4]; //!< @brief プレビューのフォントカラー R,G,B,A (reserve)
    uint8_t targetSe;          //!< @brief 有効・無効化する対象となるSEのID (5.0NUP～)
    uint8_t _padding2[3];

    // ユーザー辞書・カスタマイズ辞書は LargeStorage / TransferStorage を使っているので省略
};

/**
 *
 */
class KeyboardStorageInfoV01
{
public:
    static const uint16_t cMajorVersion = 0;
    static const uint16_t cMinorVersion = 1;

    static ProtocolVersion Version() NN_NOEXCEPT
    {
        ProtocolVersion version = { cMajorVersion, cMinorVersion };
        return version;
    }

    KeyboardStorageInfoV01() NN_NOEXCEPT
    {
        SetDefault();
    }
    void SetDefault() NN_NOEXCEPT
    {
        version = Version();

        isChangeInitializeArg = false;
        isChangeVolume = false;
        isChangeAppearArg = false;
        isChangeStringInfo = false;
        isChangeCursorPos = false;
        isChangeUtf8Mode = false;
        isUnsetCustomizeDic = false;
        isRequestDisappear = false;
        InitializeArg initialize_arg;
        initializeArg = initialize_arg;
        volume = 1.0f; // 音量のデフォルト値は 1
        AppearArg appear_arg;
        appearArg = appear_arg;
        InputText input_text;
        inputText = input_text;
        cursorPos = 0;
        isUseUtf8 = false; // デフォルトは UTF-16

        std::memset(_padding2, 0, sizeof(_padding2));
    }
    //! 変更要望が 1 つでもあったかどうか
    bool IsChanged() const NN_NOEXCEPT
    {
        return ( isChangeInitializeArg ||
            isChangeVolume ||
            isChangeAppearArg ||
            isChangeStringInfo ||
            isChangeCursorPos ||
            isChangeUtf8Mode ||
            isUnsetCustomizeDic ||
            isRequestDisappear );
    }

    ProtocolVersion  version;           //!< ストレージ構造体のバージョン
    bool isChangeInitializeArg; //!<
    bool isChangeVolume;        //!<
    bool isChangeAppearArg;     //!<
    bool isChangeStringInfo;    //!<
    bool isChangeCursorPos;     //!<
    bool isChangeUtf8Mode;      //!<
    bool isUnsetCustomizeDic;   //!<
    bool isRequestDisappear;    //!<

    InitializeArg initializeArg;
    float volume;
    AppearArg appearArg;
    InputText inputText;
    int32_t cursorPos;
    bool isUseUtf8;
    uint8_t _padding2[7];
    // ユーザー辞書・カスタマイズ辞書は LargeStorage / TransferStorage を使っているので省略

    /**
    * @brief MovedTabCallback で渡される構造体
    */
    void UpgradeTo( KeyboardStorageInfo& info ) const NN_NOEXCEPT
    {
        info.SetDefault();

        info.updateFlags.Set( UpdateFlags_InitializeArg, isChangeInitializeArg );
        info.updateFlags.Set( UpdateFlags_Volume,        isChangeVolume );
        info.updateFlags.Set( UpdateFlags_AppearArg,     isChangeAppearArg );
        info.updateFlags.Set( UpdateFlags_StringInfo,    isChangeStringInfo );
        info.updateFlags.Set( UpdateFlags_CursorPos,     isChangeCursorPos );
        info.updateFlags.Set( UpdateFlags_Utf8Mode,      isChangeUtf8Mode );
        info.updateFlags.Set( UpdateFlags_UnsetCustomizeDic, isUnsetCustomizeDic );
        info.updateFlags.Set( UpdateFlags_RequestDisappear, isRequestDisappear );

        info.initializeArg = initializeArg;
        info.volume = volume;
        info.appearArg = appearArg;
        info.inputText = inputText;
        info.cursorPos = cursorPos;
        info.isUseUtf8 = isUseUtf8;
    }
};


#if defined (NN_BUILD_CONFIG_COMPILER_CLANG)
#pragma clang diagnostic pop
#endif


/**
* @brief MovedTabCallback で渡される構造体
*/
struct MovedTabArg
{
    char16_t text[ TextMaxLength + 1 ]; //!< @brief 文字列です。
    int32_t textLength;                 //!< @brief 文字列の長さです。
    TabType tabType;                    //!< @brief タブの移動先です。
};

/**
* @brief
*/
typedef void( *MovedTabCallback )( MovedTabArg* pMovedTabArg );


}} // namespace nn::swkbd
